//---------------------------------------------------------------------------------
// 
//  mv10q010.v
//
//  Description: Everspin 1 Mb Quad SPI MRAM Memory Model
//
//---------------------------------------------------------------------------------
// 
//  This confidential and proprietary software may be used only as authorized 
//  by the included licensing agreement from Everspin titled 'EverspinSLA.txt'     
//                                                                               
//                   Copyright 2016 Everspin Technologies, Inc
//                                                                               
//  The entire notice above must be reproduced and the Everspin SLA included for 
//  all authorized copies.                                                                 
//
//---------------------------------------------------------------------------------
//
//  Current Revision:  1.0
//
//  Revision History:
//  1.0   05.02.16    Initial Release   
//
//--------------------------------------------------------------------------------- 

// Note - timescale must be set to 1ps/1ps
`timescale 1ps / 1ps

`include "mr10q010.vh"

module mr10q010 (
			// SPI Mode  	    Quad SPI Mode
	cs_b,	// chip select		chip select
	so,		// serial output 	I/O1
	wp_b,	// write protect	I/O2
	si,		// Serial input		I/O0
	sck,	// clock			clock
	hold_b  // Hold				I/O3
);

input	cs_b;
input 	sck;
inout	si;
inout 	hold_b;
inout	wp_b;
inout	so;


wire 	si;
wire 	hold_b;
wire 	wp_b;
wire 	so;
wire 	dly_xfr_done;
wire [3:0]	quad_data_in;


reg [7:0]	shift;
reg [2:0]	shftcnt;
reg [3:0]   spifsm, next_spifsm;
reg		srwd;
reg		bp1;
reg		bp0;
reg		wel;
reg [7:0]	so_byte;
reg [7:0]	so_out_byte;
reg [7:0]	write_byte;
reg [7:0]	read_byte;
reg 	so_ld;
reg 	xfr_done;
reg [1024:1] 	tmp_model_dir;
reg [23:0]	spi_addr;
reg		read;
reg		did_read;
reg		td_read;
reg		fread;
reg		fwrite;
reg		set_read;
reg		set_did_read;
reg		set_td_read;
reg		set_fread;
reg		set_fwrite;
reg		write;
reg		set_write;
reg		start_data_xfr;
reg 	so_en;
reg 	qpi_en;
reg 	so_out;
reg     so_mux_sel;
reg		mode; 		// =0 mode0; =1 mode3
reg		sleep_mode;
reg		set_sleep_mode;
reg		clr_sleep_mode;
reg		set_xip_mode;
reg		clr_xip_mode;
reg		set_qpi_mode;
reg		clr_qpi_mode;
reg		set_td_mode;
reg		clr_td_mode;
reg [39:0]	did;
reg		xip_mode;	// =0 reset; =1 set
reg		td_mode;	// =0 reset; =1 set
reg 	qpi_mode;
reg 	quad_dir;	// =0 output; =1 input
reg [2:0]	xfrcnt;
reg		quad_data;
reg		quad_shift;
reg		quad_mode;
reg		quad_addr;
reg [2:0]	initial_status;
reg		status_protect;
reg 	write_to_protected_area;

integer		memfd;
integer		statfd;
integer 	did_rd_cnt;
integer 	td_rd_cnt;

integer		sck_per;
integer		sck_p_width;
integer		sck_p_ts;
integer		sck_n_ts;
integer		cs_b_ts;
integer		cs_ts;
integer		write_op;


parameter SPI_IDLE        = 4'b0000;
parameter SPI_ADDR        = 4'b0001;
parameter SPI_RD_SR       = 4'b0010;
parameter SPI_WR_SR       = 4'b0011;
parameter SPI_WAIT_FOR_CS = 4'b0100;
parameter SPI_ADDR_MSB    = 4'b0101;
parameter SPI_ADDR_MID    = 4'b0110;
parameter SPI_ADDR_LSB    = 4'b0111;
parameter SPI_MODE_BYTE   = 4'b1000;

parameter RDSR  = 8'h05;
parameter WREN  = 8'h06;
parameter WRDI  = 8'h04;
parameter WRSR  = 8'h01;
parameter READ  = 8'h03;
parameter FREAD = 8'h0B;
parameter WRITE = 8'h02;
parameter SLEEP = 8'hB9;
parameter WAKE  = 8'hAB;
parameter TDET  = 8'h17;
parameter TDETX = 8'h07;
parameter RDID  = 8'h4B;
parameter EQPI  = 8'h38;
parameter DQPI  = 8'hFF;
parameter FRQO  = 8'h6B;
parameter FWQD  = 8'h32;
parameter FRQAD = 8'hEB;
parameter FWQAD = 8'h12;



initial
begin : file_io_open

    if (!$value$plusargs("model_data=%s", tmp_model_dir))
    begin
	        tmp_model_dir = "/tmp";
	        $display("%m Time: %20t WARNING: no +model_data option specified, using /tmp directory.", $time);
	        memfd = open_file();
	        statfd = open_stat();
		initial_status = read_status(statfd);
		if(initial_status === 3'bxxx) begin
			$display("%m Time: %20t Initializing status non volatile data to 000", $time);
			status_write(statfd, 3'b000);
			{srwd, bp1, bp0} = 3'b000;
		end else begin
			{srwd, bp1, bp0} = initial_status;
			$display("%m Time: %20t Initializing status non volatile data", $time);
		end
    end
end

//
// output tri-state buffers
//
bufif0 so_buf   (so,     so_out,   so_hiz);
bufif0 si_buf   (si,     si_out,   si_hiz);
bufif0 hold_buf (hold_b, hold_out, hold_hiz);
bufif0 wp_buf   (wp_b,   wp_out,   wp_hiz);

//
// models the tDIS and tHO parameters
//
assign #TDIS so_hiz = quad_data & quad_dir | cs_b;
assign hold_hiz = quad_dir;
assign si_hiz = quad_dir;
assign wp_hiz = quad_dir;
assign #THO so_sel  = so_en;
assign #THO qpi_sel = qpi_en;
assign si_out   =                          (qpi_sel) ? so_byte[4] : 1'bx;
assign so_out   =  (so_sel) ? so_byte[7] : (qpi_sel) ? so_byte[5] : 1'bx;
assign wp_out   =                          (qpi_sel) ? so_byte[6] : 1'bx;
assign hold_out =                          (qpi_sel) ? so_byte[7] : 1'bx;
assign quad_data_in = {hold_b, wp_b, so, si};

//
// Initial states for internal registers
//
initial begin
	xfrcnt     = 3'b111;
	shftcnt    = 3'b000;
	spifsm     = SPI_IDLE;
    mode       = 1'b0;		// mode0
    wel        = 1'b0;		// WEL is reset on power-up
	qpi_mode   = 0;			// QPI_Mode is reset on power-up
	sleep_mode = 0;
	xip_mode   = 0;
	quad_data  = 0;
	quad_shift = 0;
	quad_addr  = 0;
	quad_dir   = 1;			// all bidis are inputs
	td_mode    = 0;
	status_protect = 0;
	did        = 40'b0000_0111_0110_1011_0001_0001_0001_0001_0001_0001;
	write_op = 0;
	cs_ts = 0;

end

//
// when cs_n falls, the mode is selected
//
always @(negedge cs_b) begin
	if(sck) mode = 1'b1; 
        else mode = 1'b0;
end

//
// inputs are always captured on rising edge of sck
// outputs are always generated on the falling edge of sck
//
// main shift in register
//
always @(posedge sck) begin
	if(!cs_b) begin
		if(quad_shift) shift = {shift[3:0], quad_data_in};
		else shift <= {shift[6:0], si};
                if(xfr_done) shftcnt <= 3'b000;
		else shftcnt <= shftcnt + 1'b1;
	end
end

//
// main shift out register
//
always @(negedge sck) begin
	if(!cs_b) begin
		if(so_ld)
			#TV so_byte <= so_out_byte;
		else
			if(quad_shift) begin
				#TV so_byte <= {so_byte[3:0], 4'bx};
			end else begin
				#TV so_byte <= {so_byte[6:0], 1'bx};
			end
	end
end

//
// so output mux
//
always@(*) begin
	case(so_mux_sel)
		1'b0: so_out_byte = {srwd, qpi_mode, 1'b0, 1'b0, bp1, bp0, wel, 1'b0}; 
	    1'b1: so_out_byte = read_byte;
		default: so_out_byte = 4'hx; 
	endcase
end

//
// delay for file IO
assign #5 dly_xfr_done = xfr_done;

//
// FILE IO
//
always @(negedge sck) begin
	if(start_data_xfr & dly_xfr_done) begin
		if(write | fwrite) begin
			write_to_protected_area = 0;
			case ({bp1, bp0})
				2'b01: if(spi_addr > 24'hBF_FF_FF) write_to_protected_area = 1;
				2'b10: if(spi_addr > 24'h7F_FF_FF) write_to_protected_area = 1;
				2'b11: write_to_protected_area = 1;
				default: write_to_protected_area = 0;
            endcase
			
			if(write_to_protected_area) begin
					$display("%m Time: %20t - WARNING: Attempt to Write to Protected Memory @ %06h:%02h", $time, spi_addr, write_byte);
					if(STOP_ON_WARNING) $stop;
				end else begin
					write_to_file(memfd, spi_addr, write_byte);
					write_op = 1;
					if(VERBOSE_LEVEL > 0) $display("%m Time: %20t - Write to Memory @ %06h:%02h", $time, spi_addr, write_byte);
				end
        end
            
        if(read | fread) begin
           	read_byte = prefetch_mem(memfd, spi_addr);
			if(VERBOSE_LEVEL > 0) $display("%m Time: %20t - Prefetch from Memory @ %06h:%02h", $time, spi_addr, read_byte);
		end
        
        if(did_read) begin
            did_rd_cnt = did_rd_cnt + 1;
			case(did_rd_cnt)
			1: read_byte = did[31:24];
			2: read_byte = did[23:16];
			3: read_byte = did[15:08];
			4: read_byte = did[07:00];
            endcase
		end

        if(td_read) begin
           	td_rd_cnt = td_rd_cnt + 1;
			case(td_rd_cnt)
			1: read_byte = 8'h00;
			2: read_byte = 8'h00;
			3: read_byte = 8'h00;
			4: read_byte = 8'h00;
            endcase
		end
        spi_addr = spi_addr + 1;
	end
end

//
// Main SPI State Machine
//
// Sequential Part
always @(negedge sck or posedge cs_b) begin
	if(cs_b) begin
       	spifsm <= SPI_IDLE; 
		fread = 1'b0;
		fwrite = 1'b0;
		read = 1'b0;
		write = 1'b0;
		did_read = 1'b0;
		td_read = 1'b0;
	end else begin
    	if(set_read) read <= 1'b1;
    	if(set_did_read) did_read <= 1'b1;
    	if(set_td_read) td_read <= 1'b1;
    	if(set_fread) fread <= 1'b1;
    	if(set_fwrite) fwrite <= 1'b1;
    	if(set_write) write <= 1'b1;
        if(set_sleep_mode) #1 sleep_mode <= 1;
        else if(clr_sleep_mode) #1 sleep_mode <= 0;
        if(set_xip_mode) #1 xip_mode <= 1;
        else if(clr_xip_mode) #1 xip_mode <= 0;
        if(set_qpi_mode) #1 qpi_mode <= 1;
        else if(clr_qpi_mode) #1 qpi_mode <= 0;
		if(set_td_mode) td_mode <= 1;
		else if(clr_td_mode) td_mode <= 0;
		if(spifsm == SPI_IDLE & xfr_done) begin
          		if(xip_mode) $display("%m Time: %20t Received Fast Read Command.", $time); 
				else begin
					case(shift)
						RDSR:  $display("%m Time: %20t Received Read Status Register Command.", $time);
						WREN:  $display("%m Time: %20t Received Write Enable Command.", $time);
						WRDI:  $display("%m Time: %20t Received Write Disable Command.", $time);
						WRSR:  $display("%m Time: %20t Received Write Status Register Command.", $time);
						READ:  $display("%m Time: %20t Received Read Command.", $time);
						FREAD: $display("%m Time: %20t Received Initial Fast Read Command.", $time);
						WRITE: $display("%m Time: %20t Received Write Command.", $time);
						SLEEP: $display("%m Time: %20t Received Sleep Command.", $time);
						WAKE:  $display("%m Time: %20t Received Wake Command.", $time);
						TDET:  $display("%m Time: %20t Received Tamper Detect Command.", $time);
						TDETX: $display("%m Time: %20t Received Tamper Detect Exit Command.", $time);
						RDID:  $display("%m Time: %20t Received Read ID Command.", $time);
						EQPI:  $display("%m Time: %20t Received Enable QPI Command.", $time);
						DQPI:  $display("%m Time: %20t Received Disable QPI Command.", $time);
						FRQO:  $display("%m Time: %20t Received Fast Read Quad Output Command.", $time);
						FWQD:  $display("%m Time: %20t Received Fast Write Quad Data Command.", $time);
						FRQAD: $display("%m Time: %20t Received Fast Read Quad Address and Data Command.", $time);
						FWQAD: $display("%m Time: %20t Received Fast Write Quad Address and Data Command.", $time);
						default: begin
							$display("%m Time: %20t WARNING - Received Unknown Command(=%02h).", $time, shift);
							if(STOP_ON_WARNING) $stop;
						end
					endcase
				end
		end
		xfr_done <= (shftcnt == xfrcnt);
		spifsm <= next_spifsm;
	end
end

// Combinational Part
always@(*) begin
	next_spifsm = SPI_IDLE;
    so_mux_sel = 'b0;
    so_en = 1'b0;
    so_ld = 1'b0;
    start_data_xfr = 1'b0;
    set_write = 1'b0;
    set_fwrite = 1'b0;
    set_read = 1'b0;
    set_did_read = 1'b0;
    set_fread = 1'b0;
	set_sleep_mode = 0;
	clr_sleep_mode = 0;
	set_xip_mode = 0;
	clr_xip_mode = 0;
	set_td_mode = 0;
	clr_td_mode = 0;
	set_qpi_mode = 0;
	clr_qpi_mode = 0;
    set_td_read = 0;
	
	case(spifsm)
		SPI_IDLE: begin
			quad_dir = 1;	// quad IO buffers are tristate outputs
			// select the next transfer size/width based on the current state
			case({qpi_mode, xip_mode, quad_addr})
				3'b000: begin xfrcnt = 3'b111; quad_shift = 0; end
				3'b001: begin xfrcnt = 3'b111; quad_shift = 0; end
				3'b010: begin xfrcnt = 3'b111; quad_shift = 0; end
				3'b011: begin xfrcnt = 3'b001; quad_shift = 1; end
				3'b100: begin xfrcnt = 3'b001; quad_shift = 1; quad_data = 1; end
				3'b101: begin xfrcnt = 3'b001; quad_shift = 1; quad_data = 1; end
				3'b110: begin xfrcnt = 3'b111; quad_shift = 0; end
				3'b111: begin xfrcnt = 3'b001; quad_shift = 1; quad_data = 1; end
	    	endcase

			if(xfr_done) begin
				if(xip_mode) begin
					if(quad_addr) begin
						quad_shift = 1;
	                	xfrcnt = 3'b001;
					end else begin
						quad_shift = 0;
	                	xfrcnt = 3'b111;
					end
					set_fread = 1'b1;
					spi_addr[23:16] = shift; 
					next_spifsm = SPI_ADDR_MID;
				end else if(~sleep_mode) begin
					case(shift)
						RDSR: begin
							quad_data = 0;
							quad_shift = 0;
							so_ld = 1'b1;
							next_spifsm = SPI_RD_SR;
						end
						WREN: begin
							wel = 1'b1;
							next_spifsm = SPI_WAIT_FOR_CS;
						end
						WRDI: begin
							wel = 1'b0;
							next_spifsm = SPI_WAIT_FOR_CS;
						end
						WRSR: begin
							quad_shift = 0;
							xfrcnt = 3'b111;
							next_spifsm = SPI_WR_SR;
						end
						READ: begin
		                    set_read = 1'b1;
							quad_addr = 0;			// Address input is serial
		                	xfrcnt = 3'b111;
							quad_data = 0;			// Data output is serial
		                    next_spifsm = SPI_ADDR_MSB;
						end
						FREAD: begin
							set_fread = 1'b1;
							quad_addr = 0;			// Address input is serial
		                	xfrcnt = 3'b111;
							quad_mode = 0;			// Mode input is serial
							quad_data = 0;			// Data output is serial
		                    next_spifsm = SPI_ADDR_MSB;
						end
						WRITE: begin
		                    set_write = 1'b1;
							quad_addr = 0;			// Address input is serial
		                	xfrcnt = 3'b111;
							quad_data = 0;			// Data input is serial
		                    next_spifsm = SPI_ADDR_MSB;
						end
						SLEEP: begin
							set_sleep_mode = 1;
							next_spifsm = SPI_WAIT_FOR_CS;
						end
						WAKE: begin
							$display("%m Time: %20t ERROR: Received WAKE command while not in Sleep Mode.", $time);
							if(STOP_ON_ERROR) $stop;
						end
						TDET: begin
							if(td_mode) begin
								$display("%m Time: %20t ERROR: Received TDET command while still in Tamper Detect Mode. Executing the command anyway!", $time);
								if(STOP_ON_ERROR) $stop;
							end 
							set_td_mode = 1;
							set_td_read = 1'b1;
	                        td_rd_cnt = 0;
							quad_data = 0;
							next_spifsm = SPI_MODE_BYTE;
						end
						TDETX: begin
							clr_td_mode = 1;
		                	xfrcnt = 3'b111;
							next_spifsm = SPI_WAIT_FOR_CS;
						end
						RDID: begin
							set_did_read = 1'b1;
		                    did_rd_cnt = 0;
							quad_data = 0;
							quad_mode = 0;
                			xfrcnt = 3'b111;
							next_spifsm = SPI_MODE_BYTE;
						end
						EQPI: begin
							set_qpi_mode = 1;
							next_spifsm = SPI_WAIT_FOR_CS;
						end
						DQPI: begin
							clr_qpi_mode = 1;
							next_spifsm = SPI_WAIT_FOR_CS;
						end
						FRQO: begin
							set_fread = 1'b1;		// This is a Fast Read
							quad_addr = 0;			// Address input is serial
                			xfrcnt = 3'b111;
							quad_mode = 1;			// Mode input is quad
							quad_data = 1;			// Data output is quad
                    		next_spifsm = SPI_ADDR_MSB;
						end
						FRQAD: begin
							set_fread = 1'b1;		// This is a Fast Read
							quad_addr = 1;			// Address input is quad
                			xfrcnt = 3'b001;
							quad_mode = 1;			// Mode input is quad		
							quad_data = 1;			// Data output is quad
                    		next_spifsm = SPI_ADDR_MSB;
						end
						FWQD: begin
							set_fwrite = 1'b1;		// This is a Fast Write
							quad_addr = 0;			// Address input is serial
                			xfrcnt = 3'b111;		
							quad_data = 1;			// Data input is quad
                    		next_spifsm = SPI_ADDR_MSB;
						end
						FWQAD: begin
							set_fwrite = 1'b1;		// This is a Fast Write
							quad_addr = 1;			// Address input is quad
                			xfrcnt = 3'b001;		
							quad_data = 1;			// Data input is quad
                    		next_spifsm = SPI_ADDR_MSB;
						end
						default: begin
						end
					endcase
				end else begin
					@(negedge sck);
					case(shift)
						WAKE: begin
							clr_sleep_mode = 1;
	                        next_spifsm = SPI_WAIT_FOR_CS;
	                    end
	                    default: begin
							$display("%m Time: %20t ERROR: Recieved command(=%02h) other than WAKE while in SLEEP MODE.", $time, shift);
							if(STOP_ON_ERROR) $stop;
						end
					endcase
				end
			end
		end
		SPI_RD_SR: begin
            xfrcnt = 3'b111;
			so_mux_sel = 1'b0;
			so_en = 1'b1;
			if(xfr_done) begin
				next_spifsm = SPI_IDLE;
			end else
				next_spifsm = SPI_RD_SR;
	    end
		SPI_WR_SR: begin
	        xfrcnt = 3'b111;
			quad_data = 0;
			if(xfr_done) begin
				if(~status_protect) begin
				      srwd = shift[7];	
				      bp1  = shift[3];
				      bp0  = shift[2];
				end else begin
					$display("%m Time: %20t WARNING: Attempt to write status register while protected.", $time);
					if(STOP_ON_WARNING) $stop;
				end
				next_spifsm = SPI_WAIT_FOR_CS;
			end else
				next_spifsm = SPI_WR_SR;
		end
	    SPI_ADDR_MSB: begin
			if(quad_addr) begin
				quad_shift = 1;
			end else begin
				quad_shift = 0;
			end
			if(xfr_done) begin
				spi_addr[23:16] = shift; 
				next_spifsm = SPI_ADDR_MID;
			end else
				next_spifsm = SPI_ADDR_MSB;
		end
	    SPI_ADDR_MID: begin
			if(xfr_done) begin
				spi_addr[15:8] = shift; 
				next_spifsm = SPI_ADDR_LSB;
			end else
				next_spifsm = SPI_ADDR_MID;
		end
        SPI_ADDR_LSB: begin
			if(xfr_done) begin
				@(negedge sck);
				spi_addr[7:0] = shift; 
                if(read) begin
					read_byte = prefetch_mem(memfd, spi_addr);
					if(VERBOSE_LEVEL > 0) $display("%m Time: %20t - Prefetch from Memory @ %06h:%02h", $time, spi_addr, read_byte);
					so_ld = 1;
                	spi_addr = spi_addr + 1;
				end
                if(fread) next_spifsm = SPI_MODE_BYTE;
				else next_spifsm = SPI_WAIT_FOR_CS;
			end else
				next_spifsm = SPI_ADDR_LSB;
		end
        SPI_MODE_BYTE: begin
			so_mux_sel = 1'b1;
			if(quad_mode) begin
				quad_shift = 1;
                xfrcnt = 3'b001;
			end else begin
				quad_shift = 0;
                xfrcnt = 3'b111;
			end
			if(xfr_done) begin
                @(negedge sck);
            	if(shift==8'hEF) set_xip_mode = 1;
            	if(shift==8'hFF) clr_xip_mode = 1;
				so_ld = 1;
                if(read | fread) begin
					read_byte = prefetch_mem(memfd, spi_addr);
					if(VERBOSE_LEVEL > 0) $display("%m Time: %20t - Fast Prefetch from Memory @ %06h:%02h", $time, spi_addr, read_byte);
                	spi_addr = spi_addr + 1;
					quad_dir = 0;
				end
                if(did_read) begin
					read_byte = did[39:32];
					quad_dir = 0;
				end
                if(td_read) begin
					read_byte = 8'h00;
					quad_dir = 0;
				end
				next_spifsm = SPI_WAIT_FOR_CS;
			end else 
				next_spifsm = SPI_MODE_BYTE;
		end
		SPI_WAIT_FOR_CS: begin		// wait for cs_b to go away
            start_data_xfr = 1'b1; 
			so_mux_sel = 1'b1;
			if(read | fread | did_read | td_read) begin
				if(quad_data) begin
					quad_dir = 0;
					qpi_en = 1'b1;
					quad_shift = 1;
				end else begin
					so_en = 1'b1;
					quad_shift = 0;
				end
			end
			if(write | fwrite) begin
				if(quad_data) begin
					quad_shift = 1;
                	xfrcnt = 3'b001;
				end else begin
					quad_shift = 0;
                	xfrcnt = 3'b111;
				end
			end
            if(xfr_done) begin
				if(write | fwrite) write_byte = shift;
				if(read | fread | did_read | td_read) so_ld = 1'b1;
			end
			next_spifsm = SPI_WAIT_FOR_CS;
		end
	endcase
end

always@(sleep_mode) begin
	if(VERBOSE_LEVEL > 0) begin
		if(sleep_mode) $display("%m Time: %20t - INFO: Entering Sleep Mode.", $time);
        else $display("%m Time: %20t - INFO: Exiting Sleep Mode.", $time);
	end
end

//
// check if status changed
//
always@(wel) begin
	if(VERBOSE_LEVEL > 0) begin
		if(wel) $display("%m Time: %20t - INFO: Write Enable Latch Bit set.", $time);
		else $display("%m Time: %20t - INFO: Write Enable Latch Bit reset.", $time); 
	end
end

always@(xip_mode) begin
	if(VERBOSE_LEVEL > 0) begin
        if(xip_mode) $display("%m Time: %20t - INFO: XIP mode set.", $time);
        else $display("%m Time: %20t - INFO: XIP mode reset.", $time);
    end
end

always@(qpi_mode) begin
	if(qpi_mode) begin
		if(VERBOSE_LEVEL > 0) $display("%m Time: %20t - INFO: QPI mode set.", $time);
    end else begin
		quad_data = 0;
		$display("%m Time: %20t - INFO: QPI mode reset.", $time);
	end
end

always@(srwd or bp1 or bp0) begin
	status_write(statfd, {srwd, bp0, bp1});
end

always@(*) begin
	@(negedge sck);
	casex({qpi_mode, wel, srwd, wp_b})
		4'b00XX: status_protect = 1;
		4'b010X: status_protect = 0;
		4'b0110: status_protect = 1;
		4'b0111: status_protect = 0;
		4'b1XXX: status_protect = 0;
		default: status_protect = 0;
	endcase
end

function integer open_file();
    integer fd;
    reg [2048:1] filename;
    begin
        $sformat( filename, "%0s/%m.mem", tmp_model_dir );
        fd = $fopen(filename, "w+");
        if (fd == 0) begin
            $display("%m Time: %20t ERROR: failed to open %0s.", $time, filename);
            $finish;
        end
        else begin
            if(VERBOSE_LEVEL > 0) $display("%m Time: %20t INFO: opening %0s.", $time, filename);
            open_file = fd;
        end
    end
endfunction

function integer open_stat();
    integer fd;
    reg [2048:1] filename;
    begin
        $sformat( filename, "%0s/%m.status", tmp_model_dir );
        fd = $fopen(filename, "w+");
        if (fd == 0) begin
            $display("%m Time: %20t ERROR: failed to open %0s.", $time, filename);
	    end 
        else begin
            if(VERBOSE_LEVEL > 0) $display("%m Time: %20t INFO: opening %0s.", $time, filename);
            open_stat = fd;
        end

    end
endfunction

function [3:1] read_status (
    input integer fd
);
    integer code;
    reg [1024:1] msg;
    reg [3:1] read_value;

    begin
        code = $fseek( fd, 0, 0 );
        // $fseek returns 0 on success, -1 on failure
        if (code != 0)
        begin
            $display("%m Time: %t - ERROR: STATUS fseek failed!", $time);
            if(STOP_ON_ERROR) $stop;
        end

        code = $fscanf(fd, "%u", read_value);
        if(VERBOSE_LEVEL > 4) $display("%m Time: %t %h:STATUS READ %03b", $time, fd, read_value);

        if (code != 1)
        begin
            if ($ferror(fd,msg) != 0)
            begin
                $display("%m Time: %t - ERROR: STATUS fscanf failed!", $time);
                $display(msg);
                $finish;
            end
            else
                read_value = 'hx;
        end
        read_status = read_value;
    end
endfunction

function [8:1] prefetch_mem(
    input integer fd,
    input integer index
);
    integer code;
    integer offset;
    reg [1024:1] msg;
    reg [8:1] read_value;

    begin
        offset = index;
        code = $fseek( fd, offset, 0 );
        // $fseek returns 0 on success, -1 on failure
        if (code != 0)
        begin
            $display("%m Time: %t - ERROR: MEMORY fseek at offset %d failed!", $time, offset);
            if(STOP_ON_ERROR) $stop;
        end

        code = $fscanf(fd, "%u", read_value);
        if(VERBOSE_LEVEL > 4) $display("%m Time: %t %h:MEMORY READ @%06h:%02h", $time, fd, offset, read_value);
        // $fscanf returns number of items read
        if (code != 1)
        begin
            if ($ferror(fd,msg) != 0)
            begin
                $display("%m Time: %t - ERROR: MEMORY fscanf failed at %d!", $time, index);
                $display(msg);
                $finish;
            end
            else
                read_value = 'hx;
        end

        prefetch_mem = read_value;
    end
endfunction

task status_write (
    input integer fd,
    input [3:1] data
);
    integer code;
    begin
        code = $fseek( fd, 0, 0 );
        if (code != 0)
        begin
            $display("%m Time: %t ERROR: STATUS fseek failed", $time);
            $finish;
        end
        if(VERBOSE_LEVEL > 4) $display("%m Time: %t %h:STATUS:%03b", $time, fd, data);
        $fwrite( fd, "%u", data );
    end
endtask

task write_to_file(
    input integer fd,
    input integer index,
    input [8:1] data
);
    integer code;
    integer offset;

    begin
        offset = index;
        code = $fseek( fd, offset, 0 );
        if (code != 0)
        begin
            $display("%m Time: %t ERROR: MEMORY fseek to %d failed", $time, offset);
            $finish;
        end

        if(VERBOSE_LEVEL > 4) $display("%m Time: %t %h:MEMORY WRITE @%06h:%02h", $time, fd, offset, data);
        $fwrite( fd, "%u", data );
    end
endtask

//TIMING CHECKS

always @(negedge cs_b) begin

	cs_b_ts <= $time;
	sck_p_ts <= 0;
	sck_n_ts <= 0;


    if(write_op) begin
		if($time - cs_ts < TCSW) begin  
			$display("%m Time: %t ERROR: Chip select high time after write (tCSW) violated by %d ps!", $time, (TCSW - ($time - cs_ts)));
			if(STOP_ON_ERROR) $stop;
		end
	end else begin
		if($time - cs_ts < TCS) begin  
			$display("%m Time: %t ERROR: Chip select high time (tCS) violated by %d ps!", $time, (TCS - ($time - cs_ts)));
			if(STOP_ON_ERROR) $stop;
		end
	end

	write_op = 0;

end

always @(posedge cs_b) begin
	
	if($time - sck_n_ts < TCSH) begin  
		$display("%m Time: %t ERROR: Chip select hold time (tCSH) violated by %d ps!", $time, (TCSH - ($time - sck_n_ts)));
		if(STOP_ON_ERROR) $stop;
	end

	cs_ts <= $time;

end

always @(posedge sck) begin
	if(!cs_b) begin

		if($time - cs_b_ts < TCSS) begin  
			$display("%m Time: %t ERROR: Chip select setup time (tCSS) violated by %d ps!", $time, (TCSS - ($time - cs_b_ts)));
			if(STOP_ON_ERROR) $stop;
		end

		if(read && !qpi_mode) begin
			if($time - sck_n_ts < TWHR) begin  
				$display("%m Time: %t ERROR: SCK low time during READ (tWLR) violated by %d ps!", $time, (TWLR - ($time - sck_n_ts)));
				if(STOP_ON_ERROR) $stop;
			end
		end else begin
			if($time - sck_n_ts < TWH) begin  
				$display("%m Time: %t ERROR: SCK low time (tWH) violated by %d ps!", $time, (TWL - ($time - sck_n_ts)));
				if(STOP_ON_ERROR) $stop;
			end
		end

		sck_p_ts = $time;
	end
end

always @(negedge sck) begin
	if(!cs_b) begin
		
		if(read && !qpi_mode) begin
			if($time - sck_p_ts < TWHR) begin  
				$display("%m Time: %t ERROR: SCK high time during READ (tWHR) violated by %d ps!", $time, (TWHR - ($time - sck_p_ts)));
				if(STOP_ON_ERROR) $stop;
			end
		end else begin
			if($time - sck_p_ts < TWH) begin  
				$display("%m Time: %t ERROR: SCK high time (tWH) violated by %d ps!", $time, (TWH - ($time - sck_p_ts)));
				if(STOP_ON_ERROR) $stop;
			end
		end
		
		sck_n_ts = $time;
	end
end

endmodule
