//---------------------------------------------------------------------------------
// 
//  tb_mv10q010.v
//
//  Description: Everspin 1 Mb Quad SPI MRAM Memory Model TEST BENCH 
//
//---------------------------------------------------------------------------------
// 
//  This confidential and proprietary software may be used only as authorized 
//  by the included licensing agreement from Everspin titled 'EverspinSLA.txt'     
//                                                                               
//                   Copyright 2016 Everspin Technologies, Inc
//                                                                               
//  The entire notice above must be reproduced and the Everspin SLA included for 
//  all authorized copies.                                                                 
//
//---------------------------------------------------------------------------------
//
//  Current Revision:  1.0
//
//  Revision History:
//  1.0   05.02.16    Initial Release   
//
//--------------------------------------------------------------------------------- 

`timescale 1ps / 1ps

module tb_mr10q010;

`include "mr10q010.vh"

reg	cs_b;
reg 	sck;
wire	si;
wire 	hold_b;
wire	wp_b;
wire	so;

reg     si_en;
reg     qpi_en;
reg     wp_en;
reg     hold_en;
reg     si_in;
reg     so_in;
reg     hold_in;
reg     wp_in;
reg [23:0]	test_addr;

integer i,j;
integer testnum;
integer loop;
integer read;

parameter CKHI = 4800; 
parameter CKLO = 4800; 

parameter RD_CKHI = 11000; 
parameter RD_CKLO = 11000; 

initial
  begin
    $shm_open("waves.shm");
    $shm_probe("ASMT");
  end

initial // Reset generation
  begin
    cs_b    = 1'b1;
    sck     = 1'b0;
    si_en   = 1'b0;
    si_in   = 1'b0;
    so_in   = 1'b0;
    wp_en   = 1'b1;
    wp_in   = 1'b1;
    hold_en = 1'b1;
    hold_in = 1'b1;
    qpi_en  = 1'b0;
    testnum = 1;
    read = 0;
  end

assign si     = (si_en | qpi_en)   ? si_in   : 1'bz;
assign so     = (qpi_en)           ? so_in   : 1'bz;
assign hold_b = (qpi_en | hold_en) ? hold_in : 1'bz;
assign wp_b   = (wp_en | qpi_en)   ? wp_in   : 1'bz;

mr10q010 u1 (
	.cs_b(cs_b),
	.so(so),
	.wp_b(wp_b),
	.si(si),
	.sck(sck),
	.hold_b(hold_b)
);
initial
begin
	$display("**************************************************************************************");
	$display("*********************          SPI Tests          ************************************");
	$display("**************************************************************************************");
	#TCS;
	sck  = 1'b0;
	$display("*********************          %2d. Set Write Enable", testnum);
	// Write Enable
	cs_active;
	do_cmd(8'h06);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Write 1's to Status Register", testnum);
	// write status register to all 1s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'hFF);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Read Status Register", testnum);
	// read status register
	cs_active;
	do_cmd(8'h05);
	rd_data(8'h8E);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Write 0's to Status Register", testnum);
	// write status register to all 0s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h00);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Read Status Register", testnum);
	// read status register
	cs_active;
	do_cmd(8'h05);
	rd_data(8'h02);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Write 4 bytes @ 000000", testnum);
        // Write some data
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h00_00_00);
	wr_data(8'ha5);
	wr_data(8'h00);
	wr_data(8'h5a);
	wr_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Write 4 bytes @ 000004", testnum);
        // Write some data
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h00_00_04);
	wr_data(8'h11);
	wr_data(8'h22);
	wr_data(8'h33);
	wr_data(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Read 4 bytes @ 000000", testnum);
        // Read data back
	cs_active;
	do_cmd(8'h03);
    do_addr(24'h00_00_00);
	rd_data(8'ha5);
	rd_data(8'h00);
	rd_data(8'h5a);
	rd_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read 4 bytes @ 000004", testnum);
        // Read data back
	cs_active;
	do_cmd(8'h03);
        do_addr(24'h00_00_04);
	rd_data(8'h11);
	rd_data(8'h22);
	rd_data(8'h33);
	rd_data(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read 4 bytes @ 000000, set XIP Mode", testnum);
        // Fast Read data back
	cs_active;
	do_cmd(8'h0b);
        do_addr(24'h00_00_00);
	do_cmd(8'hef);		// mode byte - set XIP
	rd_data(8'ha5);
	rd_data(8'h00);
	rd_data(8'h5a);
	rd_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read 4 bytes @ 000004, reset XIP Mode", testnum);
        // Fast Read data back
	cs_active;
        do_addr(24'h00_00_04);
	do_cmd(8'hff);		// mode byte - reset XIP
	rd_data(8'h11);
	rd_data(8'h22);
	rd_data(8'h33);
	rd_data(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Set Sleep Mode", testnum);
	// set sleep mode
	cs_active;
	do_cmd(8'hb9);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Exit Sleep Mode(Wake)", testnum);
	// exit sleep mode
	cs_active;
	do_cmd(8'hab);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Set Sleep Mode", testnum);
	// set sleep mode
	cs_active;
	do_cmd(8'hb9);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Send Command(not Wake) in Sleep Mode (should generate error)", testnum);
	// Write Enable
	cs_active;
	do_cmd(8'h06);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Exit Sleep Mode(Wake)", testnum);
	// exit sleep mode
	cs_active;
	do_cmd(8'hab);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read Device ID", testnum);
	// read device ID
	cs_active;
	do_cmd(8'h4b);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0111);	// read the device ID
	rd_data(8'b0110_1011);
	rd_data(8'b0001_0001);
	rd_data(8'b0001_0001);
	rd_data(8'b0001_0001);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect", testnum);
        // Tamper Detect Command
	cs_active;
	do_cmd(8'h17);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read Data after Tamper Detect(no TDETX)", testnum);
        // Read data back
	cs_active;
	do_cmd(8'h03);
        do_addr(24'h00_00_00);
	rd_data(8'ha5);
	rd_data(8'h00);
	rd_data(8'h5a);
	rd_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect before TDETX (should generate error)", testnum);
        // Tamper Detect Command
	cs_active;
	do_cmd(8'h17);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect Exit(TDETX)", testnum);
	// Tampser Detect Exit Command
	cs_active;
	do_cmd(8'h07);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect after TDETX (should not generate error)", testnum);
        // Tamper Detect Command
	cs_active;
	do_cmd(8'h17);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect Exit(TDETX)", testnum);
	// Tamper Detect Exit Command
	cs_active;
	do_cmd(8'h07);
        cs_inactive;
	testnum++;
	wp_en = 1'b0;
	wp_in = 1'bx;
	hold_en = 1'b0;
	hold_in = 1'bx;
	// Fast Write Quad Data Command
	$display("*********************         %2d. Fast Write Quad Data", testnum);
	cs_active;
	do_cmd(8'h32);
        do_addr(24'h00_00_00);
	wr_qdata(8'h5a);
	wr_qdata(8'hff);
	wr_qdata(8'ha5);
	wr_qdata(8'h00);
        cs_inactive;
	testnum++;
	// Fast Write Quad Data Command
	$display("*********************         %2d. Fast Write Quad Data", testnum);
	cs_active;
	do_cmd(8'h32);
        do_addr(24'h00_00_04);
	wr_qdata(8'h44);
	wr_qdata(8'h33);
	wr_qdata(8'h22);
	wr_qdata(8'h11);
        cs_inactive;
	testnum++;
	// Fast Read Quad Output Command
	$display("*********************         %2d. Fast Read Quad Output, set XIP", testnum);
	cs_active;
	do_cmd(8'h6B);
        do_addr(24'h00_00_00);
        do_qpi_cmd(8'hEF);
	rd_qdata(8'h5a);
	rd_qdata(8'hff);
	rd_qdata(8'ha5);
	rd_qdata(8'h00);
        cs_inactive;
	testnum++;
	// Fast Read Quad Output Command with XIP
	$display("*********************         %2d. Fast Read Quad Output, reset XIP", testnum);
	cs_active;
        do_addr(24'h00_00_04);
        do_qpi_cmd(8'hFF);
	rd_qdata(8'h44);
	rd_qdata(8'h33);
	rd_qdata(8'h22);
	rd_qdata(8'h11);
        cs_inactive;
	testnum++;
	// Fast Read Quad Output Command
	$display("*********************         %2d. Fast Read Quad Output, no XIP", testnum);
	cs_active;
	do_cmd(8'h6B);
        do_addr(24'h00_00_00);
        do_qpi_cmd(8'hFF);
	rd_qdata(8'h5a);
	rd_qdata(8'hff);
	rd_qdata(8'ha5);
	rd_qdata(8'h00);
        cs_inactive;
	testnum++;
	// Fast Read Quad Output Command with XIP
	$display("*********************         %2d. Fast Read Quad Output, no XIP", testnum);
	cs_active;
	do_cmd(8'h6B);
        do_addr(24'h00_00_04);
        do_qpi_cmd(8'hFF);
	rd_qdata(8'h44);
	rd_qdata(8'h33);
	rd_qdata(8'h22);
	rd_qdata(8'h11);
        cs_inactive;
	testnum++;
	// Fast Write Quad Address and Data Command
	$display("*********************         %2d. Fast Write Quad Address and Data", testnum);
	cs_active;
	do_cmd(8'h12);
        do_qaddr(24'h00_00_00);
	wr_qdata(8'ha5);
	wr_qdata(8'h00);
	wr_qdata(8'h5a);
	wr_qdata(8'hff);
        cs_inactive;
	testnum++;
	// Fast Write Quad Address and Data Command
	$display("*********************         %2d. Fast Write Quad Address and Data", testnum);
	cs_active;
	do_cmd(8'h12);
        do_qaddr(24'h00_00_04);
	wr_qdata(8'h11);
	wr_qdata(8'h22);
	wr_qdata(8'h33);
	wr_qdata(8'h44);
        cs_inactive;
	testnum++;
	// Fast Read Quad Address and Data Command
	$display("*********************         %2d. Fast Read Quad Address and Data, set XIP", testnum);
	cs_active;
	do_cmd(8'hEB);
        do_qaddr(24'h00_00_00);
        do_qpi_cmd(8'hEF);		
	rd_qdata(8'ha5);
	rd_qdata(8'h00);
	rd_qdata(8'h5a);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	// Fast Read Quad Address and Data Command
	$display("*********************         %2d. Fast Read Quad Address and Data, reset XIP", testnum);
	cs_active;
        do_qaddr(24'h00_00_04);
        do_qpi_cmd(8'hFF);		
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	testnum++;
	// Fast Read Quad Address and Data Command
	$display("*********************         %2d. Fast Read Quad Address and Data, no XIP", testnum);
	cs_active;
	do_cmd(8'hEB);
        do_qaddr(24'h00_00_00);
        do_qpi_cmd(8'hFF);		
	rd_qdata(8'ha5);
	rd_qdata(8'h00);
	rd_qdata(8'h5a);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	// Fast Read Quad Address and Data Command
	$display("*********************         %2d. Fast Read Quad Address and Data, no XIP", testnum);
	cs_active;
	do_cmd(8'hEB);
        do_qaddr(24'h00_00_04);
        do_qpi_cmd(8'hFF);		
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	testnum++;
	// Enable QPI Mode
	$display("*********************         %2d. Enable QPI Mode", testnum);
	cs_active;
	do_cmd(8'h38);
        cs_inactive;
	cs_active;
	testnum++;
	//
	// All tests beyond this point are in Quad SPI mode
	//
	$display("**************************************************************************************");
	$display("*********************          QPI Tests          ************************************");
	$display("**************************************************************************************");
	$display("*********************         %2d. Read Status Register", testnum);
        // Read Status Register
	do_qpi_cmd(8'h05);
	rd_data(8'h42);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Write 1's to Status Register", testnum);
        // write status register to all 1s
        cs_active;
        do_qpi_cmd(8'h01);
        wr_data(8'hFF);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read Status Register", testnum);
        // read status register
        cs_active;
        do_qpi_cmd(8'h05);
        rd_data(8'hCE);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Write 0's to Status Register", testnum);
	// write status register to all 0s
	cs_active;
	do_qpi_cmd(8'h01);
	wr_data(8'h00);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read Status Register", testnum);
	// read status register
	cs_active;
	do_qpi_cmd(8'h05);
	rd_data(8'h42);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Set Write Enable", testnum);
	// Write Enable
	cs_active;
	do_qpi_cmd(8'h06);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Set Write Disable", testnum);
	// Write Disable
	cs_active;
	do_qpi_cmd(8'h04);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Write 4 bytes @ 000010", testnum);
        // Write some data
	cs_active;
	do_qpi_cmd(8'h02);
        do_addr(24'h00_00_10);
	wr_data(8'h5a);
	wr_data(8'h00);
	wr_data(8'ha5);
	wr_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Write 4 bytes @ 000014", testnum);
        // Write some data
	cs_active;
	do_qpi_cmd(8'h02);
        do_addr(24'h00_00_14);
	wr_data(8'h11);
	wr_data(8'h22);
	wr_data(8'h33);
	wr_data(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read 4 bytes @ 000010", testnum);
        // Read data back
	cs_active;
	do_qpi_cmd(8'h03);
    do_addr(24'h00_00_10);
	rd_data(8'h5a);
	rd_data(8'h00);
	rd_data(8'ha5);
	rd_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read 4 bytes @ 000014", testnum);
        // Read data back
	cs_active;
	do_qpi_cmd(8'h03);
        do_addr(24'h00_00_14);
	rd_data(8'h11);
	rd_data(8'h22);
	rd_data(8'h33);
	rd_data(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read 4 bytes @ 000010, set XIP Mode", testnum);
        // Fast Read data back
	cs_active;
	do_qpi_cmd(8'h0b);
        do_addr(24'h00_00_10);
	do_cmd(8'hef);		// mode byte - set XIP
	rd_data(8'h5a);
	rd_data(8'h00);
	rd_data(8'ha5);
	rd_data(8'hff);
        cs_inactive;
	testnum++;
 	$display("*********************         %2d. Fast Read 4 bytes @ 000014, reset XIP Mode", testnum);
        // Fast Read data back
	cs_active;
        do_addr(24'h00_00_14);
	do_cmd(8'hff);		// mode byte - reset XIP
	rd_data(8'h11);
	rd_data(8'h22);
	rd_data(8'h33);
	rd_data(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Set Sleep Mode", testnum);
	// set sleep mode
	cs_active;
	do_qpi_cmd(8'hb9);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Exit Sleep Mode(Wake)", testnum);
	// exit sleep mode
	cs_active;
	do_qpi_cmd(8'hab);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Set Sleep Mode", testnum);
	// set sleep mode
	cs_active;
	do_qpi_cmd(8'hb9);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Send Command(not Wake) in Sleep Mode (should generate error)", testnum);
	// Write Enable
	cs_active;
	do_qpi_cmd(8'h06);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Exit Sleep Mode(Wake)", testnum);
	// exit sleep mode
	cs_active;
	do_qpi_cmd(8'hab);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read Device ID", testnum);
	// read device ID
	cs_active;
	do_qpi_cmd(8'h4b);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0111);	// read the device ID
	rd_data(8'b0110_1011);
	rd_data(8'b0001_0001);
	rd_data(8'b0001_0001);
	rd_data(8'b0001_0001);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect", testnum);
        // Tamper Detect Command
	cs_active;
	do_qpi_cmd(8'h17);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Read Data after Tamper Detect(no TDETX)", testnum);
        // Read data back
	cs_active;
	do_qpi_cmd(8'h03);
        do_addr(24'h00_00_10);
	rd_data(8'h5a);
	rd_data(8'h00);
	rd_data(8'ha5);
	rd_data(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect before TDETX (should generate error)", testnum);
        // Tamper Detect Command
	cs_active;
	do_qpi_cmd(8'h17);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect Exit(TDETX)", testnum);
	// Tampser Detect Exit Command
	cs_active;
	do_qpi_cmd(8'h07);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect after TDETX (should not generate error)", testnum);
        // Tamper Detect Command
	cs_active;
	do_qpi_cmd(8'h17);
	do_cmd(8'hff);		// mode byte (must be FF)
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
	rd_data(8'b0000_0000);	// read the tamper data
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Tamper Detect Exit(TDETX)", testnum);
	// Tampser Detect Exit Command
	cs_active;
	do_qpi_cmd(8'h07);
        cs_inactive;
	testnum++;
	$display("**************************************************************************************");
	$display("*********************          Quad Data Tests          ******************************");
	$display("**************************************************************************************");
	$display("*********************         %2d. Fast Read Quad Output, set XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'h6B);
        do_addr(24'h00_00_10);
	do_qpi_cmd(8'hEF); 	//set XIP mode
	rd_qdata(8'h5a);
	rd_qdata(8'h00);
	rd_qdata(8'ha5);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Output, reset XIP mode", testnum);
	cs_active;
        do_addr(24'h00_00_14);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Output, no XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'h6B);
        do_addr(24'h00_00_10);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h5a);
	rd_qdata(8'h00);
	rd_qdata(8'ha5);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Output, no XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'h6B);
        do_addr(24'h00_00_14);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, set XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'hEB);
        do_qaddr(24'h00_00_10);
	do_qpi_cmd(8'hEF); 	//reset XIP mode
	rd_qdata(8'h5a);
	rd_qdata(8'h00);
	rd_qdata(8'ha5);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, reset XIP mode", testnum);
	cs_active;
        do_qaddr(24'h00_00_14);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, no XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'hEB);
        do_qaddr(24'h00_00_10);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h5a);
	rd_qdata(8'h00);
	rd_qdata(8'ha5);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, no XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'hEB);
        do_qaddr(24'h00_00_14);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Write Quad Data", testnum);
	cs_active;
	do_qpi_cmd(8'h32);
        do_addr(24'h00_00_10);
	wr_qdata(8'ha5);
	wr_qdata(8'hff);
	wr_qdata(8'h5a);
	wr_qdata(8'h00);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Write Quad Data", testnum);
	cs_active;
	do_qpi_cmd(8'h32);
        do_addr(24'h00_00_14);
	wr_qdata(8'h44);
	wr_qdata(8'h33);
	wr_qdata(8'h22);
	wr_qdata(8'h11);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, no XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'hEB);
        do_qaddr(24'h00_00_10);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'ha5);
	rd_qdata(8'hff);
	rd_qdata(8'h5a);
	rd_qdata(8'h00);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, no XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'hEB);
        do_qaddr(24'h00_00_14);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h44);
	rd_qdata(8'h33);
	rd_qdata(8'h22);
	rd_qdata(8'h11);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Write Quad Address and Data", testnum);
	cs_active;
	do_qpi_cmd(8'h12);
        do_qaddr(24'h00_00_10);
	wr_qdata(8'h5a);
	wr_qdata(8'h00);
	wr_qdata(8'ha5);
	wr_qdata(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Write Quad Address and Data", testnum);
	cs_active;
	do_qpi_cmd(8'h12);
        do_qaddr(24'h00_00_14);
	wr_qdata(8'h11);
	wr_qdata(8'h22);
	wr_qdata(8'h33);
	wr_qdata(8'h44);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, set XIP mode", testnum);
	cs_active;
	do_qpi_cmd(8'hEB);
        do_qaddr(24'h00_00_10);
	do_qpi_cmd(8'hEF); 	//set XIP mode
	rd_qdata(8'h5a);
	rd_qdata(8'h00);
	rd_qdata(8'ha5);
	rd_qdata(8'hff);
        cs_inactive;
	testnum++;
	$display("*********************         %2d. Fast Read Quad Address and Data, reset XIP mode", testnum);
	cs_active;
        do_qaddr(24'h00_00_14);
	do_qpi_cmd(8'hFF); 	//reset XIP mode
	rd_qdata(8'h11);
	rd_qdata(8'h22);
	rd_qdata(8'h33);
	rd_qdata(8'h44);
        cs_inactive;
	// Disable QPI Mode
	$display("*********************         %2d. Disable QPI Mode", testnum);
	cs_active;
	do_qpi_cmd(8'hFF);
        cs_inactive;
	cs_active;
	testnum++;
	$display("**************************************************************************************");
	$display("*****************         Write Protection Tests        ******************************");
	$display("**************************************************************************************");
	$display("*****************             Status Register           ******************************");
	$display("**************************************************************************************");
	wp_en = 1;
	wp_in = 0;
	hold_en = 1;
	hold_in = 1;
	$display("*********************          %2d. Set Write Disable", testnum);
	// Write Disable
	cs_active;
	do_cmd(8'h04);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Attempt Write to Status Register (should be blocked by WEL)", testnum);
	// write status register to all 1s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'hFF);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Set Write Enable", testnum);
	// Write Disable
	cs_active;
	do_cmd(8'h06);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Attempt Write to Status Register (SRWD=1, should be blocked by wp_b)", testnum);
	// write status register to all 1s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'hFF);
        cs_inactive;
	testnum++;
        wp_in = 1'b1;
	$display("*********************          %2d. Attempt Write to Status Register (SRWD=1, should be allowed by wp_b)", testnum);
	// write status register to all 1s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h00);
        cs_inactive;
	testnum++;
	$display("*********************          %2d. Attempt Write to Status Register (SRWD=0, should ignore WP_B(=%b)", testnum, wp_in);
	// write status register to all 1s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h00);
        cs_inactive;
	testnum++;
        wp_in = 1'b0;
	$display("*********************          %2d. Attempt Write to Status Register (SRWD=0, should ignore WP_B(=%b)", testnum, wp_in);
	// write status register to all 1s
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h00);
        cs_inactive;
	testnum++;
	$display("**************************************************************************************");
	$display("*********************          Memory Blocks            ******************************");
	$display("**************************************************************************************");
	$display("*********************      All Memory Protected         ******************************");
	$display("*********************          %2d. Write to Status Register (BP=11)", testnum);
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h0C);
        cs_inactive;
	$display("*********************               Attempt Write to Memory");
	$display("*********************               Write 1 byte @ 000000 (should generate warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h00_00_00);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ FFFFFF (should generate warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'hFF_FF_FF);
	wr_data(8'ha5);
        cs_inactive;
	testnum++;
	$display("**************************************************************************************");
	$display("*********************  Upper Half Memory Protected      ******************************");
	$display("**************************************************************************************");
	$display("*********************          %2d. Write to Status Register (BP=10)", testnum);
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h08);
        cs_inactive;
	$display("*********************               Attempt Write to Memory");
	$display("*********************               Write 1 byte @ 000000 (no warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h00_00_00);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ 7FFFFF (no warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h7F_FF_FF);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ 800000 (should generate warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h80_00_00);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ FFFFFF (should generate warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'hFF_FF_FF);
	wr_data(8'ha5);
        cs_inactive;
	testnum++;
	$display("**************************************************************************************");
	$display("*********************  Upper Quarter Memory Protected   ******************************");
	$display("**************************************************************************************");
	$display("*********************          %2d. Write to Status Register (BP=01)", testnum);
	cs_active;
	do_cmd(8'h01);
	wr_data(8'h04);
        cs_inactive;
	$display("*********************               Attempt Write to Memory");
	$display("*********************               Write 1 byte @ 000000 (no warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'h00_00_00);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ BFFFFF (no warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'hBF_FF_FF);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ C00000 (should generate warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'hC0_00_00);
	wr_data(8'ha5);
        cs_inactive;
	$display("*********************               Write 1 byte @ FFFFFF (should generate warning)");
	cs_active;
	do_cmd(8'h02);
        do_addr(24'hFF_FF_FF);
	wr_data(8'ha5);
        cs_inactive;
	testnum++;
/*
*/
end

task cs_active;
  cs_b = 1'b0;
  #(TCSS-TSU);
endtask

task cs_inactive;
  #TCSH;
  cs_b = 1'b1;
  read <= 0;
  #TCSW;
endtask


task do_cmd;
input [7:0] cmd;

   	if(cmd == 8'h03) read = 1;  //slow the clock down for single SPI reads

    fork	// clock process
	begin
        #TSU;
		for(i=0;i<8;i=i+1) begin
			sck = 1'b1;
			if(read) #RD_CKHI; 	
			else #CKHI;
			sck = 1'b0;
			if(i==7) begin
				if(read) #(RD_CKLO-TSU);
				else #(CKLO-TSU);
			end 
			else begin 
				if(read) #RD_CKLO;
				else #CKLO;
			end
		end
	end
	begin	// data process
        si_en = 1'b1;
		for(j=0;j<8;j=j+1) begin
			si_in = cmd[7-j];
			if(read) #(RD_CKHI+RD_CKLO);
			else #(CKHI+CKLO);
		end	
        si_en = 1'b0;
	end
	join


endtask

task do_qpi_cmd;
input [7:0] cmd;

    fork	// clock process
	begin
        #TSU;
		for(i=0;i<2;i=i+1) begin
			sck = 1'b1;
			#CKHI;
			sck = 1'b0;
			if(i==1) #(CKLO-TSU); else #CKLO;
		end
	end
	begin	// data process
        qpi_en = 1'b1;
		for(j=0;j<2;j=j+1) begin
			case(j)
			0: {hold_in, wp_in, so_in, si_in} = cmd[7:4];
			1: {hold_in, wp_in, so_in, si_in} = cmd[3:0];
			endcase
			#(CKHI+CKLO);
		end	
        qpi_en = 1'b0;
	end
	join
endtask

task do_addr;
input [23:0] addr;

    fork	// clock process
	begin
        #TSU;
		for(i=0;i<24;i=i+1) begin
			sck = 1'b1;
			if(read) #RD_CKHI;
			else #CKHI;
			sck = 1'b0;
            if(i==23) begin
            	if(read) #(RD_CKLO);
            	else #(CKLO);
            end else begin
            	if(read) #RD_CKLO;
            	else #CKLO;
            end
		end
	end
	begin	// data process
        si_en = 1'b1;
		for(j=0;j<24;j=j+1) begin
			si_in = addr[23-j];
			if(read) #(RD_CKHI+RD_CKLO);
			else #(CKHI+CKLO);
		end	
        si_en = 1'b0;
	end
	join
endtask

task do_qaddr;
input [23:0] addr;

    fork	// clock process
	begin
        #TSU;
		for(i=0;i<6;i=i+1) begin
			sck = 1'b1;
			#CKHI;
			sck = 1'b0;
            if(i==2) #(CKLO); else #CKLO;
		end
	end
	begin	// data process
        qpi_en = 1'b1;
		for(j=0;j<6;j=j+1) begin
			case(j)
			0: {hold_in, wp_in, so_in, si_in} = addr[23:20];
			1: {hold_in, wp_in, so_in, si_in} = addr[19:16];
			2: {hold_in, wp_in, so_in, si_in} = addr[15:12];
			3: {hold_in, wp_in, so_in, si_in} = addr[11:08];
			4: {hold_in, wp_in, so_in, si_in} = addr[07:04];
			5: {hold_in, wp_in, so_in, si_in} = addr[03:00];
			endcase
			#(CKHI+CKLO);
		end	
        qpi_en = 1'b0;
	end
	join
endtask

task wr_qdata;
input [7:0] data;

        fork	// clock process
	begin
                #TSU;
		for(i=0;i<2;i=i+1) begin
			sck = 1'b1;
			#CKHI;
			sck = 1'b0;
			if(i==1) #(CKLO-TSU); else #CKLO;
		end
	end
	begin	// data process
                qpi_en = 1'b1;
		for(j=0;j<2;j=j+1) begin
			case(j)
			0: {hold_in, wp_in, so_in, si_in} = data[7:4];
			1: {hold_in, wp_in, so_in, si_in} = data[3:0];
			endcase
			#(CKHI+CKLO);
		end	
                qpi_en = 1'b0;
	end
	join
endtask

task wr_data;
input [7:0] data;

        fork	// clock process
	begin
                #TSU;
		for(i=0;i<8;i=i+1) begin
			sck = 1'b1;
			#CKHI;
			sck = 1'b0;
			if(i==7) #(CKLO-TSU); else #CKLO;
		end
	end
	begin	// data process
                si_en = 1'b1;
		for(j=0;j<8;j=j+1) begin
			si_in = data[7-j];
			#(CKHI+CKLO);
		end	
                si_en = 1'b0;
	end
	join
endtask

task rd_data;
input [7:0] chk_data;

reg [7:0] data;

    fork	// clock process
	begin
        #TSU;
		for(i=0;i<8;i=i+1) begin
			sck = 1'b1;
			if(read) #RD_CKHI;
			else #CKHI;
			sck = 1'b0;
			if(i==7) #(CKLO-TSU); 
			else begin
				if(read) #RD_CKLO;
				else #CKLO;
			end                      
		end
	end
	begin	// data process
		for(j=0;j<8;j=j+1) begin
			@(negedge sck);
			data[7-j]= so;
			if(j==7)begin 
				if(read) #(RD_CKLO-TSU); 
				else #(CKLO-TSU); 
			end
			else begin
				if(read) #(RD_CKHI+RD_CKLO);
				else #(CKHI+CKLO);
			end
		end	
	end
	join
	if(data !== chk_data) begin
		$display("%m Time: %t - ERROR Recieved data(%h) does not match expected data(%h)", $time, data, chk_data);
                if(STOP_ON_ERROR) $stop;
        end else if(VERBOSE_LEVEL > 4) $display("%m Time: %t - Recieved data(%h) matches expected data(%h)", $time, data, chk_data);
endtask

task rd_qdata;
input [7:0] chk_data;

reg [7:0] data;

        fork	// clock process
	begin
                #TSU;
		for(i=0;i<2;i=i+1) begin
			sck = 1'b1;
			#CKHI;
			sck = 1'b0;
			if(i==1) #(CKLO-TSU); else #CKLO;
                      
		end
	end
	begin	// data process
		for(j=0;j<2;j=j+1) begin
			@(negedge sck);
			case(j)
			0: data[7:4]= {hold_b, wp_b, so, si};
			1: data[3:0]= {hold_b, wp_b, so, si};
			endcase
			if(j==1) #(CKLO-TSU); else #(CKHI+CKLO);
		end	
	end
	join
	if(data !== chk_data) begin
		$display("%m Time: %t - ERROR Recieved data(%h) does not match expected data(%h)", $time, data, chk_data);
                if(STOP_ON_ERROR) $stop;
        end else if(VERBOSE_LEVEL > 4) $display("%m Time: %t - Recieved data(%h) matches expected data(%h)", $time, data, chk_data);
endtask

endmodule
