/*
 * Everspin Technologies - The MRAM Company
 *  MR10Q010-EVAL SHIELD
 *  MR10Q010CSCR - 1Mb MRAM (128Kx8), 40MHz SPI
 */


#include "mbed.h"

#define SERIAL_BAUD 115200
#define SPI_FREQ 40000000

//SPI Mode Commands for MR10Q010
#define CMD_RDSR 0x05
#define CMD_WREN 0x06
#define CMD_WRDI 0x04
#define CMD_WRSR 0x01
#define CMD_READ 0x03
#define CMD_FREAD 0x0B
#define CMD_WRITE 0x02
#define CMD_SLEEP 0xB9
#define CMD_WAKE 0xAB
#define CMD_TDET 0x17
#define CMD_RDID 0x48

//pin setup for interfacing with the device
SPI MR10Q010(D11, D12, D13); // mosi, miso, sclk
DigitalOut SS(D10);
DigitalOut WP(D9);
DigitalOut HOLD(D8);

//create a serial object for output to the terminal
Serial pc(SERIAL_TX, SERIAL_RX);

char wrbuf[256] = {0};
char rdbuf[256] = {0};

//MR10Q010 read wrapper function
int mram_read (int addr, char *buf, int len)
{
    int i;

    SS=0;
    MR10Q010.write(CMD_READ);
    MR10Q010.write((addr >> 16) & 0xff);
    MR10Q010.write((addr >> 8) & 0xff);
    MR10Q010.write(addr & 0xff);

    for (i = 0; i < len; i ++) {
        rdbuf[i] = MR10Q010.write(0);
    }

    SS=1;
    return 1;

}

//MR10Q010 write wrapper function
int mram_write (int addr, char *buf, int len)
{
    int i;

    SS=0;
    MR10Q010.write(CMD_WRITE);
    MR10Q010.write((addr >> 16) & 0xff);
    MR10Q010.write((addr >> 8) & 0xff);
    MR10Q010.write(addr & 0xff);

    for (i = 0; i < len; i ++) {
        MR10Q010.write(wrbuf[i]);
    }

    SS=1;
    return 1;
}

//Enable writes
int mram_wren (void)
{

    WP=1;

    SS=0;
    MR10Q010.write(CMD_WREN);
    SS=1;

    return 1;
}

//Disable writes
int mram_wrdi (void)
{

    SS=0;
    MR10Q010.write(CMD_WRDI);
    SS=1;

    WP=0;
    return 1;
}



//main function showing basic mram function usage with a nvm performance test
int main()
{
    int i,j;
    int run_cnt = 0;
    Timer t;

    HOLD=1;   //make sure HOLD is released

    MR10Q010.frequency(SPI_FREQ); //setup SPI frequency
    pc.baud(SERIAL_BAUD);  //setup UART baud
    memset(rdbuf,0,256);  //init read buffer

    pc.printf("\r\n\r\n");
    pc.printf("================================================\r\n");
    pc.printf("||           Everspin Technologies            ||\r\n");
    pc.printf("||              The MRAM Company              ||\r\n");
    pc.printf("================================================\r\n");
    pc.printf("||           MR10Q010 mbed Example            ||\r\n");
    pc.printf("||            1Mb MRAM, 40MHz SPI             ||\r\n");
    pc.printf("================================================\r\n");

    //check the last word of the mem to see if this demo has ever been run on the mram...  initialize if not.
    mram_read(0x1FFFF, rdbuf, 1);
    if(rdbuf[0] != 0xAA) {
        mram_wren();  //Enable writes to MRAM
        pc.printf("First time running...  init MRAM!\r\n");
        memset(wrbuf,0,256);
        mram_write(0, wrbuf, 256);
        wrbuf[0] = 0xAA;
        mram_write(0x1FFFF, wrbuf, 1);
        mram_wrdi();
    }

    while(1) {
        
        mram_read(0, rdbuf, 256);
        run_cnt = rdbuf[0] + 1;

        pc.printf("MRAM Read: Show Test Count\r\n");
    
        for (i = 0; i < 256; i ++) {
            pc.printf(" %02x", rdbuf[i]);
            if ((i & 0x0f) == 0x0f)
                pc.printf("\r\n");
        }
    
        pc.printf("\r\nWrite/Read time for 1048576 bits @ 40MHz:\r\n\r\n");
        mram_wren();
        pc.printf("MRAM writing\r\n");
        t.reset();
        t.start();
        for (i = 0; i < 0x20000; i += 256) {
            for(j = 0; j < 256; j++) {
                wrbuf[j] = run_cnt & 0xff;
            }
            mram_write(i, wrbuf, 256);
            if ((i & 0x0fff) == 0) pc.printf(".");
        }
        t.stop();
        pc.printf("\r\nTime: %f s   Rate: %f KBytes/sec\r\n\r\n", t.read(), (float)0x20000 / 1024 / t.read());
        mram_wrdi();
    
        printf("MRAM reading (and error checking)\r\n");
        t.reset();
        t.start();
        for (i = 0; i < 0x20000; i += 256) {
            mram_read(i, rdbuf, 256);
            for(j = 0; j < 256; j++) {
                if (rdbuf[j] != (run_cnt & 0xff)) {
                    pc.printf("Error %d\r\n", i+j);
                    break;
                }
            }
            if ((i & 0x0fff) == 0) pc.printf(".");
        }
        t.stop();
        pc.printf("\r\nTime: %f s   Rate: %f KBytes/sec\r\n\r\n", t.read(), 0x20000 / 1024 / t.read());
    
        mram_wren();
        wrbuf[0] = 0xAA;  //rewrite init word
        mram_write(0x1FFFF, wrbuf, 1);
        memset(rdbuf, 0, 256);
        mram_wrdi();
        
        pc.printf("Press any key to run again...\r\n\r\n\r\n");
        
        while(!pc.getc());

    }

}
