/*********************************************************************

-- VERILOG Model for the 256K Serial MRAM
-- Device Part Number: MR25H256
-- Device Data Sheet: http://www.everspin.com/PDF/EST_MR25H256_prod.pdf

-- Model Revision: 1.0
-- Model Release Date: September 2010

---------------------------------------------------------------------
-- This Verilog models is provided "as is" without warranty of
-- any kind, included but not limited to, implied warranty of
-- merchant ability and fitness for a particular purpose.
-- Please refer to the product datasheet for all specifications. 
-- The Product Datasheet serves as the final arbiter of product performance
---------------------------------------------------------------------

***********************************************************************/

`timescale 1 ns/1 ns
`define MRAM10
`include "CONFIG_MR25H256.v"

`timescale 1ns/10ps

module MR25H10 (SI, SO, SCK, CS_N, WP_N, HOLD_N);

   input                SI;                             // serial data input
   input                SCK;                            // serial data clock
   input                CS_N;                           // chip select - active low
   input                HOLD_N;                         // interface suspend - active low
   input 		WP_N;				// Write Protect Signal
   output               SO;                             // serial data output


// *******************************************************************************************************
// **   Set Variables                                                                                   **
// *******************************************************************************************************

   reg  [31:00]         DataShifterI;                   // serial input data shifter
   reg  [07:00]         TempRegisterOut;                   // serial output data shifter
   reg  [31:00]         BitCounter;                     // serial input bit counter
   reg  [31:00]         TempAddressRegister;
   reg  [07:00]         TempInstRegister;
   reg  [07:00]         InstRegister;
   reg  [07:00]         TempDataRegister;                   // instruction register
   reg  [14:00]         AddressRegister;                   // address register
   reg  [14:00]         LocalAddress;			// Temp Address register for writes and reads

   reg                  SOT;
   
   reg  [07:00]         StatusRegister;			// Status Register

   wire                 InstructionREAD;                // decoded instruction byte
   wire                 InstructionRDSR;                // decoded instruction byte
   wire                 InstructionWRSR;                // decoded instruction byte
   wire                 InstructionWRITE;               // decoded instruction byte
   wire                 InstructionWRDI;
   wire			InstructionSLEEP;
   wire			InstructionWAKE;
   wire			InstructionWREN;


   reg			AddressCount;
   
   reg			SETSLEEPSTATE;			// Set Sleep State


   reg                  OpMode0;                        // operation mode
   reg                  OpMode1;                        // operation mode
   
   
   wire                 Hold;                           // hold function

   reg  [07:00]         MemoryBlock [0:131072];         // MRAM memory array (131072x8)


   integer              i;

   integer              tV;                             // timing parameter
   integer              tHZ;                            // timing parameter
   integer              tHV;                            // timing parameter
   integer              tDIS;                           // timing parameter
   
`define WREN      8'b0000_0110                          // Write Enable
`define WRDI      8'b0000_0100                          // Write Disable
`define RDSR      8'b0000_0101                          // Read Status Register instruction
`define WRSR      8'b0000_0001                          // Write Status Register instruction
`define READ      8'b0000_0011                          // Read instruction
`define WRITE     8'b0000_0010                          // Write instruction
`define SLEEP     8'b1011_1001                          // Enter Sleep
`define WAKE      8'b1010_1011                          // Exit Sleep



assign SO = SOT;

// *******************************************************************************************************
// **   INITIALIZATION                                                                                  **
// *******************************************************************************************************

   initial begin
      	
 	 i = 0;	 
 	  tV   = 9;                                     // output valid from SCK low
         tHZ  = 10;                                     // HOLD_N low to output high-z
         tHV  = 50;                                     // HOLD_N high to output valid
         tDIS = 20;  
   end
   


//Reset
   always @(negedge CS_N) begin
     BitCounter <= 0;
     AddressCount <= 0;
     i = 0;
   end
   
//Output Disable  
   always @(posedge CS_N) begin

      SOT <= #(tDIS) 1'bz;
   end

// Input Data Shift
   always @(posedge SCK) begin
      if (Hold == 0) begin
         if (CS_N == 0)         DataShifterI <= {DataShifterI[30:00],SI};
      end
   end

// Input Bit Count
   always @(posedge SCK) begin
      if (Hold == 0) begin
         if (CS_N == 0)         BitCounter <= BitCounter + 1;
      end
   end

// Parse thro. Input Bits for the Instruction Register
   always @(posedge SCK) begin
      if (Hold == 0) begin
         if (BitCounter == 7)   begin
         TempInstRegister <= {DataShifterI[06:00],SI};
         
          for (i=0; i < 8; i = i +1) begin
           InstRegister [i] <= TempInstRegister [7 - i];
          end
         end
      end
   end

   assign InstructionWREN = (InstRegister[7:0] == `WREN);
   assign InstructionWRDI = (InstRegister[7:0] == `WRDI);
   assign InstructionRDSR = (InstRegister[7:0] == `RDSR);
   assign InstructionWRSR  = (InstRegister[7:0] == `WRSR);
   assign InstructionREAD  = (InstRegister[7:0] == `READ);
   assign InstructionWRITE = (InstRegister[7:0] == `WRITE);
   assign InstructionSLEEP = (InstRegister[7:0] == `SLEEP);
   assign InstructionWAKE = (InstRegister[7:0] == `WAKE);

// Latch in Address - 16 bits after Instruction
   always @(posedge SCK) begin
      if (Hold == 0) begin
         if (BitCounter == 23) 

              TempAddressRegister <= {DataShifterI[23:08],SI};
              
              for (i=0; i <15; i = i +1) begin
                 AddressRegister[i] <= TempAddressRegister [23 -i];
              end
	      
              LocalAddress[14:00] <= AddressRegister[14:00];
      end
   end

//Sleep based on Instruction
  always @(posedge SCK) begin
    if (Hold == 0) begin
      if (InstructionSLEEP) begin
         SETSLEEPSTATE <= 1'b1;
      end
    end
   end

//Wake based on Instruction
    always @(posedge SCK) begin
    if (Hold == 0) begin
      if (InstructionWAKE) begin
         SETSLEEPSTATE <= 1'b0;
      end
    end
   end

// Update Status Register bit to allow writes
  always @(posedge SCK) begin
      if (Hold == 0) begin
        if (InstructionWREN & (SETSLEEPSTATE == 1'b0)) begin
          StatusRegister[1] <= 1'b0;
        end 
       end
  end

//Disable Status Register Writes
   always @(posedge SCK) begin
      if (Hold == 0) begin
        if (InstructionWRDI & (SETSLEEPSTATE == 1'b0)) begin
          StatusRegister[1] <= 1'b1;
        end 
       end
   end
 
// Status Register Updates
  always @(posedge SCK) begin
      if (Hold == 0) begin
         if ((BitCounter == 15) & InstructionWRSR & (StatusRegister[1] == 1'b0) & (SETSLEEPSTATE == 1'b0)) begin
            StatusRegister[1] <= DataShifterI[14];
 	    StatusRegister[2] <= DataShifterI[13];
            StatusRegister[3] <= DataShifterI[12];
	    StatusRegister[7] <= DataShifterI[8];
         end
      end
   end

   
//Data Transfer into Array

always @(posedge SCK) begin
      if (Hold == 0) begin
        if (SETSLEEPSTATE == 1'b0) begin

	  if ((BitCounter >= 23) & InstructionWRITE) begin

	       TempDataRegister[7-i] <= SI;
               i = i + 1;
               
               if(i == 7) begin

		 i = 0;
		 
		 // The next 3 blocks of code are for block wise protection
		 
                 if (StatusRegister[3:2] == 2'b00) begin
		
                   LocalAddress = LocalAddress + AddressCount;
                   MemoryBlock [LocalAddress] <= TempDataRegister;
		   AddressCount = AddressCount + 1;
                   
                  
 		    if (LocalAddress == 15'b111111111111111) begin
                      LocalAddress = 15'b000000000000000;
                      AddressCount = 0;
                    end

                 end

                 if (StatusRegister[3:2] == 2'b10) begin
                   LocalAddress = LocalAddress + AddressCount;
                   if (LocalAddress[16:00] > 15'b001111111111111) begin
                    AddressCount = AddressCount + 1;
		    
		    if (LocalAddress == 15'b111111111111111) begin
                      LocalAddress = 15'b000000000000000;
                      AddressCount = 0;
                    end

                   end 
                   else begin
                    MemoryBlock [LocalAddress] <= TempDataRegister;
                    AddressCount = AddressCount + 1;
                   end
                  end

		  if (StatusRegister[3:2] == 2'b01) begin
                   LocalAddress = LocalAddress + AddressCount;
                   if (LocalAddress[16:00] > 15'b011111111111111) begin
                    AddressCount = AddressCount + 1;
		    
		    if (LocalAddress == 15'b111111111111111) begin
                      LocalAddress = 15'b000000000000000;
                      AddressCount = 0;
                    end

                   end 
                   else begin
                    MemoryBlock [LocalAddress] <= TempDataRegister;
                    AddressCount = AddressCount + 1;
                   end
	          end
               
               end

         end
        end
      end
   end


// Read Cycle

   always @(negedge SCK) begin

     if (Hold == 0) begin
        if (SETSLEEPSTATE == 1'b0) begin

	        if ((BitCounter >= 24) & InstructionREAD) begin

	            TempRegisterOut <= MemoryBlock[LocalAddress];
           
            if (i < 8) begin
               SOT <= #(tV) TempRegisterOut[i];
               i = i + 1;
               
              if ( i == 8) begin
                i = 0;
                LocalAddress = LocalAddress + AddressCount;
                AddressCount = AddressCount + 1;
                  if (LocalAddress == 15'b111111111111111) begin
                    LocalAddress = 15'b000000000000000;
                    AddressCount = 0;
                  end
              end
            end
          end 
         else begin
          SOT <= 1'bz;
         end
     end
   end
 end

// Simple Timing Checks
   specify
   
           specparam
                                     // CS_N high to output disable
               tHI  = 25,                                  // Clock high time
	       tLO  = 25,                                  // Clock low time
	       tSU  = 10,                                  // Data setup time
	       tHD  = 10,                                  // Data hold time
	       tHS  = 10,                                  // HOLD_N setup time
	       tHH  = 10,                                  // HOLD_N hold time
	       tCSD = 25,                                  // CS_N disable time
	       tCSS = 25,                                  // CS_N setup time
	       tCSH = 50,                                  // CS_N hold time
               tCLD = 25;                                  // Clock delay time
            
      $width (posedge SCK,  tHI);
      $width (negedge SCK,  tLO);
      $width (posedge CS_N, tCSD);

      $setup (SI, posedge SCK, tSU);
      $setup (negedge CS_N, posedge SCK, tCSS);
      $setup (negedge SCK, negedge HOLD_N, tHS);
      $setup (posedge CS_N, posedge SCK, tCLD);

      $hold  (posedge SCK, SI,   tHD);
      $hold  (posedge SCK , posedge CS_N, tCSH);
      $hold  (posedge HOLD_N, posedge SCK,  tHH);
  endspecify

endmodule
