/*********************************************************************

-- VERILOG Model for a 128K x 8 MRAM
-- Device Part Number: MR0A08B
-- Device Data Sheet: http://www.everspin.com/PDF/EST_MR0A08B_prod.pdf

-- Model Revision: 1.0
-- Model Release Date: September 2010

---------------------------------------------------------------------
-- This Verilog models is provided "as is" without warranty of
-- any kind, included but not limited to, implied warranty of
-- merchant ability and fitness for a particular purpose.
-- Please refer to the datasheet for all specifications. 
-- Datasheet serves as the final arbiter of product performance
---------------------------------------------------------------------

***********************************************************************/

`timescale 1 ns/1 ns
`define MRAM35
`include "config_MR0A08B.v"

/***********************************************************************
Top Level Signals

Address - 17 Bit Address Bus
DQ      - 8 Bit Data Bus
G_b	- Active Low Chip Select
E_b	- Active Low Output Enable
W_b	- Active Low Write Enable

************************************************************************/
module MR0A08A (Address, DQ, G_b, E_b, W_b);

/* All Models are generated from same base using the following parameters */

parameter Bits_in_Address_Bus = 15;
parameter Bytes_in_Array = 131072;
parameter Bus_Width =  8;
						
input [Bits_in_Address_Bus-1:0] Address;		
inout [Bus_Width-1:0]  DQ;				
input E_b, W_b, G_b;

reg   [Bus_Width-1:0] Volatile_memory [0:Bytes_in_Array-1];	
reg   [Bus_Width-1:0] MRAM_Array [0:Bytes_in_Array-1];	//Mram_array
reg   [Bus_Width-1:0] data_temp, Write_Data, Read_Data;	//Data Buffers
reg   [Bits_in_Address_Bus-1:0] Write_A1, Write_A2, Read_A1, Read_A2;		//Address Buffers
reg   W_Delay, ini_cebar, ini_webar, ini_wecebar,ini_ceset,ini_weset;
reg   RD_Trigger1, RD_Trigger2, WR_Trigger1, WR_Trigger2; 

time write_E_bar_start,write_W_bar_start,write_address,write_data,write_end;
time read_address,read_E_bar_start,read_W_bar_start,E_Transition, write_address1,write_data1;


/************************************************************************************************

Model Start

*************************************************************************************************/

wire [7:0] DQ =  (W_Delay) ?  Read_Data [7:0] : 8'bz ;


initial
begin
    WR_Trigger1 = 1'b0;
    WR_Trigger2 = 1'b0;
    RD_Trigger1 =1'b0;
    RD_Trigger2 =1'b0;
    read_address =0;  
    data_temp = 8'hzz;       
end



/**********************************************************************************************

Write Routines

***********************************************************************************************/
always@( E_b or W_b or G_b or Address or DQ )
begin
	if ((E_b==1'b0) && (W_b==1'b0))	// Write in progress!
	begin
		Write_A1 <= Address;
		Write_A2 <= Write_A1;
		Write_Data  <= DQ ; 
		Volatile_memory[Write_A1] <=  Write_Data ;
	end
end


always@(negedge E_b)
begin
	write_E_bar_start <= $time;
	read_E_bar_start <=$time;
	ini_cebar <= 1'b0;
	ini_wecebar <=1'b0;
	WR_Trigger1 <= 1'b0;
	WR_Trigger2 <= 1'b0;
	
	
	if (($time - write_address) >= `tavwl)	// Address Setup Time
		ini_ceset <= 1'b1; 
	else
		ini_ceset <= 1'b0;
end


always@(posedge E_b)
begin
   ini_ceset <= 1'b0;
        if (($time - write_E_bar_start) >= `teleh)	// If CE bar to end of write
	  begin
	    if ( (W_b == 1'b0) && ( ($time - write_W_bar_start) >= `twlwh) && (ini_ceset == 1'b1) ) // WE to end of write
	      begin
		Write_A2 <= Write_A1;
		Volatile_memory[Write_A1] <= Write_Data[15:0];  
		ini_cebar <= 1'b1;
	      end
	  end            
end


always@(negedge W_b)
begin
	write_W_bar_start <= $time;
	ini_webar <= 1'b0;
	ini_wecebar <=1'b0;
	WR_Trigger1 <= 1'b0;
	WR_Trigger2 <= 1'b0;
	
	if ( ($time - write_address) >= `tavwl)
		ini_weset <= 1'b1; 
	else
		ini_weset <= 1'b0;
		
	#`twlqz W_Delay <= W_b;
end


always@(posedge W_b)
begin
   ini_ceset <= 1'b0;
   ini_weset <= 1'b0;
   write_end <= $time; 
	W_Delay <= W_b;
	read_W_bar_start <=$time;
	ini_weset <= 1'b0;
	if (($time - write_W_bar_start) >=`twlwh)
	begin
		if ( (E_b == 1'b0) && ( ($time - write_E_bar_start) >= `teleh) && (ini_weset == 1'b1) )
		begin
			Write_A2 <= Write_A1;  
			Volatile_memory[Write_A1] <= Write_Data[15:0]; 
			ini_webar <= 1'b1;
		end 

	end       

end    

always@( E_b && W_b )
begin
     if ( (E_b ==1'b1) && (W_b ==1'b1) && (ini_weset == 1'b1 && ini_ceset == 1'b1))
     begin 
           if ( ( ( ($time - write_W_bar_start) >=`twlwh)&& ( ( ($time - write_E_bar_start) >=`teleh) )) )
	        begin
				   Write_A2 <= Write_A1;  
				   Volatile_memory[Write_A1] <= Write_Data[15:0];
				   ini_wecebar <=1'b1;
	        end 
 
      end 

end


always@(DQ)
begin
     write_data <= $time;
     write_data1 <= write_data;

end

always@(Address)
begin
     write_address <= $time;

     if ( (write_address - write_end) >= `twhax)
     begin 
      write_address1 <= write_address;      // Time of previous address change
        
     end 
     if (WR_Trigger1 == 1'b1 && E_b == 1'b0 && W_b == 1'b0) 
     begin
         
     	// ERROR: Transition on Address lines between writes
       	$display("%d ERROR: ADDRESS CHANGING BEFORE WRITE COMPLETION @ ", $time);
     	WR_Trigger1 <= 1'b0;
     end
end


always@(ini_cebar or ini_webar or ini_wecebar) 
begin
     if ( (ini_cebar == 1'b1) || (ini_webar == 1'b1) || (ini_wecebar == 1'b1) ) 
     begin
       
         if (  (($time - write_data) >= `tdvwh || ($time == write_data && $time - write_data1 > `tdvwh))   && 
            (($time - write_address) >= `tavav || ($time == write_address && $time - write_address1 >= `tavav))  )    
         begin
             WR_Trigger1 <= 1'b1;
         end

     end

end 

always@( WR_Trigger1 )   
begin
    if ( ( ($time - write_W_bar_start) >=`twlwh) && ( ($time - write_E_bar_start) >=`teleh) )
	           
    begin	       
       MRAM_Array[Write_A2] <= Volatile_memory[Write_A2];   		         		
    end
end    


/**********************************************************************************************

Read Routines

***********************************************************************************************/
      
always@(posedge E_b)
begin
    RD_Trigger1 <= 1'b0;
    RD_Trigger2 <= 1'b0;
    read_address <= 0;
end

always@(Address)
begin 
     read_address <=$time;
     Read_A1 <= Address;
     Read_A2 <= Address;
     #`taxqx Read_Data <= 16'bz;
      if ( (E_b == 1'b0) && (W_b == 1'b1) )	// Address controlled Read
		   RD_Trigger1 <= 1'b1;
      else
		   RD_Trigger1 <= 1'b0;
end

always #1
begin 
	if ( ($time - read_address) >= `tavav)
	begin
		if ( (E_b == 1'b0) && (W_b == 1'b1) && ( ($time - read_W_bar_start) >=`tavav) && ( ($time -read_E_bar_start) >=`tavav) )
		begin
			Read_A2 <= Read_A1;
			RD_Trigger2 <= 1'b1;
			#50 RD_Trigger2 <= 1'b0;
		end
	end
end

	


always@(RD_Trigger2)
begin
	if ( (RD_Trigger1 == 1'b1) || (RD_Trigger2 == 1'b1) )
	begin
		if ( (E_b == 1'b0) && (W_b ==1'b1) )
		begin
			if ( ( ($time - read_W_bar_start) >=`tavav) && ( ($time -read_E_bar_start) >=`tavav) )
			begin
			   
			  data_temp[Bus_Width-1:0]  <= MRAM_Array[Read_A2];
				
			  if (G_b == 1'b0 )
			    begin
		              if (($time - E_Transition >= `tglqv) && (E_Transition > 0))
				begin
				  Read_Data[Bus_Width-1:0] <= MRAM_Array[Read_A2];
            			end
            		      else
              			begin
                		  #(`tglqv-$time-E_Transition) Read_Data[Bus_Width-1:0] <= MRAM_Array[Read_A2];
              			end
 		             end        
     			   else
       			    begin
               		      Read_Data[Bus_Width-1:0] <= 16'bz;
       			    end  
            	        end
	        end
	end
 
end

always @(G_b)
begin
   if (G_b == 1'b0)
      E_Transition <= $time;
   else
      E_Transition <= 0;
      if (E_b == 1'b0 && W_b == 1'b1 && G_b == 1'b0)
	begin
  	  #`tglqv Read_Data[Bus_Width-1:0]  <= data_temp[Bus_Width-1:0];
  	end	 
end

always@(posedge G_b)
begin
    #`tghqz Read_Data[Bus_Width-1:0] <= 16'bz;
end

always@(posedge E_b)
begin
    #`tehqz Read_Data[Bus_Width-1:0] <= 16'bz;
end


// endtask;


endmodule
