
-- VHDL Model for a 256K SPI MRAM
-- Device Part Number: MR25H256
-- Device Data Sheet: http://www.everspin.com/PDF/EST_MR25H256_prod.pdf

-- Model Revision: 1.0
-- Model Release Date: August 2010

---------------------------------------------------------------------
-- These VHDL models are provided "as is" without warranty of
-- any kind, included but not limited to, implied warranty of
-- merchant ability and fitness for a particular purpose.
---------------------------------------------------------------------

LIBRARY IEEE;
USE IEEE.STD_LOGIC_1164.ALL;
LIBRARY STD;
USE STD.TEXTIO.ALL;
LIBRARY WORK;
USE WORK.PACKAGE_UTILITY.ALL;


--===================================================================
-- Entity for MR25H256
--===================================================================
ENTITY MR25H256 IS
    GENERIC (
             tRI	:	Time	:= 50 ns;	-- Input Rise Time
             tRF	:	Time	:= 50 ns;	-- Input Fall Time
             tWH	:	Time	:= 11 ns;	-- SCK High Time
             tWL	:	Time	:= 11 ns;	-- SCK Low Time
             tCS	:	Time	:= 40 ns;	-- CS High Time
             tCSS	:	Time	:= 10 ns;	-- CS Setup Time
             tCSH	:	Time	:= 10 ns;	-- CS Hold Time
             tSU	:	Time	:= 5 ns;	-- Data In Setup Time
             tH		:	Time	:= 5 ns;	-- Data Hold Time
             tV		:	Time	:= 0 ns;	-- Output Valid Time
             tHO	:	Time	:= 0 ns;	-- Output Hold Time
             tHD	:	Time	:= 10 ns;	-- Hold Setup Time
             tCD	:	Time	:= 10 ns;	-- Hold Hold Time
             tLZ	:	Time	:= 20 ns;	-- Hold to Output Low Impedance
             tHZ	:	Time	:= 20 ns;	-- Hold to Output High Impedance
             tWPS	:	Time	:= 5 ns;	-- WP Setup to CS
             tWPH	:	Time	:= 5 ns;	-- WP Hold from CS
             tDP	:	Time	:= 3 ns;	-- Sleep Mode Entry Time
      	     tRDP	:	Time	:= 3 ns;	-- Sleep Mode Exit Time
             tDIS	:	Time	:= 12 ns;	-- Output Disable Time
     
             Datafile      : STRING;			-- Memory File to be Uploaded into the MRAM
             memoryUpdate  : BOOLEAN   := TRUE 		-- DataStore into external File on every write

    );
    
    PORT (
        SCK		: IN    std_logic;		-- Serial clock input
        SI		: IN    std_logic;		-- VHDLSerial data input
        SO		: OUT   std_logic;		-- Serial Data Out
        CS		: IN    std_logic;		-- Chip select input
        HOLD		: IN    std_logic;		-- Hold input
        WP		: IN   std_logic		-- Write protect input
    );

END MR25H256;

--============================================================================
-- Architecture for entity MR25H256
--============================================================================

ARCHITECTURE Behavioral OF MR25H256 IS
   

    CONSTANT HiAddrBit     : NATURAL := 14;
    CONSTANT BYTE          : NATURAL := 8;
    
   TYPE meminstance IS ARRAY (32767 DOWNTO 0) of std_logic_Vector(7 downto 0);
   
   TYPE instruction_type IS (        NONE, 	-- Binary Code 0000 0110
                                     WREN, 	-- Binary Code 0000 0100
                                     WRDI, 	-- Binary Code 0000 0101
                                     RDSR, 	-- Binary Code 0000 0001
                                     WRSR, 	-- Binary Code 0000 0110
                                     READ, 	-- Binary Code 0000 0011
                                     WRITE, 	-- Binary Code 0000 0010
                                     SLEEP, 	-- Binary Code 1011 1001
                                     WAKE 	-- Binary Code 1010 1011
                                  );
	SIGNAL Instruct         : instruction_type;
     
        SIGNAL SCK_th, SCK_LH	        : std_logic;
        
        SIGNAL read_out         : std_logic := '0';

        SIGNAL fast_rd          : boolean   := true;
        SIGNAL rd               : boolean   := false;

        SIGNAL frst_rd          : boolean   := false;
    
        SIGNAL Data_in		: std_logic_vector(7 downto 0);      
        SIGNAL Data_out		: std_logic_vector (7 downto 0);
    
        SIGNAL Status_reg   : std_logic_vector(7 downto 0)
                                                := (others => '0');

        SIGNAL Status_reg_in         : std_logic_vector(7 downto 0)
                                              := (others => '0');

        SIGNAL Status_reg_temp : std_logic_vector(7 downto 0)
                                              := (others => '0');

        ALIAS WIPD	:std_logic IS Status_reg(0);
        ALIAS WEL	:std_logic IS Status_reg(1);
        ALIAS BP0	:std_logic IS Status_reg(2);
        ALIAS BP1	:std_logic IS Status_reg(3);
        ALIAS BP2D	:std_logic IS Status_reg(4);
        ALIAS PE_ERRD	:std_logic IS Status_reg(6);
        ALIAS SRWD	:std_logic IS Status_reg(7);
                                    
 ---------------------------------------------------------------------
 -- PROCEDURE READ FROM FILE
 ---------------------------------------------------------------------
 PROCEDURE read_from_file ( file_name : STRING; memory : out meminstance) IS
 
 file data_file : text open read_mode is file_name;
 
 VARIABLE L : line;
 VARIABLE LSB, MSB : STD_LOGIC_VECTOR(3 downto 0);
 VARIABLE dr : STRING (1 to 3071);
 VARIABLE index_m,i : natural := 0;
 VARIABLE ROWSIZE: Integer := 1024;
 
 BEGIN
 
 WHILE NOT endfile(data_file) LOOP
 
 readline(data_file,L);
 READ(L,dr);
 deallocate(L);
 
 FOR i IN 1 TO ROWSIZE LOOP
 
 CASE dr(3*i - 2) IS 
 WHEN '0'=> MSB := "0000";
 WHEN '1'=> MSB := "0001";
 WHEN '2'=> MSB := "0010";
 WHEN '3'=> MSB := "0011";
 WHEN '4'=> MSB := "0100";
 WHEN '5'=> MSB := "0101";
 WHEN '6'=> MSB := "0110";
 WHEN '7'=> MSB := "0111";
 WHEN '8'=> MSB := "1000";
 WHEN '9'=> MSB := "1001";
 WHEN 'A'=> MSB := "1010";
 WHEN 'B'=> MSB := "1011";
 WHEN 'C'=> MSB := "1100";
 WHEN 'D'=> MSB := "1101";
 WHEN 'E'=> MSB := "1110";
 WHEN 'F'=> MSB := "1111";
 WHEN 'a'=> MSB := "1010";
 WHEN 'b'=> MSB := "1010";
 WHEN 'c'=> MSB := "1100";
 WHEN 'd'=> MSB := "1101";
 WHEN 'e'=> MSB := "1110";
 WHEN 'f'=> MSB := "1111";
 WHEN OTHERS => null;
 END CASE;
 
 CASE dr(3*i -1) IS 
 WHEN '0'=> LSB := "0000";
 WHEN '1'=> LSB := "0001";
 WHEN '2'=> LSB := "0010";
 WHEN '3'=> LSB := "0011";
 WHEN '4'=> LSB := "0100";
 WHEN '5'=> LSB := "0101";
 WHEN '6'=> LSB := "0110";
 WHEN '7'=> LSB := "0111";
 WHEN '8'=> LSB := "1000";
 WHEN '9'=> LSB := "1001";
 WHEN 'A'=> LSB := "1010";
 WHEN 'B'=> LSB := "1011";
 WHEN 'C'=> LSB := "1100";
 WHEN 'D'=> LSB := "1101";
 WHEN 'E'=> LSB := "1110";
 WHEN 'F'=> LSB := "1111";
 WHEN 'a'=> LSB := "1010";
 WHEN 'b'=> LSB := "1010";
 WHEN 'c'=> LSB := "1100";
 WHEN 'd'=> LSB := "1101";
 WHEN 'e'=> LSB := "1110";
 WHEN 'f'=> LSB := "1111";
 WHEN OTHERS => null;
 END CASE;
 
 memory(index_m):=(MSB(3),MSB(2),MSB(1),MSB(0),LSB(3),LSB(2),LSB(1),LSB(0));
 index_m:=index_m+1;
 END LOOP;
 END LOOP;
 
 END read_from_file;
 
 ---------------------------------------------------------------------
 -- PROCEDURE -  WRITE TO FILE
 ---------------------------------------------------------------------
 PROCEDURE write_to_file (file_name : STRING; memory : IN meminstance) IS
 
 file data_file : text open write_mode is file_name;
 
 VARIABLE L : line;
 VARIABLE LSB, MSB : std_logic_vector(3 downto 0);
 VARIABLE dr : STRING (1 to 3071);
 VARIABLE index_m: natural := 0;
 VARIABLE ROWSIZE: Integer := 1024;
 
 BEGIN
 
 WHILE (index_m < 32768) LOOP
 
 FOR i in 1 TO ROWSIZE LOOP
 FOR j IN 0 TO 3 LOOP
 LSB(j):= memory(index_m)(j);
 MSB(j):= memory(index_m)(j+4);
 END LOOP;
 index_m:=index_m+1;
 CASE MSB IS 
 WHEN "0000" => dr(3*i - 2):='0';
 WHEN "0001" => dr(3*i - 2):='1';
 WHEN "0010" => dr(3*i - 2):='2';
 WHEN "0011" => dr(3*i - 2):='3';
 WHEN "0100" => dr(3*i - 2):='4';
 WHEN "0101" => dr(3*i - 2):='5';
 WHEN "0110" => dr(3*i - 2):='6';
 WHEN "0111" => dr(3*i - 2):='7';
 WHEN "1000" => dr(3*i - 2):='8';
 WHEN "1001" => dr(3*i - 2):='9';
 WHEN "1010" => dr(3*i - 2):='A';
 WHEN "1011" => dr(3*i - 2):='B';
 WHEN "1100" => dr(3*i - 2):='C';
 WHEN "1101" => dr(3*i - 2):='D';
 WHEN "1110" => dr(3*i - 2):='E';
 WHEN "1111" => dr(3*i - 2):='F';
 WHEN OTHERS => null;
 END CASE;
 
 CASE LSB IS
 WHEN "0000" => dr(3*i - 1):='0';
 WHEN "0001" => dr(3*i - 1):='1';
 WHEN "0010" => dr(3*i - 1):='2';
 WHEN "0011" => dr(3*i - 1):='3';
 WHEN "0100" => dr(3*i - 1):='4';
 WHEN "0101" => dr(3*i - 1):='5';
 WHEN "0110" => dr(3*i - 1):='6';
 WHEN "0111" => dr(3*i - 1):='7';
 WHEN "1000" => dr(3*i - 1):='8';
 WHEN "1001" => dr(3*i - 1):='9';
 WHEN "1010" => dr(3*i - 1):='A';
 WHEN "1011" => dr(3*i - 1):='B';
 WHEN "1100" => dr(3*i - 1):='C';
 WHEN "1101" => dr(3*i - 1):='D';
 WHEN "1110" => dr(3*i - 1):='E';
 WHEN "1111" => dr(3*i - 1):='F';
 WHEN OTHERS => null;
 END CASE;
 if (i < 1024) then dr(3*i) := ' ';end if;
 END LOOP;
 WRITE(L,dr);
 writeline(data_file,L);
 END LOOP;
 
END write_to_file;


   BEGIN
   
   -----------------------------------------------------------
   -- Process for Clock Creation
   -- Create a Delayed Clock signal by tH for internal timing
   -----------------------------------------------------------
   SCK_THDELAY: PROCESS (SCK)
   BEGIN
    SCK_tH <= SCK After Th;
   END PROCESS SCK_THDELAY;
   
   SCK_LHDELAY: PROCESS (SCK)
   BEGIN
    SCK_LH <= SCK After tH + 1 ns;
   END PROCESS SCK_LHDELAY;
   
   ----------------------------------------------------------
   -- Main Control Path
   ----------------------------------------------------------
   PROCESS (CS, SCK, SCK_TH, SCK_LH, SI)
   
   VARIABLE MESSAGE : LINE;
   VARIABLE MRAM_Core: meminstance;
   VARIABLE FIRSTRUN : BOOLEAN := TRUE;
   
    TYPE bus_cycle_type IS (    STAND_BY,
                                CODE_BYTE,
                                ADDRESS_BYTES,
                                DATA_BYTES
                                );
    VARIABLE bus_cycle_state   : bus_cycle_type;
    VARIABLE previous_state: instruction_type;
         
    VARIABLE code            : std_logic_vector(7 downto 0);
    VARIABLE code_in         : std_logic_vector(7 downto 0);
    VARIABLE Byte_slv        : std_logic_vector(7 downto 0);
    VARIABLE addr_bytes      : std_logic_vector(HiAddrBit downto 0);
    VARIABLE Address_in      : std_logic_vector(15 downto 0);
    VARIABLE Address		: NATURAL;
      
    VARIABLE data_cnt		: NATURAL;
    VARIABLE addr_cnt		: NATURAL;
    VARIABLE code_cnt		: NATURAL;
    VARIABLE dummy_cnt		: NATURAL;
    VARIABLE bit_cnt		: NATURAL;
    VARIABLE Status_Cnt		: NATURAL;
    
                                   
   BEGIN
-----------------------------------------------------------------
-- Initialisation of memory array
-----------------------------------------------------------------
  	 IF (FIRSTRUN) THEN
   
  	 WRITE (message,STRING'("Trying to load"));
   	 WRITELINE(output, message);
  	 ---------------------------------------
   	 read_from_file(datafile,MRAM_Core);
   	 FIRSTRUN := FALSE;

   	 END IF;
   	 
----------------------------------------------------------------
-- Initialization Complete
----------------------------------------------------------------
 
---------------------------------------------------------------------------
-- Start Operations on falling edge of CS, Reset all states on rising edge
---------------------------------------------------------------------------

     IF rising_edge(CS) THEN
            Previous_State := Instruct;
            bus_cycle_state := STAND_BY;
            SO <= 'Z' after tDIS;
            Data_cnt := 0;
            Addr_cnt := 0;
            code_cnt := 0;
            Status_cnt := 8;
        ELSE
        CASE bus_cycle_state IS
            WHEN STAND_BY => 
                IF falling_edge(CS) THEN
                    Instruct <= NONE;
                    code_cnt := 0;
                    addr_cnt := 0;
                    data_cnt := 0;
                    Status_cnt := 8;
                    dummy_cnt := 0;
                    bus_cycle_state := CODE_BYTE;
                    
                END IF;

            WHEN CODE_BYTE =>
                IF rising_edge(SCK) AND HOLD = '1' THEN
                    IF code_cnt < 8 THEN
                        Code_in(7 - code_cnt) := SI;
                        Code_cnt := Code_cnt + 1;
                    END IF;
                END IF;
                
                IF rising_edge(SCK_TH) AND HOLD = '1' THEN
                      
                       IF (Code_cnt = 8) THEN
                            
                            IF ((Previous_state = SLEEP ) and (Code_in /= "10101011" )) THEN CODE_in := "11111111"; END IF;
                            
                            CASE code_in IS
                                WHEN "00000110" =>
                                    Instruct <= WREN;
                                    bus_cycle_state := DATA_BYTES;
                                WHEN "00000100" =>
                                    Instruct <= WRDI;
                                    bus_cycle_state := DATA_BYTES;
                                WHEN "00000101" =>
                                    Instruct <= RDSR;
                                    bus_cycle_state := DATA_BYTES;
                                WHEN "00000001" =>
                                    Instruct <= WRSR;
                                    bus_cycle_state := DATA_BYTES;
                                WHEN "00000011" =>
                                    Instruct <= READ;
                                    bus_cycle_state := ADDRESS_BYTES;
                                WHEN "00000010" =>
                                    Instruct <= WRITE;
                                    bus_cycle_state := ADDRESS_BYTES;
                               WHEN "01000101" =>
                                    Instruct <= SLEEP;
                                    bus_cycle_state := STAND_BY;
                                WHEN "10101011" =>
                                    Instruct <= WAKE;
                                    bus_cycle_state := STAND_BY;
                                WHEN others =>
                                    null;
                            END CASE;
                        END IF;  
                     END IF;
                          
             WHEN ADDRESS_BYTES =>
                
               -- Capture Address for Read or Write Cycles
               -- 16 Bits are latched in, but only 15 are used for the Reads or Writes.
               
                IF rising_edge(SCK) AND HOLD = '1' THEN
                    IF(addr_cnt < 16) THEN
                     Address_in(addr_cnt) := SI;
                     addr_cnt := addr_cnt + 1;
                     IF addr_cnt = 2*BYTE THEN
                        FOR I IN 14 DOWNTO 0 LOOP
                            addr_bytes(14-i) := Address_in(i);
                        END LOOP;
                        Address := conv_integer(addr_bytes);
                       -- change_addr <= '1','0' AFTER 1 ns;
                     END IF;
                    END IF;
                END IF;
  
  		
  		IF (Falling_edge(SCK) AND HOLD = '1' AND Instruct = WRITE AND ADDR_CNT = 16) THEN
  		    Addr_CNT := Addr_CNT + 1;
  		END IF;
  		
  		-- Complete Read using Captured Address
  		
  		IF (Instruct = Read and Addr_cnt = 16) THEN
  		   IF falling_edge(SCK) and HOLD = '1' THEN
  		      Data_out <= MRAM_CORE(Address);
  		      SO <= MRAM_CORE(Address)(Data_CNT) AFTER tV;
  		      Data_CNT := Data_CNT + 1;
                   END IF;
                   
                   IF rising_edge(SCK) and HOLD = '1' THEN
                     IF (Data_CNT = 8) THEN
                      Data_CNT := 0;
                      Address := Address + 1;
                      IF (Address = 32767) THEN
                       Address := 0;
                      END IF;
                     END IF;
                   END IF;
                END IF;
    
                -- Complete Write using captured address
                            
                IF (Instruct = Write and Addr_cnt = 17) THEN
                   IF rising_edge(SCK) and HOLD = '1' THEN
                      Data_in(Data_cnt) <= SI;
                      Data_cnt := Data_cnt + 1;
                    END IF;
                    
                     IF (Data_cnt = 8) THEN
                      IF rising_edge(SCK_TH) THEN
                        Data_cnt := 0;
                        if (BP0 = '0' AND BP1 = '0') THEN
                         MRAM_CORE(Address) := Data_in;
                         if (MemoryUpdate = TRUE) THEN Write_to_file(Datafile, MRAM_CORE); END IF;
                         Address := Address + 1;
                         IF (Address = 32767) THEN
                           Address := 0;
                         END IF;
                        END IF;
                        
                        IF (BP0 = '0' AND BP1 = '1' ) THEN
                         IF (Address >= 8191) THEN
                           MRAM_CORE(Address) := Data_in;
                           if (MemoryUpdate = TRUE) THEN Write_to_file(Datafile, MRAM_CORE); END IF;
                           Address := Address + 1;
                           IF (Address = 32767) THEN
                            Address := 0;
                           END IF;
                          END IF;
                         END IF;
                        
                        IF (BP0 = '1' AND BP1 = '0' ) THEN
                         IF (Address >= 16383) THEN
                          MRAM_CORE(Address) := Data_in;
                          if (MemoryUpdate = TRUE) THEN Write_to_file(Datafile, MRAM_CORE); END IF;
                          Address := Address + 1;
                          IF (Address = 32767) THEN
                           Address := 0;
                          END IF;
                         END IF;
                        END IF;
                        
                     END IF;
                 END IF;
                END IF;
                
            When Data_Bytes =>
              
                -- RDSR Instruction
                IF (Instruct = RDSR) THEN
                 IF falling_edge(SCK) and HOLD = '1' THEN
                   SO <= status_reg(Status_Cnt - 1);
                   Status_Cnt := Status_Cnt - 1;
                   IF (STATUS_CNT = 0) THEN STATUS_CNT := 8; 
                   END IF;
                 END IF;
                END IF;  
               
               -- WREN Instruction
               IF (Instruct = WREN) THEN
                WEL <= '1';
               END IF;
               
               -- WRDI Instruction
               IF (Instruct = WRDI) THEN
                WEL <= '0';
               END IF;
               
               -- WRSR Instruction
               IF(Instruct = WRSR) THEN
                IF rising_edge(SCK) and HOLD = '1' THEN
                 Status_reg_temp(Status_cnt - 1) <= SI;
                 Status_cnt := Status_Cnt - 1;
                END IF;
                IF ( rising_edge(SCK_TH) AND ( Status_CNT = 0 AND WEL = '1' )AND ((SRWD = '0') OR (WP = '1'))) THEN
                  Status_reg <= Status_reg_temp;
                  Status_cnt := 8;
                END IF;
               END IF;
              
              
            WHEN others =>
               null;
         END CASE;
       END IF;  
                   
                      
   END PROCESS;
   
 --------------------------------------------------------------------------------------------------------------
   -- Process for Timing Checks
   -- The following timings are checked Tss, Tsu, Th, Tcss, Tcsh, Thd, Tcd
 -------------------------------------------------------------------------------------------------------------- 
  
   Timing_Checks: Process  (SCK, CS, SI, HOLD)
   
   VARIABLE RisingTIME, FallingTIME : TIME;
   VARIABLE TestTime, TestTime2, TestNOW: TIME;
   VARIABLE MESSAGE : LINE;
   
   Begin
   
    if (SCK'EVENT and SCK = '1') THEN
    
    TestTime := CS'LAST_EVENT;
    TestTime := SI'LAST_EVENT;
    TestNOW  := NOW;
    
    ASSERT (CS'LAST_EVENT >= TCSS)
    REPORT "ERROR: CHIP SELECT SETUP TIME VIOLATION"
    SEVERITY ERROR;
    
    ASSERT (SI'LAST_EVENT >= TSU)
    REPORT "ERROR: SERIAL INPUT SETUP TIME VIOLATION"
    SEVERITY ERROR;
    
    ASSERT (HOLD'LAST_EVENT >= TCD)
    REPORT "ERROR: HOLD TIME SETUP VIOLATION"
    SEVERITY ERROR;
    
    END IF;
    
    IF (SCK'EVENT and SCK = '0') THEN
     FallingTIME := NOW;
    END IF;
    
    IF (SCK'EVENT and SCK = '1') THEN
     RisingTIME := NOW;
    END IF;
    
    IF (CS'EVENT and CS = '1') THEN
     IF (NOW - FallingTIME) <= TCSH THEN
      WRITE (message,STRING'("ERROR: CS HOLD TIME VIOLATION"));
      WRITELINE(output, message);
     END IF;
    END IF;
   
    IF (SI'EVENT and CS = '0') THEN
     IF (NOW - RisingTIME) <= TH THEN
      WRITE (message, STRING'("ERROR: SI HOLD TIME VIOLATION"));
      WRITELINE (output, message);
     END IF;
    END IF;
    
    IF (HOLD'EVENT) THEN
     IF (NOW - FallingTIME) <= THD THEN
      WRITE (message, STRING' ("ERROR: HOLD TIME VIOLATION"));
      WRITELINE (Output, message);
     END IF;
    END IF;
    
   END PROCESS Timing_Checks;
   
  
   
   END Behavioral;




