-- VHDL Model for a 256K x 8 MRAM
-- Device Part Number: MR256A08B
-- Device Data Sheet: http://www.everspin.com/PDF/EST_MR256A08B_prod.pdf

-- Model Revision: 1.0
-- Model Release Date: July 2010

---------------------------------------------------------------------
-- These VHDL models are provided "as is" without warranty of
-- any kind, included but not limited to, implied warranty of
-- merchant ability and fitness for a particular purpose.
---------------------------------------------------------------------

LIBRARY IEEE;
USE IEEE.STD_LOGIC_1164.ALL;
LIBRARY STD;
USE STD.TEXTIO.ALL;
LIBRARY WORK;
USE WORK.PACKAGE_UTILITY.ALL;

--===================================================================
-- Entity for MR256A08B
--===================================================================

 Entity MR256A08B IS
	   GENERIC (Tavav:   TIME    :=   35 ns;
	            Tavqv:   TIME    :=   35 ns;
	            Taxqx:   TIME    :=   3 ns;
	            Telqv:   TIME    :=   35 ns;
	            Tqlqv:   TIME    :=   15 ns;
	            Tqhqz:   TIME    :=   15 ns;
	            Tehqz:   TIME    :=   15 ns;
	            Teleh:   TIME    :=   0 ns;
	            Tavwh:   TIME    :=   18 ns;
	            Twhax:   TIME    :=   12 ns;
	            Tavwl:   TIME    :=   0 ns;
	            Twlwh:   TIME    :=   15 ns;
	            Tdvwh:   TIME    :=   10 ns;
	            Twhdx:   TIME    :=   0 ns;
	            Datafile: STRING;
	            MemoryUpdate: Boolean := TRUE);

    PORT ( E_b, 
           W_b, 
           G_b : IN Std_Logic;
           A   : IN Std_Logic_Vector(14 downto 0);
           DQ  : INOUT Std_Logic_Vector(7 downto 0):=(others=>'Z'));

End MR256A08B;

-- End Entity Description

--============================================================================
-- Architecture for entity MR256A08B
--============================================================================
Architecture Behavioral Of MR256A08B Is

   Type meminstance Is array (32767 downto 0) of std_logic_vector(7 downto 0);
   Signal rd, wr, oe, ce, we, ce_pipe, we_pipe, r_chk : Std_logic := '0';
   Signal we_chk, ce_chk, wr_sa, wr_sa1 : Std_logic;
   Signal address, A_adr, prev_addr, addr: Std_Logic_Vector(14 downto 0);
   Signal io_reg, io_val : Std_Logic_Vector(7 downto 0);
   
 ---------------------------------------------------------------------
  -- PROCEDURES READ FROM FILE
  ---------------------------------------------------------------------
  PROCEDURE read_from_file ( file_name : STRING; memory : out meminstance) IS
  
  file data_file : text open read_mode is file_name;
  
  VARIABLE L : line;
  VARIABLE LSB, MSB : STD_LOGIC_VECTOR(3 downto 0);
  VARIABLE dr : STRING (1 to 3071);
  VARIABLE index_m,i : natural := 0;
  VARIABLE ROWSIZE: Integer := 1024;
  
  BEGIN
  
  WHILE NOT endfile(data_file) LOOP
  
  readline(data_file,L);
  READ(L,dr);
  deallocate(L);
  
  FOR i IN 1 TO ROWSIZE LOOP
  
  CASE dr(3*i - 2) IS 
  WHEN '0'=> MSB := "0000";
  WHEN '1'=> MSB := "0001";
  WHEN '2'=> MSB := "0010";
  WHEN '3'=> MSB := "0011";
  WHEN '4'=> MSB := "0100";
  WHEN '5'=> MSB := "0101";
  WHEN '6'=> MSB := "0110";
  WHEN '7'=> MSB := "0111";
  WHEN '8'=> MSB := "1000";
  WHEN '9'=> MSB := "1001";
  WHEN 'A'=> MSB := "1010";
  WHEN 'B'=> MSB := "1011";
  WHEN 'C'=> MSB := "1100";
  WHEN 'D'=> MSB := "1101";
  WHEN 'E'=> MSB := "1110";
  WHEN 'F'=> MSB := "1111";
  WHEN 'a'=> MSB := "1010";
  WHEN 'b'=> MSB := "1010";
  WHEN 'c'=> MSB := "1100";
  WHEN 'd'=> MSB := "1101";
  WHEN 'e'=> MSB := "1110";
  WHEN 'f'=> MSB := "1111";
  WHEN OTHERS => null;
  END CASE;
  
  CASE dr(3*i -1) IS 
  WHEN '0'=> LSB := "0000";
  WHEN '1'=> LSB := "0001";
  WHEN '2'=> LSB := "0010";
  WHEN '3'=> LSB := "0011";
  WHEN '4'=> LSB := "0100";
  WHEN '5'=> LSB := "0101";
  WHEN '6'=> LSB := "0110";
  WHEN '7'=> LSB := "0111";
  WHEN '8'=> LSB := "1000";
  WHEN '9'=> LSB := "1001";
  WHEN 'A'=> LSB := "1010";
  WHEN 'B'=> LSB := "1011";
  WHEN 'C'=> LSB := "1100";
  WHEN 'D'=> LSB := "1101";
  WHEN 'E'=> LSB := "1110";
  WHEN 'F'=> LSB := "1111";
  WHEN 'a'=> LSB := "1010";
  WHEN 'b'=> LSB := "1010";
  WHEN 'c'=> LSB := "1100";
  WHEN 'd'=> LSB := "1101";
  WHEN 'e'=> LSB := "1110";
  WHEN 'f'=> LSB := "1111";
  WHEN OTHERS => null;
  END CASE;
  
  memory(index_m):=(MSB(3),MSB(2),MSB(1),MSB(0),LSB(3),LSB(2),LSB(1),LSB(0));
  index_m:=index_m+1;
  END LOOP;
  END LOOP;
  
  END read_from_file;
  
  ---------------------------------------------------------------------
  -- PROCEDURES WRITE TO FILE
  ---------------------------------------------------------------------
  PROCEDURE write_to_file (file_name : STRING; memory : IN meminstance) IS
  
  file data_file : text open write_mode is file_name;
  
  VARIABLE L : line;
  VARIABLE LSB, MSB : std_logic_vector(3 downto 0);
  VARIABLE dr : STRING (1 to 3071);
  VARIABLE index_m: natural := 0;
  VARIABLE ROWSIZE: Integer := 1024;
  
  BEGIN
  
  WHILE (index_m < 32768) LOOP
  
  FOR i in 1 TO ROWSIZE LOOP
  FOR j IN 0 TO 3 LOOP
  LSB(j):= memory(index_m)(j);
  MSB(j):= memory(index_m)(j+4);
  END LOOP;
  index_m:=index_m+1;
  CASE MSB IS 
  WHEN "0000" => dr(3*i - 2):='0';
  WHEN "0001" => dr(3*i - 2):='1';
  WHEN "0010" => dr(3*i - 2):='2';
  WHEN "0011" => dr(3*i - 2):='3';
  WHEN "0100" => dr(3*i - 2):='4';
  WHEN "0101" => dr(3*i - 2):='5';
  WHEN "0110" => dr(3*i - 2):='6';
  WHEN "0111" => dr(3*i - 2):='7';
  WHEN "1000" => dr(3*i - 2):='8';
  WHEN "1001" => dr(3*i - 2):='9';
  WHEN "1010" => dr(3*i - 2):='A';
  WHEN "1011" => dr(3*i - 2):='B';
  WHEN "1100" => dr(3*i - 2):='C';
  WHEN "1101" => dr(3*i - 2):='D';
  WHEN "1110" => dr(3*i - 2):='E';
  WHEN "1111" => dr(3*i - 2):='F';
  WHEN OTHERS => null;
  END CASE;
  
  CASE LSB IS
  WHEN "0000" => dr(3*i - 1):='0';
  WHEN "0001" => dr(3*i - 1):='1';
  WHEN "0010" => dr(3*i - 1):='2';
  WHEN "0011" => dr(3*i - 1):='3';
  WHEN "0100" => dr(3*i - 1):='4';
  WHEN "0101" => dr(3*i - 1):='5';
  WHEN "0110" => dr(3*i - 1):='6';
  WHEN "0111" => dr(3*i - 1):='7';
  WHEN "1000" => dr(3*i - 1):='8';
  WHEN "1001" => dr(3*i - 1):='9';
  WHEN "1010" => dr(3*i - 1):='A';
  WHEN "1011" => dr(3*i - 1):='B';
  WHEN "1100" => dr(3*i - 1):='C';
  WHEN "1101" => dr(3*i - 1):='D';
  WHEN "1110" => dr(3*i - 1):='E';
  WHEN "1111" => dr(3*i - 1):='F';
  WHEN OTHERS => null;
  END CASE;
  if (i < 1024) then dr(3*i) := ' ';end if;
  END LOOP;
  
  WRITE(L,dr);
  writeline(data_file,L);
  END LOOP;
  
END write_to_file;


Begin

-- Signal Assignments from the outside port 
-- RD - Read Signal based on CE/OE and NOT WE
-- WR - Write Signal based on CE and WE

  CE <=  E_b;
  OE <=  G_b;
  WE <=  W_b;
  WR <= (NOT E_b) and (NOT W_b);
  RD <= (NOT E_b) and (NOT G_b) and W_b;
  io_reg <= DQ;
  address(14 downto 0) <= A(14 downto 0);

--#######################################################################################
-- Process Description for the write and read cycle 
--#######################################################################################
 
 PROCESS (CE, OE, WR, RD, IO_REG, Address)
   VARIABLE MRAM_Core: meminstance;
   VARIABLE Troe, Trce, Tro, Trc, Thdrd, Tprev, Tiopr, Tsa1, z : Time := 0 ns;
   VARIABLE wrt, ce_end, wr_end, oe_end, io_end, add_end, A_event_rd : Std_logic ;
   VARIABLE memoryaddress2, writeinteger : Integer;
   VARIABLE first_run : boolean := true;
   VARIABLE MESSAGE : LINE;
   
   begin

 ---------------------------------------
-- initialisation of memory array
---------------------------------------
IF (first_run) THEN
WRITE (message,STRING'("Trying to load"));
--WRITE (message, INIT_FILE);
writeline(output, message);
---------------------------------------
  read_from_file(datafile,MRAM_Core);
  first_run := false;
END IF;
--- Assign signals and variables for time checks before reading or writing.

   ce_pipe <= CE;
   wr_sa <= WR;

   
-- Assignment to Determine Cycle start or End
   if (CE'event and CE'last_value = '1') then
    ce_end := '1';
   else 
    ce_end := '0';
   end if;

-- Assignment to Determine Write Cycle start or End

   if (WR'event and WR'last_value = '1') then
    wr_end := '1';
   else 
    wr_end := '0';
   end if;

-- Assignment to Determine change in IO Pins to determine Setup/Hold timings
   if (io_reg(7 downto 0)'event) then
    io_end := '1';
   else 
    io_end := '0';
   end if;

-- Assignment to Determine change in Address to determine Setup/Hold timings
   if (address'event) then
   Add_end := '1';
   else 
   Add_end := '0';
   end if;

-- Assignment for Output Enable Changes
   if (oe'event and oe'last_value = '1') then
    oe_end := '1';
   else 
    oe_end := '0';
   end if;

------ Storing the last event and previous address when address transitions 
------ along with write end.

   if (Address'event) then
   prev_addr(14 downto 0)  <= address(14 downto 0);
   Tprev := address'last_event;
   end if;
    

------ Storing the address setup to write start time for performing the check 
------ before the write.

   if (wr_end = '1' and Add_end = '1') then
   Tsa1 := Tprev - wr_sa'last_event;
   elsif (wr_end = '1' and Add_end = '0') then
   Tsa1 := Address'last_event -  wr_sa'last_event;
   end if;

----- Reading or writing occurs only when CE low or CE has transitioned 
----- from low to high.

   if ((CE = '0') or (ce_end = '1' and wr_end = '1') or (ce_end = '1')) then
 
      we_pipe <= we;

      wrt := '0';
      if  (wr'event and wr'last_value = '1') then 
      wrt := '1';
      else 
      wrt:= '0';
      end if;

------------------------------------------------------------------------------------------------
----- WRITE CYCLE

----- Storing the previous value of higher order data bits and storing last event
----- of data for data setup time check if data changes along with the write end transition.

      if (io_reg'event) then
      io_val <= io_reg;
      Tiopr := io_reg'last_event;
      end if;

    --  if (DQ'event) then
    --  Tiopr := io_reg'last_event;
    --  end if;

----- Write the higher order byte after checking for the necessary
----- timings - Twhax, Tavwl, Tavwh, Tbw, Taw, Tdvwh. 
    
       if (wrt = '1') then
         if (we_pipe'last_event >= Tavwl) and (ce_pipe'last_event >= Tavwh) and (Tsa1 >= Twhax) then
           if (Add_end = '1') then
             if (Tprev >= Tavwh) then
               if (io_end = '1') then
                 if (Tiopr > Tdvwh) then
                  MRAM_Core(conv_integer(prev_addr)) := io_val;
                  if (MemoryUpdate = TRUE) then write_to_file(datafile,MRAM_Core); end if;
                 end if;
               else
                 if (DQ'last_event >= Tdvwh) then
                 MRAM_Core(conv_integer(prev_addr)) := io_reg;
                 if (MemoryUpdate = TRUE) then write_to_file(datafile,MRAM_Core); end if;
                 end if;
               end if;
             end if;
           else
              if (A'last_event >= Tavwh) then
                if (io_end = '1') then
                  if (Tiopr >= Tdvwh) then
                   MRAM_Core(conv_integer(Address)) := DQ;
                   if (MemoryUpdate = TRUE) then write_to_file(datafile,MRAM_Core); end if;
                  end if;
                else
                  if (DQ'last_event >= Tdvwh) then
                   MRAM_Core(conv_integer(A)) := DQ(7 downto 0);
                   if (MemoryUpdate = TRUE)then write_to_file(datafile,MRAM_Core); end if;
                  end if;
                end if;
              end if;
           end if;
              DQ(7 downto 0) <= "ZZZZZZZZ" AFTER Twhdx;
        end if;
      end if;                               ----- End of write.



----------------------------------------------------------------------------------------------------
----- READ CYCLE.

    if (Address'event) then
     A_event_rd := '1';
    else
     A_event_rd := '0';
    end if;

------ Determine the read initiation to data valid time.
      if (rd'event and rd = '1') then
        Tro := OE'last_event;
        Troe := Telqv-Tro;
        if (Troe < 0 ns) then
         Troe := 0 ns;
        end if;

       Trc := 0 ns;
       Trc := CE'last_event;
       Trce := Telqv-Trc;
       If (Trce < 0 ns) then
        Trce := 0 ns;
       end if;

       if (Troe > Trce) then
        if (rd = '1') then
        z := Troe;
        end if;
       else
        if (rd = '1') then
        z := Trce;
        end if;
       end if;

       if (address'last_event+z) < Tavqv then
       z := Tavqv;
       end if;
      end if;

  end if;

	
      --if (rd = '1') then
     -- DQ (7 downto 0) <= mem_Array1(Conv_integer(A)) after z;
     -- end if;
      
      if (A_event_rd = '0') then
         if (rd'event and rd = '1') then
           DQ(7 downto 0) <= MRAM_Core(conv_integer(A)) after z;
         end if;
      elsif (A_event_rd = '1') then
        if (rd'event and rd = '1') then
          DQ(7 downto 0) <= MRAM_Core(conv_integer(A)) after z;
         elsif (rd = '1') then
         memoryaddress2 := conv_integer(A);
          DQ(7 downto 0) <= MRAM_Core(conv_integer(A)) after Tavqv;
         end if;
      end if;                                                   ------- End of Read


------ Determine read end to High Z time.

------ Higher order bits read. The time is calculated as per which signal(s) terrminates
------ the read.

       if (rd'event and rd = '0') then

          if (oe = '0' and ce = '0') then
             Thdrd := Tqhqz;
             if (Thdrd < Tehqz) then
              Thdrd := Tehqz;
             end if; 
          else
             if (ce = '0') then
              Thdrd := Tehqz;
             end if;

             if (oe  = '0') then
              Thdrd := Tqhqz;
             end if;
          end if;

          if (Thdrd < 0 ns) then
           Thdrd := 0 ns;
          end if;

         DQ(7 downto 0) <= (others=>'Z') after Thdrd; 

      end if;

 END PROCESS;

  
--#######################################################################################
-- Read Cycle (Tavav) and Write Cycle (Tehqz) time checks.
--#######################################################################################

Process (A, address, wr, rd)

variable A_evnt, A_wr, A_rd, r1 : Std_logic;
variable Trd, Tadr : Time;

 Begin

   r_chk <= rd;

   if (A'event) then
   A_adr(14 downto 0)  <= address(14 downto 0);
   end if;

   if (A'event) then
   Tadr := A_adr(14 downto 0)'last_event;
   end if;

   if (wr'event and wr'last_value = '1') then
     A_wr := '1';
   end if;

   if (rd'event and rd'last_value = '1') then
     A_rd := '1';
     Trd := r_chk'last_event;
   end if;

   if address'event then
     A_evnt := '1';
   else
     A_evnt := '0';
   end if;


      if (rd'event and rd'last_value = '1') then
        if (A_evnt = '1') then
        r1 := '1';
        ASSERT (Trd >= Tavav) or (A_adr'LAST_EVENT >= Tavav) 
        REPORT "READ CYCLE TIME VIOLATION"
        SEVERITY Error;   
        elsif (A_evnt = '0') then
        r1 := '0';
        ASSERT (Trd >= Tavav) or (A'LAST_EVENT >= Tavav) 
        REPORT "READ CYCLE TIME VIOLATION"
        SEVERITY Error;   
        end if;
      end if;

      if (A_evnt = '1') then
       if (rd'event and rd'last_value = '1') then
        r1 := '0';
        
       elsif (rd'Last_value = '1' and rd = '1' )then
        ASSERT (A_adr'LAST_EVENT >= Tavav)
        REPORT "READ CYCLE TIME VIOLATION - ADDRESS SWITCH"
        SEVERITY Error; 
        
       elsif (rd'event and rd'last_value = '1') then
        ASSERT (Trd >= Tavav)
        REPORT "READ CYCLE TIME VIOLATION - SHORTENED TERMINATION"
        SEVERITY Error;  
         
      end if;

      if (wr'event and wr'last_value = '1') then
        ASSERT (Tadr >= Tehqz)
        REPORT "WRITE CYCLE TIME VIOLATION"
        SEVERITY Error;   
      elsif A_wr = '1' then
        ASSERT (Tadr >= Tehqz)
        REPORT "WRITE CYCLE TIME VIOLATION"
        SEVERITY Error;   
      end if;

   A_wr := '0';
   A_rd := '0';
   end if;

 End Process;


--#######################################################################################
-- Checks for Tavav, Tavwh, Twhax, Tavwl, Tdvwh and Tbw.
--####################################################################################### 

 Process (ce, wr, W_b, A, DQ)

 VARIABLE wr_end_chk, A_end : Std_logic := '0';
 VARIABLE Taw_chk, Tsd_chk, Tsa_chk, TempAdd3, TempAdd4 : Time;

 begin

   ce_chk <= ce;
   we_chk <= we;
   addr(14 downto 0) <= address(14 downto 0);
   wr_sa1 <= wr;

   wr_end_chk := '0';
   if (wr'event and wr'last_value = '1') then
    wr_end_chk := '1';
   else 
    wr_end_chk := '0';
   end if;

   A_end := '0';
   if (address'event) and (wr_end_chk = '1') then
   A_end := '1';
   else 
    A_end := '0';
   end if;

   if (A'event) then
    Taw_chk := address'last_event;
   end if; 

   if (DQ'event) then
    Tsd_chk := io_reg'last_event;
   end if; 

   if (wr_end_chk = '1' and A_end = '1') then
   tempadd3 := wr_sa1'last_event;
   Tsa_chk := Taw_chk - wr_sa1'last_event;
   elsif (wr_end_chk = '1' and A_end = '0') then
   Tsa_chk := A'last_event -  wr_sa1'last_event;
   end if;

      if wr_end_chk = '1' then
         -- tempadd3 := ce'last_event;
         -- ASSERT (ce'LAST_EVENT >= Tavav)
         -- REPORT "CE LOW TO WRITE END TIME VIOLATION"
         -- SEVERITY Error;

          if (we'event) then
          ASSERT (we_chk'LAST_EVENT >= Tavwl)
          REPORT "WE PULSE WIDTH TIME VIOLATION"
          SEVERITY Error;
          else
          ASSERT (W_b'LAST_EVENT >= Tavwl)
          REPORT "WE PULSE WIDTH TIME VIOLATION"
          SEVERITY Error;
          end if;

          if (A_end = '1') then
          ASSERT (Taw_chk >= Teleh)
          REPORT "ADDRESS SETUP TO WRITE END TIME VIOLATION"
          SEVERITY Error;
          else
          ASSERT (A'LAST_EVENT >= Teleh)
          REPORT "ADDRESS SETUP TO WRITE END TIME VIOLATION"
          SEVERITY Error;
          end if;

          if (io_reg'event) then
           ASSERT (Tsd_chk >= Tdvwh)
           REPORT "DATA SETUP TO WRITE END TIME VIOLATION"
           SEVERITY Error;
          else 
           ASSERT (DQ'LAST_EVENT >= Tdvwh)
           REPORT "DATA SETUP TO WRITE END TIME VIOLATION"
           SEVERITY Error;
          end if;

          ASSERT (Tsa_chk >= Twhax) or (Tsa_chk < 0 ns)
          REPORT "ADDRESS SETUP TO WRITE START TIME VIOLATION"
          SEVERITY Error; 
      end if;

   end Process;

--#######################################################################################
-- Address Hold (Tavwh) and Data Hold (Twhdx) time checks on write.
--####################################################################################### 

Process (wr'delayed(Twhdx), wr'delayed(Tavwh), A, DQ)

    begin

     if (wr'delayed(Twhdx) = '0' and wr'delayed(Twhdx)'last_value = '1') and (wr'delayed(Twhdx)'event) then
 
         ASSERT (DQ'LAST_EVENT = 0 ns) or (DQ'LAST_EVENT > Twhdx)
         REPORT "DATA HOLD FROM WRITE END TIME VIOLATION"
         SEVERITY Error;
 
     end if;

     if (wr'delayed(Tavwh) = '0' and wr'delayed(Tavwh)'last_value = '1') and (wr'delayed(Tavwh)'event) then

         ASSERT (A'LAST_EVENT = 0 ns) or (A'LAST_EVENT > Tavwh)
         REPORT "ADDRESS HOLD FROM WRITE END TIME VIOLATION"
         SEVERITY Error;

     end if;

   End Process;

--#######################################################################################
-- Telqv, Tdbe, Telqv, Tavqv and Taxqx time checks on read.
--####################################################################################### 

Process (DQ, A'delayed(Taxqx))
    begin

      if ((DQ'event) and (rd = '1')) then
        
          if (rd'last_event <= A'last_event) then
           ASSERT (G_b'LAST_EVENT >= Telqv)
           REPORT "OE LOW TO DATA VALID TIME VIOLATION"
           SEVERITY Error;

           ASSERT (E_b'LAST_EVENT >= Telqv)
           REPORT "CE LOW TO DATA VALID TIME VIOLATION"
           SEVERITY Error;
          end if;

          if (rd'last_event > A'last_event) then
           ASSERT (A'LAST_EVENT >= Tavqv)
           REPORT "ADDRESS TO DATA VALID TIME VIOLATION"
           SEVERITY Error;
          end if;

      end if;

     if (A'delayed(Taxqx)'event and (rd = '1')) then

      if (rd'last_event > Taxqx) then
         ASSERT (DQ'LAST_EVENT = 0 ns) or (DQ'LAST_EVENT > Taxqx)
         REPORT "DATA HOLD FROM ADDRESS CHANGE TIME VIOLATION"
         SEVERITY Error;
      end if;

    end if;

End Process;

--
------ Thzbe, Tqlqv and Thzce time checks.
--
--Process (G_b'delayed(Tqlqv), E_b'delayed(Tehqz))
--   begin
--
--   if (G_b'delayed(Tqlqv)'event and G_b'delayed'last_value = '0' and G_b'delayed(Tqlqv) = '1') then
--
--       ASSERT (DQ'LAST_EVENT = 0 ns) or (DQ'LAST_EVENT > Tqlqv)
--       REPORT "OE DISABLE TO HIGH Z TIME VIOLATION"
--       SEVERITY Error;
--
--   end if;

--   if (E_b'delayed(Tqhqz)'event and E_b'delayed'last_value = '0' and E_b'delayed(Tqhqz) = '1') then
--
--       ASSERT (DQ'LAST_EVENT = 0 ns) or (DQ'LAST_EVENT > Tqhqz)
--       REPORT "CE DISABLE TO HIGH Z TIME VIOLATION"
--       SEVERITY Error;
--
--   end if;

--End Process;               
End Behavioral;






