//-------------------------------------------------------------------------------------------------
// 
//  EMxxLX_spi_model.sv
//
//  Description: Everspin XSPI Behavioral Model
//
//-------------------------------------------------------------------------------------------------
// 
//  This confidential and proprietary software may be used only as authorized
//  by the included licensing agreement from Everspin titled 'EverspinSLA.txt'
//
//                   Copyright 2021 Everspin Technologies, Inc
//
//  The entire notice above must be reproduced and the Everspin SLA included for 
//  all authorized copies.
//
//-------------------------------------------------------------------------------------------------
//  Current Revision:           1.0
//
//  Revision History:
//  1.0     2021.07.27      ADO             - Initial release
//  1.5     2024.03.19      ADO             - Various fixes for DS
//  1.6     2024.04.17      ADO             - Fix problem detecting jesd reset vs initialization
//  1.7     2024.04.21      ADO             - Fix DCC for read status and read flag status regs
//
//------------------------------------------------------------------------------------------------- 

// Define this for the DFN package.
// Otherwise default is BGA package.
// `define EMxxLX_PKG_DFN

`timescale 1ps/1ps

module EMxxLX_spi_model #(
    parameter P_DENSITY   = 64,         // 8|16|32|64|128|256 Mb
    parameter SHADOW      = 0,          // for debug only
    parameter ADDR_WIDTH = 24
    ) (
`ifdef EMxxLX_PKG_DFN
    input   wire            VDD,        // For simulated power sequence, tie to 1 and use RESET_N for normal reset sequence
    input   wire            CS_N,
    input   wire            CK,
    inout   wire            IO0,
    inout   wire            IO1,
    inout   wire            IO2,
    inout   wire            IO3
`else
    input   wire            RESET_N,
    input   wire            VDD,        // For simulated power sequence, tie to 1 and use RESET_N for normal reset sequence
    input   wire            CS_N,
    input   wire            CK,
    inout   wire            IO7,
    inout   wire            IO6,
    inout   wire            IO1,
    inout   wire            IO0,
    inout   wire            IO5,
    inout   wire            IO2,
    inout   wire            IO3,
    inout   wire            IO4,
    output  wire            DS,
    output  wire            INT_N
`endif
    );

localparam    CMD_NOP                                           = 8'h00;
localparam    CMD_WRITE_SR                                      = 8'h01;
localparam    CMD_PROGRAM_PAGE                                  = 8'h02;
localparam    CMD_READ_0_LAT                                    = 8'h03;
localparam    CMD_WRITE_DIS                                     = 8'h04;
localparam    CMD_READ_SR                                       = 8'h05;
localparam    CMD_WRITE_ENB                                     = 8'h06;
localparam    CMD_READ_FAST                                     = 8'h0B;
localparam    CMD_READ_FAST_4_BYTE_ADDR                         = 8'h0C;
localparam    CMD_READ_FAST_DTR                                 = 8'h0D;
localparam    CMD_READ_FAST_DTR_4_BYTE_ADDR                     = 8'h0E;
localparam    CMD_PROGRAM_4_BYTE_ADDR                           = 8'h12;
localparam    CMD_READ_4_BYTE_ADDR                              = 8'h13;
localparam    CMD_ERASE_4K                                      = 8'h20;
localparam    CMD_ERASE_4K_4_BYTE                               = 8'h21;
localparam    CMD_PROGRAM_FAST_QUAD_INPUT                       = 8'h32;
localparam    CMD_PROGRAM_FAST_QUAD_INPUT_4_BYTE_ADDR           = 8'h34;
localparam    CMD_ENTER_QUAD_IO                                 = 8'h35;
localparam    CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT              = 8'h38;
localparam    CMD_READ_FAST_DUAL_O                              = 8'h3B;
localparam    CMD_READ_FAST_DUAL_O_4_BYTE_ADDR                  = 8'h3C;
localparam    CMD_READ_FAST_DUAL_O_DTR                          = 8'h3D;
localparam    CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT_4_BYTE_ADDR  = 8'h3E;
localparam    CMD_WRITE_OTP                                     = 8'h42;
localparam    CMD_READ_OTP                                      = 8'h4B;
localparam    CMD_CLEAR_FLAG_SR                                 = 8'h50;
localparam    CMD_ERASE_32K                                     = 8'h52;
localparam    CMD_READ_SFDP                                     = 8'h5A;
localparam    CMD_ERASE_32K_4_BYTE                              = 8'h5C;
localparam    CMD_ERASE_BULK_60                                 = 8'h60;
localparam    CMD_WRITE_EVCFG_REG                               = 8'h61;
localparam    CMD_READ_EVCFG_REG                                = 8'h65;
localparam    CMD_READ_REG                                      = 8'h65;
localparam    CMD_RESET_ENABLE                                  = 8'h66;
localparam    CMD_READ_FAST_QUAD_O                              = 8'h6B;
localparam    CMD_READ_FAST_QUAD_O_4_BYTE_ADDR                  = 8'h6C;
localparam    CMD_READ_FAST_QUAD_O_DTR                          = 8'h6D;
localparam    CMD_READ_FLAG_SR                                  = 8'h70;
localparam    CMD_WRITE_REG                                     = 8'h71;
localparam    CMD_PE_SUSPEND                                    = 8'h75;
localparam    CMD_PE_RESUME                                     = 8'h7a;
localparam    CMD_READ_FAST_OCTAL_O_4_BYTE_ADDR                 = 8'h7C;
localparam    CMD_WRITE_VCFG_REG                                = 8'h81;
localparam    CMD_PROGRAM_FAST_OCTAL_INPUT                      = 8'h82;
localparam    CMD_PROGRAM_FAST_OCTAL_INPUT_4_BYTE_ADDR          = 8'h84;
localparam    CMD_READ_VCFG_REG                                 = 8'h85;
localparam    CMD_READ_FAST_OCTAL_O                             = 8'h8B;
localparam    CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT_4_BYTE_ADDR = 8'h8E;
localparam    CMD_READ_GPRR                                     = 8'h96;
localparam    CMD_RESET_MEMORY                                  = 8'h99;
localparam    CMD_IF_ACT                                        = 8'h9B;
localparam    CMD_READ_FAST_OCTAL_O_DTR                         = 8'h9D;
localparam    CMD_READ_ID_9E                                    = 8'h9E;
localparam    CMD_READ_ID_9F                                    = 8'h9F;
localparam    CMD_PROGRAM_FAST_DUAL_INPUT                       = 8'hA2;
localparam    CMD_EXIT_DEEP_POWER_DOWN                          = 8'hAB;
localparam    CMD_READ_ID_MIO                                   = 8'hAF;
localparam    CMD_WRITE_NVCFG_REG                               = 8'hb1;
localparam    CMD_READ_NVCFG_REG                                = 8'hb5;
localparam    CMD_ENTER_4_BYTE_ADDR                             = 8'hB7;
localparam    CMD_ENTER_DEEP_POWER_DOWN                         = 8'hB9;
localparam    CMD_READ_FAST_DUAL_IO                             = 8'hBB;
localparam    CMD_READ_FAST_DUAL_IO_4_BYTE_ADDR                 = 8'hBC;
localparam    CMD_READ_FAST_DUAL_IO_DTR                         = 8'hBD;
localparam    CMD_READ_FAST_DUAL_IO_DTR_4_BYTE_ADDR             = 8'hBE;
localparam    CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT             = 8'hC2;
localparam    CMD_WRITE_DIE_SELECT                              = 8'hC4;
localparam    CMD_WRITE_EXTENDED_ADDR_REG                       = 8'hc5;
localparam    CMD_ERASE_BULK                                    = 8'hC7;
localparam    CMD_ERASE_BULK_C7                                 = 8'hC7;
localparam    CMD_READ_EXTENDED_ADDR_REG                        = 8'hc8;
localparam    CMD_READ_FAST_OCTAL_IO                            = 8'hCB;
localparam    CMD_READ_FAST_OCTAL_IO_4_BYTE_ADDR                = 8'hCC;
localparam    CMD_PROGRAM_FAST_EXTENDED_DUAL_INPUT              = 8'hD2;
localparam    CMD_ERASE_SECTOR                                  = 8'hD8;
localparam    CMD_ERASE_SECTOR_4_BYTE                           = 8'hDC;
localparam    CMD_READ_WORD_QUAD_IO_NO_DTR                      = 8'hE7;
localparam    CMD_EXIT_4_BYTE_ADDR                              = 8'hE9;
localparam    CMD_READ_FAST_QUAD_IO                             = 8'hEB;
localparam    CMD_READ_FAST_QUAD_IO_4_BYTE_ADDR                 = 8'hEC;
localparam    CMD_READ_FAST_QUAD_IO_DTR                         = 8'hED;
localparam    CMD_READ_FAST_QUAD_IO_DTR_4_BYTE_ADDR             = 8'hEE;
localparam    CMD_TDP_WRITE                                     = 8'hF0;
localparam    CMD_TDP_READ                                      = 8'hF1;
localparam    CMD_TDP_READ_DTR                                  = 8'hF2;
localparam    CMD_EXIT_QUAD_IO                                  = 8'hF5;
localparam    CMD_READ_DIE_SELECT                               = 8'hF8;
localparam    CMD_READ_FAST_OCTAL_IO_DTR                        = 8'hFD;

localparam P_TDP_DATA = { 128'hFF0F_FF00_FFCC_C3CC_C33C_CCFF_FEFF_FEEF,
                          128'hFFDF_FFDD_FFFB_FFFB_BFFF_7FFF_77F7_BDEF,
                          128'hFFF0_FFF0_0FFC_CC3C_CC33_CCCF_FEFF_FFEE,
                          128'hFFFD_FFFD_DFFF_BFFF_BBFF_F7FF_F77F_7BDE};

localparam P_NUM_NV_REG = 16;
localparam P_NUM_VOL_REG = 17;

localparam P_ADDR_4BYTE = 8'hfe;

parameter P_XIP_ENB         = 8'hFE;
parameter P_XIP_BOOT        = 8'hFC;


typedef logic [7:0] data_t;
typedef logic [ADDR_WIDTH-1:0] addr_t;
typedef enum logic [7:0] {CFG_WRAP_16=8'b1111_1100, CFG_WRAP_32=8'b1111_1101,
                          CFG_WRAP_64=8'b1111_1110, CFG_WRAP_NO=8'b1111_1111} t_cfg_wrap;

typedef enum logic [2:0] {MODE_1S=3'd000, MODE_1D=3'b100, MODE_2S=3'b001, MODE_2D=3'b101,
                          MODE_4S=3'b010, MODE_4D=3'b110, MODE_8S=3'b011, MODE_8D=3'b111} t_mode;

typedef enum logic [7:0] {DEC_USPI_DS=8'hFF,        DEC_USPI_NO_DS=8'hDF,     DEC_DSPI_DS=8'hFD,
                          DEC_DSPI_NO_DS=8'hDD,     DEC_QSPI_DS=8'hFB,        DEC_QSPI_NO_DS=8'hDB,
                          DEC_QSPI_DS_DTR=8'hEB,    DEC_QSPI_NO_DS_DTR=8'hCB, DEC_OSPI_DS_DTR=8'hE7,
                          DEC_OSPI_NO_DS_DTR=8'hC7, DEC_OSPI_DS=8'hB7,        DEC_OSPI_NO_DS=8'h97} t_spi_mode_dec;

typedef struct packed {
    logic [3:0]     cmd_datawidth;
    logic [3:0]     addr_datawidth;
    logic           addr_dtr;
    logic [2:0]     addr_bytes;
    logic           erase_val;
    logic           rpe;
    logic           pmm;
    logic [7:0]     wrap;
    logic [5:0]     dcc;
    logic           xip_en;
    logic           xip_boot;
    logic [31:0]    bp_addr_min;
    logic [31:0]    bp_addr_max;
    logic           powerdown;
    logic           soft_reset_en;
} t_mode_cfg;

typedef struct packed {
    logic [7:0]  cmd;
    logic [31:0] addr;
    logic [5:0]  dcc;
    logic [3:0]  addr_datawidth;
    logic [3:0]  data_datawidth;
    logic [2:0]  addr_bytes;
    logic        isread;
    logic        iswrite;
    logic        dtr;
} t_cmd_packet;

typedef struct packed {
    logic CS_N;
    logic CK;
    logic IO0;
} t_jesd_rst;

data_t       mem[addr_t];
logic [7:0]  read_id_reg[32];
logic [7:0]  cfg_tdp_reg[64];
logic [7:0]  cfg_otp_reg[256+1];
logic [7:0]  cfg_gprr_reg[8];
logic [7:0]  cfg_if_cmd_reg[20];
logic [7:0]  cfg_otp_locked;
logic [7:0]  cfg_reg_nv [P_NUM_NV_REG];
logic [7:0]  cfg_reg_vol [P_NUM_VOL_REG];
logic        cfg_reg_vol_dfim;
logic [7:0]  cfg_reg_op  [P_NUM_VOL_REG]; // "operating" config regs, for jesd reset cases
logic [7:0]  status_reg;
logic [7:0]  flag_status_reg;
t_mode_cfg   mode_cfg;
t_cmd_packet pkt;
logic [31:0] cmd_bit_cnt;
logic        cmd_valid;
logic        cmd_valid_q1;
logic [31:0] addr_bit_cnt;
logic        addr_valid;
logic [7:0]  wr_data;
logic [7:0]  rd_data;
logic [31:0] data_bit_cnt;

logic [7:0]  nxt_cmd;
logic [31:0] nxt_cmd_bit_cnt;
logic        nxt_cmd_valid;
logic [31:0] nxt_addr;
logic [31:0] nxt_addr_bit_cnt;
logic        nxt_addr_valid;
logic [7:0]  nxt_wr_data;
logic [7:0]  nxt_rd_data;
logic [31:0] nxt_data_bit_cnt;
logic        nxt_xip_confirm;
logic        xip_confirm;

logic        wren;
logic        soft_reset;
t_jesd_rst [9:0] jesd_rst_seq;
logic jesd_rst_seq_detected;

logic        int_rst_n;
logic        mux_rst_n;

logic [7:0] nxt_IO_oe;
logic [7:0] nxt_IO_o;
logic [7:0] IO_oe;
logic [7:0] IO_o;
logic       DS_oe;
logic       DS_o;
logic       nxt_DS_oe;
logic       nxt_DS_o;
logic       INT_N_oe;
logic       INT_N_o;

// Make a nice reset pulse for T0 initialization
initial begin
    int_rst_n = 0;
    #1;
    int_rst_n = 1;
end
assign rst_n = int_rst_n & (~mode_cfg.rpe | RESET_N) & VDD;

assign cfg_otp_locked = cfg_otp_reg[256][0] == 0;

logic debug_xip;

always @(*) begin
    nxt_cmd = pkt.cmd;
    nxt_cmd_bit_cnt = cmd_bit_cnt;
    nxt_addr = pkt.addr;
    nxt_addr_bit_cnt = addr_bit_cnt;
    nxt_wr_data = wr_data;
    //nxt_rd_data = rd_data;
    nxt_data_bit_cnt = data_bit_cnt;
    nxt_xip_confirm = xip_confirm;
    if (CS_N) begin
        nxt_cmd_bit_cnt = 0;
        nxt_cmd = 0;
        nxt_cmd_valid = 0;
        nxt_addr_bit_cnt = 0;
        nxt_addr = 0;
        nxt_addr_valid = 0;
        nxt_wr_data = 0;
        nxt_data_bit_cnt = 0;
        debug_xip = 0;
        //nxt_xip_confirm = xip_confirm;  do not reset on CS deassert
    end else begin
        if (!cmd_valid && xip_confirm) begin
            nxt_cmd = CMD_READ_FAST;
            nxt_cmd_valid = 1;
            debug_xip = 1;
        end

        if (!cmd_valid && !xip_confirm) begin
            case (mode_cfg.cmd_datawidth)
                1: nxt_cmd = {pkt.cmd, IO0};
                2: nxt_cmd = {pkt.cmd, IO1, IO0};
                4: nxt_cmd = {pkt.cmd, IO3, IO2, IO1, IO0};
                8: nxt_cmd = {IO7, IO6, IO5, IO4, IO3, IO2, IO1, IO0};
            endcase
            nxt_cmd_bit_cnt = cmd_bit_cnt + mode_cfg.cmd_datawidth;
            nxt_cmd_valid = cmd_bit_cnt + mode_cfg.cmd_datawidth == 8;
        end else if (!addr_valid && (pkt.addr_bytes > 0)) begin
            case(pkt.addr_datawidth)
                1: nxt_addr = {pkt.addr, IO0};
                2: nxt_addr = {pkt.addr, IO1, IO0};
                4: nxt_addr = {pkt.addr, IO3, IO2, IO1, IO0};
                8: nxt_addr = {pkt.addr, IO7, IO6, IO5, IO4, IO3, IO2, IO1, IO0};
            endcase
            nxt_addr_bit_cnt = addr_bit_cnt + pkt.addr_datawidth;
            nxt_addr_valid = addr_bit_cnt + pkt.addr_datawidth == pkt.addr_bytes*8;
        end else begin
            if (data_bit_cnt == 0) begin
               if ((mode_cfg.xip_boot || mode_cfg.xip_en) && get_cmd_isxip(pkt.cmd)) begin
                   nxt_xip_confirm = !IO0;
               end else begin
                   nxt_xip_confirm = 0;
               end
            end
            case(pkt.data_datawidth)
                1: nxt_wr_data = {wr_data, IO0};
                2: nxt_wr_data = {wr_data, IO1, IO0};
                4: nxt_wr_data = {wr_data, IO3, IO2, IO1, IO0};
                8: nxt_wr_data = {IO7, IO6, IO5, IO4, IO3, IO2, IO1, IO0};
            endcase
            nxt_data_bit_cnt = data_bit_cnt + pkt.data_datawidth;
        end
    end
end

always @(CS_N or CK or IO0 or rst_n) begin
    if (!rst_n) begin
        for (int i=9; i>=0; i=i-1) begin
            jesd_rst_seq[i] = 0;
        end
    end else begin
        for (int i=9; i>=1; i=i-1) begin
            jesd_rst_seq[i] = jesd_rst_seq[i-1];
        end
        jesd_rst_seq[0].CS_N = CS_N;
        jesd_rst_seq[0].CK = CK;
        jesd_rst_seq[0].IO0 = IO0;
        jesd_rst_seq_detected = jesd_rst_seq_detected | get_jesd_rst_seq_detected();
    end
end

function get_jesd_rst_seq_detected;
begin
    get_jesd_rst_seq_detected = 1;
    //                                                                            CS_N      IO0
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[9] === {1'b1, CK, 1'b0});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[8] === {1'b1, CK, 1'b1});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[7] === {1'b0, CK, 1'b1});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[6] === {1'b1, CK, 1'b1});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[5] === {1'b1, CK, 1'b0});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[4] === {1'b0, CK, 1'b0});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[3] === {1'b1, CK, 1'b0});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[2] === {1'b1, CK, 1'b1});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[1] === {1'b0, CK, 1'b1});
    get_jesd_rst_seq_detected = get_jesd_rst_seq_detected && (jesd_rst_seq[0] === {1'b1, CK, 1'b1});
    return get_jesd_rst_seq_detected;
end
endfunction

task update_mode_cfg;
begin
    case (cfg_reg_op[0])
        DEC_USPI_NO_DS:     begin mode_cfg.cmd_datawidth = 1; mode_cfg.addr_datawidth = 1; mode_cfg.addr_dtr = 0; end
        DEC_DSPI_DS:        begin mode_cfg.cmd_datawidth = 2; mode_cfg.addr_datawidth = 2; mode_cfg.addr_dtr = 0; end
        DEC_DSPI_NO_DS:     begin mode_cfg.cmd_datawidth = 2; mode_cfg.addr_datawidth = 2; mode_cfg.addr_dtr = 0; end
        DEC_QSPI_DS:        begin mode_cfg.cmd_datawidth = 4; mode_cfg.addr_datawidth = 4; mode_cfg.addr_dtr = 0; end
        DEC_QSPI_NO_DS:     begin mode_cfg.cmd_datawidth = 4; mode_cfg.addr_datawidth = 4; mode_cfg.addr_dtr = 0; end
        DEC_QSPI_DS_DTR:    begin mode_cfg.cmd_datawidth = 4; mode_cfg.addr_datawidth = 4; mode_cfg.addr_dtr = 1; end
        DEC_QSPI_NO_DS_DTR: begin mode_cfg.cmd_datawidth = 4; mode_cfg.addr_datawidth = 4; mode_cfg.addr_dtr = 1; end
        DEC_OSPI_DS_DTR:    begin mode_cfg.cmd_datawidth = 8; mode_cfg.addr_datawidth = 8; mode_cfg.addr_dtr = 1; end
        DEC_OSPI_NO_DS_DTR: begin mode_cfg.cmd_datawidth = 8; mode_cfg.addr_datawidth = 8; mode_cfg.addr_dtr = 1; end
        DEC_OSPI_DS:        begin mode_cfg.cmd_datawidth = 8; mode_cfg.addr_datawidth = 8; mode_cfg.addr_dtr = 0; end
        DEC_OSPI_NO_DS:     begin mode_cfg.cmd_datawidth = 8; mode_cfg.addr_datawidth = 8; mode_cfg.addr_dtr = 0; end
        default:            begin mode_cfg.cmd_datawidth = 1; mode_cfg.addr_datawidth = 1; mode_cfg.addr_dtr = 0; end
        //DEC_USPI_DS: (see default)
    endcase
    mode_cfg.addr_bytes = ((mode_cfg.addr_datawidth==8 && mode_cfg.addr_dtr==1) || (cfg_reg_op[5] == P_ADDR_4BYTE)) ? 4 : 3;
    mode_cfg.xip_en = (cfg_reg_op[6] == P_XIP_ENB);
    mode_cfg.xip_boot = (rst_n==0 || soft_reset) ? (cfg_reg_op[6] == P_XIP_BOOT) : 0;
    mode_cfg.erase_val = cfg_reg_op[8][7];
    mode_cfg.pmm = cfg_reg_op[8][0];
    mode_cfg.rpe = cfg_reg_op[8][1];
    case (cfg_reg_op[7])
        CFG_WRAP_16: mode_cfg.wrap = 16;
        CFG_WRAP_32: mode_cfg.wrap = 32;
        CFG_WRAP_64: mode_cfg.wrap = 64;
        default: mode_cfg.wrap = 0;
    endcase
    if (cfg_reg_op[1] >= 1 && cfg_reg_op[1] <= 31) mode_cfg.dcc = cfg_reg_op[1];
    else mode_cfg.dcc = 16;
    case({status_reg[5], status_reg[6], status_reg[4:2]})
       // start block protect from top
       5'b0_0000: begin mode_cfg.bp_addr_min = '1; mode_cfg.bp_addr_max = 0; end // none, set min>max
       5'b0_0001: begin mode_cfg.bp_addr_min = 'h7f_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_0010: begin mode_cfg.bp_addr_min = 'h7e_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_0011: begin mode_cfg.bp_addr_min = 'h7d_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_0100: begin mode_cfg.bp_addr_min = 'h7c_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_0101: begin mode_cfg.bp_addr_min = 'h7b_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_0110: begin mode_cfg.bp_addr_min = 'h7a_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_0111: begin mode_cfg.bp_addr_min = 'h79_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_1000: begin mode_cfg.bp_addr_min = 'h78_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_1001: begin mode_cfg.bp_addr_min = 'h70_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_1010: begin mode_cfg.bp_addr_min = 'h60_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_1011: begin mode_cfg.bp_addr_min = 'h40_0000; mode_cfg.bp_addr_max = '1; end
       5'b0_1100: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       5'b0_1101: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       5'b0_1110: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       5'b0_1111: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       // start block protect from bottom
       5'b1_0000: begin mode_cfg.bp_addr_min = '1; mode_cfg.bp_addr_max = 0; end // none, set min>max
       5'b1_0001: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h00_ffff; end
       5'b1_0010: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h01_ffff; end
       5'b1_0011: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h02_ffff; end
       5'b1_0100: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h03_ffff; end
       5'b1_0101: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h04_ffff; end
       5'b1_0110: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h05_ffff; end
       5'b1_0111: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h06_ffff; end
       5'b1_1000: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h07_ffff; end
       5'b1_1001: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h0f_ffff; end
       5'b1_1010: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h1f_ffff; end
       5'b1_1011: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = 'h3f_ffff; end
       5'b1_1100: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       5'b1_1101: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       5'b1_1110: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
       5'b1_1111: begin mode_cfg.bp_addr_min = 0; mode_cfg.bp_addr_max = '1; end
    endcase
    flag_status_reg[0] = cfg_reg_op[5] == P_ADDR_4BYTE;
    if (cfg_reg_vol[15] & cfg_reg_vol[16]) begin
        INT_N_oe = 1;
        INT_N_o  = 0;
    end else begin
        INT_N_oe  = 'b0;
        INT_N_o   = 'bz;
    end
end
endtask

logic [31:0] addr_page;
logic [31:0] addr_page_byte;
logic [31:0] addr_page_full;
logic [31:0] debug_addr;
logic  [7:0] debug_data;
logic        debug_dtr;
always @(CK or posedge CS_N or negedge rst_n or posedge soft_reset or posedge jesd_rst_seq_detected) begin
    debug_addr = 0;
    debug_data = 0;
    debug_dtr = 0;
    if (!rst_n || jesd_rst_seq_detected || soft_reset) begin
        mode_cfg.powerdown = 0;
    end
    if (!int_rst_n || jesd_rst_seq_detected) begin
        for (int i=0; i<P_NUM_NV_REG; i=i+1) begin
            // TODO: Had to make this blocking...
            // only reset on initial reset, not pin resets
            cfg_reg_op[i] = '1;
            if (!int_rst_n) begin
                cfg_reg_nv[i] = '1;
                cfg_reg_vol[i] = (i<=8) ? '1 : (i<=14) ? 0 : 4;
            end
        end
        cfg_reg_vol[16] = 0;
        cfg_reg_vol_dfim = 0;
        mode_cfg.soft_reset_en = 0;
        update_mode_cfg();
        if (!int_rst_n) begin
            status_reg = 8'hfc;
            flag_status_reg = 8'h80;
            read_id_reg[0] = 'h6b; // jedec
            read_id_reg[1] = 'hbb; // 1.8v
            read_id_reg[2] = 'h17; // 64Mb
            read_id_reg[3] = 'h10; // num remaining bytes
            read_id_reg[4] = 'h01; // fake uid
            read_id_reg[5] = 'h23;
            read_id_reg[6] = 'h45;
            read_id_reg[7] = 'h67;
            read_id_reg[8] = 'h89;
            read_id_reg[9] = 'hab;
            read_id_reg[10] = 'hcd;
            read_id_reg[11] = 'hef;
            read_id_reg[12] = 'hfe;
            read_id_reg[13] = 'hdc;
            read_id_reg[14] = 'hba;
            read_id_reg[15] = 'h98;
            read_id_reg[16] = 'h76;
            read_id_reg[17] = 'h54;
            read_id_reg[18] = 'h32;
            read_id_reg[19] = 'h10;
            read_id_reg[20] = 'hff; // filler
            read_id_reg[21] = 'hff;
            read_id_reg[22] = 'hff;
            read_id_reg[23] = 'hff;
            read_id_reg[24] = 'hff;
            read_id_reg[25] = 'hff;
            read_id_reg[26] = 'hff;
            read_id_reg[27] = 'hff;
            read_id_reg[28] = 'hff;
            read_id_reg[29] = 'hff;
            read_id_reg[30] = 'hff;
            read_id_reg[31] = 'hff;
            for (int i=0; i<64; i=i+1) begin
                cfg_tdp_reg[i] = (P_TDP_DATA >> (i*8)) & 8'hff;
            end
            for (int i=0; i<8; i=i+1) begin
                cfg_gprr_reg[i] = '0;
            end
            for (int i=0; i<256+1; i=i+1) begin
                cfg_otp_reg[i] = '1;
            end
        end
    end

    if (!rst_n || CS_N || soft_reset || jesd_rst_seq_detected) begin
        pkt.cmd <= 0;
        cmd_bit_cnt <= 0;
        cmd_valid <= 0;
        cmd_valid_q1 <= 0;
        pkt.addr <= 0;
        addr_bit_cnt <= 0;
        addr_valid <= 0;
        wr_data <= 0;
        data_bit_cnt <= 0;
        if (!rst_n || soft_reset || jesd_rst_seq_detected) begin
            for (int i=0; i<P_NUM_VOL_REG; i=i+1) begin
                // TODO: Had to make this blocking...
                if (jesd_rst_seq_detected) begin
                    cfg_reg_op[i] = '1;
                end else begin
                    cfg_reg_vol[i] = (i<=8) ? '1 : (i<=14) ? 0 : 4;
                    cfg_reg_op[i] = cfg_reg_nv[i];
                end
            end
            cfg_reg_vol[16] = 0;
            cfg_reg_vol_dfim = 0;
            mode_cfg.soft_reset_en = 0;
            update_mode_cfg();
            xip_confirm = mode_cfg.xip_boot;
            if (xip_confirm) begin
                pkt.addr_datawidth <= get_addr_datawidth(CMD_READ_FAST);
                pkt.data_datawidth <= get_data_datawidth(CMD_READ_FAST);
                pkt.addr_bytes <= get_addr_bytes(CMD_READ_FAST);
                pkt.isread <= get_cmd_isread(CMD_READ_FAST);
                pkt.iswrite <= get_cmd_iswrite(CMD_READ_FAST);
                pkt.dcc <= get_cmd_dcc(CMD_READ_FAST);
                pkt.dtr <= get_cmd_isdtr(CMD_READ_FAST);
            end
            soft_reset = 0;
            jesd_rst_seq_detected = 0;
        end
        if (CS_N) begin
            // For now just update again every time CS deasserts incase there was a config write
            update_mode_cfg();
        end
    end else if (CK || ((cmd_valid_q1 || xip_confirm) && pkt.dtr)) begin
        if (cmd_valid_q1) debug_dtr = pkt.dtr;
        pkt.cmd <= nxt_cmd;
        cmd_bit_cnt <= nxt_cmd_bit_cnt;
        cmd_valid <= nxt_cmd_valid;
        cmd_valid_q1 <= cmd_valid;
        pkt.addr <= nxt_addr;
        addr_bit_cnt <= nxt_addr_bit_cnt;
        addr_valid <= nxt_addr_valid;
        wr_data <= nxt_wr_data;
        //rd_data <= nxt_rd_data;
        data_bit_cnt <= nxt_data_bit_cnt;
        xip_confirm <= nxt_xip_confirm;
        if (nxt_cmd_valid && !cmd_valid) begin
            pkt.addr_datawidth = get_addr_datawidth(nxt_cmd);
            pkt.data_datawidth = get_data_datawidth(nxt_cmd);
            pkt.addr_bytes = get_addr_bytes(nxt_cmd);
            pkt.isread = get_cmd_isread(nxt_cmd);
            pkt.iswrite = get_cmd_iswrite(nxt_cmd);
            pkt.dcc = get_cmd_dcc(nxt_cmd);
            pkt.dtr = get_cmd_isdtr(nxt_cmd);
        end
        if (nxt_cmd_valid && !cmd_valid && nxt_cmd==CMD_ENTER_DEEP_POWER_DOWN) begin
            mode_cfg.powerdown = 1;
        end else if (nxt_cmd_valid && !cmd_valid && nxt_cmd==CMD_EXIT_DEEP_POWER_DOWN) begin
            mode_cfg.powerdown = 0;
        end else if (nxt_cmd_valid && !cmd_valid && nxt_cmd==CMD_RESET_ENABLE) begin
            mode_cfg.soft_reset_en = 1;
        end else if (!mode_cfg.powerdown) begin
            if (wren && cmd_valid && pkt.iswrite && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                addr_page = pkt.addr[31:0]&32'h7fff00;
                addr_page_byte = (pkt.addr + data_bit_cnt[31:3]) & 32'hff;
                if (mode_cfg.pmm) begin
                    addr_page_full = (pkt.addr + data_bit_cnt[31:3]) & 32'h7fffff;
                end else begin
                    addr_page_full = addr_page | addr_page_byte;
                end
                //$display("aolsen pmm=%b orig_addr=%08x addr_page=%08x row=%x col=%x wr_data=%02x dbg=%02x",
                    //mode_cfg.pmm, pkt.addr+data_bit_cnt[31:3], addr_page_full, addr_page, addr_page_byte, nxt_wr_data, mem['h5bbacf]);
                debug_addr = addr_page_full;
                debug_data = nxt_wr_data;
                if ((addr_page_full >= mode_cfg.bp_addr_min) && (addr_page_full <= mode_cfg.bp_addr_max)) begin
                    flag_status_reg[1] = 1;
                end else begin
                    mem[addr_page_full] = nxt_wr_data;
                end
            end else if (wren && cmd_valid && pkt.cmd==CMD_WRITE_NVCFG_REG && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                //$display("aolsen nvcfg addr=%h data=%h", pkt.addr + data_bit_cnt[31:3], nxt_wr_data);
                cfg_reg_nv[pkt.addr+data_bit_cnt[31:3]] = nxt_wr_data;
            end else if (wren && cmd_valid && pkt.cmd==CMD_WRITE_VCFG_REG && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                //$display("aolsen nvcfg addr=%h data=%h", pkt.addr + data_bit_cnt[31:3], nxt_wr_data);
                if (pkt.addr+data_bit_cnt[31:3] == 30) begin
                    cfg_reg_vol_dfim = nxt_wr_data == 8'h6b;
                end else if (pkt.addr+data_bit_cnt[31:3] == 16) begin
                    // W1C -- Write 1 to clear
                    cfg_reg_vol[pkt.addr+data_bit_cnt[31:3]] = cfg_reg_vol[pkt.addr+data_bit_cnt[31:3]] & (~nxt_wr_data);
                    cfg_reg_op[pkt.addr+data_bit_cnt[31:3]] = cfg_reg_op[pkt.addr+data_bit_cnt[31:3]] & (~nxt_wr_data);
                end else begin
                    cfg_reg_vol[pkt.addr+data_bit_cnt[31:3]] = nxt_wr_data;
                    cfg_reg_op[pkt.addr+data_bit_cnt[31:3]] = nxt_wr_data;
                end
            end else if (wren && cmd_valid && pkt.cmd==CMD_TDP_WRITE && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                cfg_tdp_reg[pkt.addr+data_bit_cnt[31:3]] = nxt_wr_data;
            end else if (wren && !cfg_otp_locked && cmd_valid && pkt.cmd==CMD_WRITE_OTP && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                cfg_otp_reg[pkt.addr+data_bit_cnt[31:3]] = nxt_wr_data;
            end else if (wren && cmd_valid && pkt.cmd==CMD_WRITE_SR && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                // for octal dtr, only use posedge of CK
                if (mode_cfg.addr_datawidth != 8 || mode_cfg.addr_dtr != 1 || CK != 0) begin
                    status_reg[7:2] = nxt_wr_data[7:2];
                end
            end else if (cmd_valid && pkt.cmd==CMD_IF_ACT && nxt_data_bit_cnt > 0 && (nxt_data_bit_cnt%8 == 0)) begin
                cfg_if_cmd_reg[data_bit_cnt[31:3]+2] = nxt_wr_data;
            end else if (nxt_cmd_valid && !cmd_valid && nxt_cmd==CMD_CLEAR_FLAG_SR) begin
                // clear the error flags, leave the status flags alone
                flag_status_reg[6:1] = 0;
            end
        end
    end
end

logic [31:0] crc_start_addr;
logic [31:0] crc_end_addr;
logic [63:0] crc_exp;
logic [63:0] crc;
logic [7:0] tmp_crc_byte;
always @(posedge CS_N or negedge rst_n) begin
    if (rst_n && (cfg_if_cmd_reg[2] == 'h27)) begin
        crc = 0;
        if (cfg_if_cmd_reg[3] == 8'hff || cfg_if_cmd_reg[3] == 8'hfe) begin
            if (cfg_if_cmd_reg[3] == 'hff) begin
                crc_start_addr = 0;
                crc_end_addr = 'h7fffff;
            end else if (cfg_if_cmd_reg[3] == 'hfe) begin
                crc_start_addr = {cfg_if_cmd_reg[14], cfg_if_cmd_reg[13], cfg_if_cmd_reg[12]};
                crc_end_addr = {cfg_if_cmd_reg[18], cfg_if_cmd_reg[17], cfg_if_cmd_reg[16]};
            end
            for (int i=0; i<8; i=i+1) begin
                crc_exp[i*8+:8] = cfg_if_cmd_reg[4+i];
            end
            crc = 0;
            for (int addr=crc_start_addr; addr <= crc_end_addr; addr=addr+1) begin
                tmp_crc_byte = mem[addr];
                if (tmp_crc_byte == 8'bx) tmp_crc_byte = 8'hee;
                crc = gen_crc_64(crc, tmp_crc_byte);
            end
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            crc = gen_crc_64(crc, 0);
            if (crc != crc_exp) begin
                flag_status_reg[3] = 1;
                for (int i=0; i<8; i=i+1) begin
                    cfg_gprr_reg[i] = crc[i*8+:8];
                end
            end else begin
                for (int i=0; i<8; i=i+1) begin
                    cfg_gprr_reg[i] = 0;
                end
            end
            cfg_reg_vol[16][1] = 1;
        end
    end
    for (int i=0; i<20; i=i+1) begin
        cfg_if_cmd_reg[i] = 0;
    end
end

always @(posedge CK or posedge CS_N or negedge rst_n) begin
    if (!rst_n) begin
        wren <= 0;
        soft_reset <= 0;
    end else begin
        if (mode_cfg.soft_reset_en && cmd_valid && (pkt.cmd == CMD_RESET_MEMORY)) begin
            // TODO: Better internal delay for this?
            soft_reset <= #1ns 1;
            mode_cfg.powerdown = 0;
        end else if (!mode_cfg.powerdown) begin
            if (cmd_valid && (pkt.cmd == CMD_WRITE_ENB)) begin
                wren <= 1;
                status_reg[1] = 1;
            end else if (cmd_valid && (pkt.cmd == CMD_WRITE_DIS)) begin
                wren <= 0;
                status_reg[1] = 0;
            end else if (cmd_valid && pkt.cmd==CMD_ENTER_QUAD_IO) begin
                cfg_reg_vol[0] = DEC_QSPI_NO_DS;
                cfg_reg_op[0] = DEC_QSPI_NO_DS;
            end else if (cmd_valid && pkt.cmd==CMD_EXIT_QUAD_IO && mode_cfg.cmd_datawidth == 4) begin
                cfg_reg_op[0] = DEC_USPI_NO_DS;
                cfg_reg_vol[0] = DEC_USPI_NO_DS;
            end else if (cmd_valid && pkt.cmd==CMD_ENTER_4_BYTE_ADDR) begin
                // Don't change the vol register, just the operation mode
                //cfg_reg_vol[5] = P_ADDR_4BYTE;
                cfg_reg_op[5] = P_ADDR_4BYTE;
            end else if (cmd_valid && pkt.cmd==CMD_EXIT_4_BYTE_ADDR) begin
                // Don't change the vol register, just the operation mode
                //cfg_reg_vol[5] = '1;
                cfg_reg_op[5] = '1;
            end else if (wren && cmd_valid && addr_valid &&
                ((pkt.cmd==CMD_ERASE_4K) ||
                 (pkt.cmd==CMD_ERASE_4K_4_BYTE) ||
                 (pkt.cmd==CMD_ERASE_32K) ||
                 (pkt.cmd==CMD_ERASE_32K_4_BYTE) ||
                 (pkt.cmd==CMD_ERASE_SECTOR) ||
                 (pkt.cmd==CMD_ERASE_SECTOR_4_BYTE) ||
                 (pkt.cmd==CMD_ERASE_BULK) ||
                 (pkt.cmd==CMD_ERASE_BULK_C7) ||
                 (pkt.cmd==CMD_ERASE_BULK_60))) begin
                do_erase();
            end
        end
    end
end

assign IO0 = !SHADOW && IO_oe[0] ? IO_o[0] : 1'bz;
assign IO1 = !SHADOW && IO_oe[1] ? IO_o[1] : 1'bz;
assign IO2 = !SHADOW && IO_oe[2] ? IO_o[2] : 1'bz;
assign IO3 = !SHADOW && IO_oe[3] ? IO_o[3] : 1'bz;
assign IO4 = !SHADOW && IO_oe[4] ? IO_o[4] : 1'bz;
assign IO5 = !SHADOW && IO_oe[5] ? IO_o[5] : 1'bz;
assign IO6 = !SHADOW && IO_oe[6] ? IO_o[6] : 1'bz;
assign IO7 = !SHADOW && IO_oe[7] ? IO_o[7] : 1'bz;
assign DS = !SHADOW && DS_oe ? DS_o : 1'bz;

assign INT_N = !SHADOW && INT_N_oe ? INT_N_o : 1'bz;

logic [31:0] tmp;
logic addr_valid_ne;

task drive_read_byte;
    input logic [7:0] data_byte;
    input logic [3:0] datawidth;
    input logic       dtr;
    logic [7:0] bit_cnt;
begin
    bit_cnt = 0;
    while(!CS_N && bit_cnt < 8) begin
        nxt_DS_o = !nxt_DS_o;
        if (datawidth==1) begin
            nxt_IO_oe[1] = 1;
            nxt_IO_o[1] = data_byte[7-bit_cnt[2:0]];
        end else if (datawidth==2) begin
            nxt_IO_oe[1:0] = 2'h3;
            nxt_IO_o[0] = data_byte[6-bit_cnt[2:0]];
            nxt_IO_o[1] = data_byte[7-bit_cnt[2:0]];
        end else if (datawidth==4) begin
            nxt_IO_oe[3:0] = 4'hf;
            nxt_IO_o[0] = data_byte[4-bit_cnt[2:0]];
            nxt_IO_o[1] = data_byte[5-bit_cnt[2:0]];
            nxt_IO_o[2] = data_byte[6-bit_cnt[2:0]];
            nxt_IO_o[3] = data_byte[7-bit_cnt[2:0]];
        end else begin
            nxt_IO_oe[7:0] = 8'hff;
            nxt_IO_o[0] = data_byte[0-bit_cnt[2:0]];
            nxt_IO_o[1] = data_byte[1-bit_cnt[2:0]];
            nxt_IO_o[2] = data_byte[2-bit_cnt[2:0]];
            nxt_IO_o[3] = data_byte[3-bit_cnt[2:0]];
            nxt_IO_o[4] = data_byte[4-bit_cnt[2:0]];
            nxt_IO_o[5] = data_byte[5-bit_cnt[2:0]];
            nxt_IO_o[6] = data_byte[6-bit_cnt[2:0]];
            nxt_IO_o[7] = data_byte[7-bit_cnt[2:0]];
        end
        if (dtr) begin
            @(CK or CS_N);
        end else begin
            @(negedge CK or CS_N);
        end
        bit_cnt = bit_cnt + datawidth;
    end
end
endtask

task drive_read_data;
    input logic [3:0] datawidth;
    input logic       dtr;
    logic [31:0] bit_cnt;
    logic [7:0] data_byte;
    logic [31:0] addr_masked;
begin
    //$display("aolsen drive_read_data");
    nxt_DS_oe = 1;
    nxt_DS_o = 0;
    fork
        if (pkt.dtr && (pkt.addr_bytes != 0)) begin
            repeat(pkt.dcc ? pkt.dcc-1 : 0) @(negedge CK);
        end else begin
            repeat(pkt.dcc) @(negedge CK);
        end
    join_any
    disable fork;
    bit_cnt = 0;
    while(!CS_N) begin
        addr_masked = (pkt.addr+bit_cnt[31:3]) & 32'h7fffff;
        if (pkt.cmd == CMD_READ_VCFG_REG || pkt.cmd == CMD_READ_NVCFG_REG || pkt.cmd == CMD_READ_ID_9E || pkt.cmd == CMD_READ_ID_9F || pkt.cmd == CMD_READ_ID_MIO) begin
            // The way the RTL is coded makes these commands act like wrap 32
            addr_masked = ((pkt.addr&32'hffffffe0)|((pkt.addr[4:0]+bit_cnt[31:3])&32'h1f)) & 32'h7fffff;
        end else if (pkt.cmd == CMD_TDP_READ || pkt.cmd == CMD_TDP_READ_DTR) begin
            // These act like wrap 64
            addr_masked = ((pkt.addr&32'hffffffc0)|((pkt.addr[5:0]+bit_cnt[31:3])&32'h3f)) & 32'h7fffff;
        end else if (pkt.cmd == CMD_READ_OTP) begin
            // When going over the size of OTP register it will repeat the last byte.
            // Let the address go over and detect it later
            addr_masked = (pkt.addr+bit_cnt[31:3]) & 32'h7fffff;
        end else begin
            case (mode_cfg.wrap)
                0: addr_masked = (pkt.addr+bit_cnt[31:3]) & 32'h7fffff;
                16: addr_masked = ((pkt.addr&32'hfffffff0)|((pkt.addr[3:0]+bit_cnt[31:3])&32'h0f)) & 32'h7fffff;
                32: addr_masked = ((pkt.addr&32'hffffffe0)|((pkt.addr[4:0]+bit_cnt[31:3])&32'h1f)) & 32'h7fffff;
                64: addr_masked = ((pkt.addr&32'hffffffc0)|((pkt.addr[5:0]+bit_cnt[31:3])&32'h3f)) & 32'h7fffff;
            endcase
        end
        if (pkt.cmd == CMD_READ_VCFG_REG) begin
            data_byte = (addr_masked >= 32) ? 8'hff :
                        (addr_masked == 30) ? cfg_reg_vol_dfim :
                        (addr_masked >= 16) ? 8'h00 :
                        cfg_reg_vol[addr_masked];
        end else if (pkt.cmd == CMD_READ_NVCFG_REG) begin
            data_byte = (addr_masked >= P_NUM_NV_REG) ? 8'hff : cfg_reg_nv[addr_masked];
        end else if (pkt.cmd == CMD_READ_ID_9E || pkt.cmd == CMD_READ_ID_9F || pkt.cmd == CMD_READ_ID_MIO) begin
            data_byte = (addr_masked >= 32) ? 8'hff : read_id_reg[addr_masked];
        end else if (pkt.cmd == CMD_TDP_READ || pkt.cmd == CMD_TDP_READ_DTR) begin
            data_byte = cfg_tdp_reg[addr_masked];
        end else if (pkt.cmd == CMD_READ_OTP) begin
            data_byte = cfg_otp_reg[addr_masked];
        end else if (pkt.cmd == CMD_READ_GPRR) begin
            data_byte = cfg_gprr_reg[addr_masked];
        end else begin
            data_byte = mem[addr_masked];
        end
        //$display("aolsen drive_read_data addr=%h bit_cnt=%h addr_masked=%h data=%h", pkt.addr, bit_cnt[31:3], addr_masked, data_byte);
        if (data_byte === 8'bx) begin
            if (pkt.cmd == CMD_READ_OTP) begin
                data_byte = cfg_otp_reg[256];
            end else if (pkt.cmd == CMD_READ_GPRR) begin
                data_byte = 8'h00;
            end else begin
                data_byte = 8'hee;
            end
        end
        drive_read_byte(data_byte, datawidth, dtr);
        bit_cnt = bit_cnt + 8;
    end
    nxt_IO_oe = 0;
    nxt_IO_o = 'bz;
    nxt_DS_oe = 0;
    nxt_DS_o = 'bz;
end
endtask

task drive_read_reg;
    input logic [7:0] data;
    input logic [3:0] datawidth;
    input logic       dtr;
    logic [31:0] bit_cnt;
begin
    //$display("aolsen drive_reg_reg %h", data);
    fork
        if (pkt.dtr && (pkt.addr_bytes != 0)) begin
            repeat(pkt.dcc ? pkt.dcc-1 : 0) @(negedge CK);
        end else begin
            repeat(pkt.dcc) @(negedge CK);
        end
    join_any
    disable fork;
    bit_cnt = 0;
    nxt_DS_oe = 1;
    nxt_DS_o = 0;
    while(!CS_N) begin
        drive_read_byte(data, datawidth, dtr);
        bit_cnt = bit_cnt + 8;
    end
    nxt_IO_oe = 0;
    nxt_IO_o = 'bz;
    nxt_DS_oe = 0;
    nxt_DS_o = 'bz;
end
endtask

task do_erase;
    logic [31:0] erase_size;
    logic [31:0] erase_mask;
begin
    erase_size = 4*1024;
    if (pkt.cmd == CMD_ERASE_32K || pkt.cmd == CMD_ERASE_32K_4_BYTE) erase_size = 32*1024;
    else if (pkt.cmd == CMD_ERASE_SECTOR || pkt.cmd == CMD_ERASE_SECTOR_4_BYTE) erase_size = 64*1024;
    else if (pkt.cmd == CMD_ERASE_BULK || pkt.cmd == CMD_ERASE_BULK_C7 || pkt.cmd == CMD_ERASE_BULK_60) erase_size = 32'h800000;
    erase_mask = ~(erase_size - 1);
    //$display("aolsen do_erase");
    if ((pkt.addr >= mode_cfg.bp_addr_min) && (pkt.addr <= mode_cfg.bp_addr_max)) begin
        flag_status_reg[1] = 1;
    end else begin
        for (int i=0; i<erase_size; i=i+1) begin
            if (mode_cfg.erase_val==1) begin
                mem[(pkt.addr&erase_mask)+i] = '1;
            end else begin
                mem[(pkt.addr&erase_mask)+i] = '0;
            end
        end
    end
    // erase done interrupt status bit
    cfg_reg_vol[16][0] = 1;
end
endtask


function [3:0] get_addr_datawidth;
    input [7:0] cmd;
begin
    case(cmd)
        //CMD_READ_FAST_OCTAL_O_4_BYTE_ADDR                 ,
        //CMD_READ_FAST_OCTAL_O                             ,
        //CMD_READ_FAST_OCTAL_O_DTR                         ,
        CMD_READ_FAST_OCTAL_IO                            ,
        CMD_READ_FAST_OCTAL_IO_4_BYTE_ADDR                ,
        CMD_READ_FAST_OCTAL_IO_DTR                        ,
        //CMD_PROGRAM_FAST_OCTAL_INPUT                      ,
        //CMD_PROGRAM_FAST_OCTAL_INPUT_4_BYTE_ADDR          ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT_4_BYTE_ADDR ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT             :
            return 8;
        //CMD_READ_FAST_QUAD_O_4_BYTE_ADDR                ,
        //CMD_PROGRAM_FAST_QUAD_INPUT                     ,
        //CMD_PROGRAM_FAST_QUAD_INPUT_4_BYTE_ADDR         ,
        CMD_ENTER_QUAD_IO                               ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT            ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT_4_BYTE_ADDR,
        //CMD_READ_FAST_QUAD_O                            ,
        //CMD_READ_FAST_QUAD_O_4_BYTE_ADDR                ,
        //CMD_READ_FAST_QUAD_O_DTR                        ,
        CMD_READ_WORD_QUAD_IO_NO_DTR                    ,
        CMD_READ_FAST_QUAD_IO                           ,
        CMD_READ_FAST_QUAD_IO_4_BYTE_ADDR               ,
        CMD_READ_FAST_QUAD_IO_DTR                       ,
        CMD_READ_FAST_QUAD_IO_DTR_4_BYTE_ADDR           :
            return 4;
        //CMD_READ_FAST_DUAL_O                            ,
        //CMD_READ_FAST_DUAL_O_4_BYTE_ADDR                ,
        //CMD_READ_FAST_DUAL_O_DTR                        ,
        //CMD_PROGRAM_FAST_DUAL_INPUT                     ,
        CMD_READ_FAST_DUAL_IO                           ,
        CMD_READ_FAST_DUAL_IO_4_BYTE_ADDR               ,
        CMD_READ_FAST_DUAL_IO_DTR                       ,
        CMD_READ_FAST_DUAL_IO_DTR_4_BYTE_ADDR           ,
        CMD_PROGRAM_FAST_EXTENDED_DUAL_INPUT            :
            return 2;
        default:
            return mode_cfg.addr_datawidth;

    endcase
end
endfunction

function [3:0] get_data_datawidth;
    input [7:0] cmd;
begin
    case(cmd)
        CMD_READ_FAST_DUAL_O                              ,
        CMD_READ_FAST_DUAL_O_4_BYTE_ADDR                  ,
        CMD_READ_FAST_DUAL_O_DTR                          ,
        CMD_PROGRAM_FAST_DUAL_INPUT                       ,
        CMD_PROGRAM_FAST_EXTENDED_DUAL_INPUT              ,
        CMD_READ_FAST_DUAL_IO                             ,
        CMD_READ_FAST_DUAL_IO_4_BYTE_ADDR                 ,
        CMD_READ_FAST_DUAL_IO_DTR                         ,
        CMD_READ_FAST_DUAL_IO_DTR_4_BYTE_ADDR             ,
        CMD_PROGRAM_FAST_EXTENDED_DUAL_INPUT              :
            return 2;
        CMD_PROGRAM_FAST_QUAD_INPUT                       ,
        CMD_PROGRAM_FAST_QUAD_INPUT_4_BYTE_ADDR           ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT              ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT_4_BYTE_ADDR  ,
        CMD_READ_FAST_QUAD_O                              ,
        CMD_READ_FAST_QUAD_O_4_BYTE_ADDR                  ,
        CMD_READ_FAST_QUAD_O_DTR                          ,
        CMD_READ_WORD_QUAD_IO_NO_DTR                      ,
        CMD_READ_FAST_QUAD_IO                             ,
        CMD_READ_FAST_QUAD_IO_4_BYTE_ADDR                 ,
        CMD_READ_FAST_QUAD_IO_DTR                         ,
        CMD_READ_FAST_QUAD_IO_DTR_4_BYTE_ADDR             :
            return 4;
        CMD_READ_FAST_OCTAL_O_4_BYTE_ADDR                 ,
        CMD_PROGRAM_FAST_OCTAL_INPUT                      ,
        CMD_PROGRAM_FAST_OCTAL_INPUT_4_BYTE_ADDR          ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT             ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT_4_BYTE_ADDR ,
        CMD_READ_FAST_OCTAL_O                             ,
        CMD_READ_FAST_OCTAL_O_DTR                         ,
        CMD_READ_FAST_OCTAL_IO                            ,
        CMD_READ_FAST_OCTAL_IO_4_BYTE_ADDR                ,
        CMD_READ_FAST_OCTAL_IO_DTR                        :
            return 8;
        default:
            return mode_cfg.addr_datawidth;
    endcase
end
endfunction

function [2:0] get_addr_bytes;
    input [7:0] cmd;
begin
    case(cmd)
        CMD_READ_FAST_4_BYTE_ADDR                         ,
        CMD_READ_FAST_DTR_4_BYTE_ADDR                     ,
        CMD_PROGRAM_4_BYTE_ADDR                           ,
        CMD_READ_4_BYTE_ADDR                              ,
        CMD_ERASE_4K_4_BYTE                               ,
        CMD_PROGRAM_FAST_QUAD_INPUT_4_BYTE_ADDR           ,
        CMD_READ_FAST_DUAL_O_4_BYTE_ADDR                  ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT_4_BYTE_ADDR  ,
        CMD_ERASE_32K_4_BYTE                              ,
        CMD_READ_FAST_QUAD_O_4_BYTE_ADDR                  ,
        CMD_READ_FAST_OCTAL_O_4_BYTE_ADDR                 ,
        CMD_READ_FAST_OCTAL_IO_DTR                        ,
        CMD_PROGRAM_FAST_OCTAL_INPUT_4_BYTE_ADDR          ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT_4_BYTE_ADDR ,
        CMD_ENTER_4_BYTE_ADDR                             ,
        CMD_READ_FAST_DUAL_IO_4_BYTE_ADDR                 ,
        CMD_READ_FAST_DUAL_IO_DTR_4_BYTE_ADDR             ,
        CMD_READ_FAST_OCTAL_IO_4_BYTE_ADDR                ,
        CMD_ERASE_SECTOR_4_BYTE                           ,
        CMD_EXIT_4_BYTE_ADDR                              ,
        CMD_READ_FAST_QUAD_IO_4_BYTE_ADDR                 ,
        CMD_READ_FAST_QUAD_IO_DTR_4_BYTE_ADDR             :
            return 4;
        CMD_READ_GPRR                                     ,
        CMD_IF_ACT                                        ,
        CMD_READ_ID_9E                                    ,
        CMD_READ_ID_9F                                    ,
        CMD_READ_ID_MIO                                   ,
        CMD_WRITE_SR                                      ,
        CMD_READ_SR                                       :
            return 0;
        default:
            return mode_cfg.addr_bytes;
    endcase
end
endfunction

function get_cmd_iswrite;
    input [7:0] cmd;
begin
    case(cmd)
        CMD_PROGRAM_PAGE                                  ,
        CMD_PROGRAM_4_BYTE_ADDR                           ,
        CMD_PROGRAM_FAST_QUAD_INPUT                       ,
        CMD_PROGRAM_FAST_QUAD_INPUT_4_BYTE_ADDR           ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT              ,
        CMD_PROGRAM_FAST_EXTENDED_QUAD_INPUT_4_BYTE_ADDR  ,
        CMD_PROGRAM_FAST_OCTAL_INPUT                      ,
        CMD_PROGRAM_FAST_OCTAL_INPUT_4_BYTE_ADDR          ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT_4_BYTE_ADDR ,
        CMD_PROGRAM_FAST_DUAL_INPUT                       ,
        CMD_PROGRAM_FAST_EXTENDED_OCTAL_INPUT             ,
        CMD_PROGRAM_FAST_EXTENDED_DUAL_INPUT              :
            return 1;
        default:
            return 0;
    endcase
end
endfunction

function get_cmd_isread;
    input [7:0] cmd;
begin
    case(cmd)
        CMD_READ_0_LAT                                   ,
        CMD_READ_SR                                      ,
        CMD_READ_FAST                                    ,
        CMD_READ_FAST_4_BYTE_ADDR                        ,
        CMD_READ_FAST_DTR                                ,
        CMD_READ_FAST_DTR_4_BYTE_ADDR                    ,
        CMD_READ_4_BYTE_ADDR                             ,
        CMD_READ_FAST_DUAL_O                             ,
        CMD_READ_FAST_DUAL_O_4_BYTE_ADDR                 ,
        CMD_READ_FAST_DUAL_O_DTR                         ,
        CMD_READ_OTP                                     ,
        CMD_READ_SFDP                                    ,
        CMD_READ_EVCFG_REG                               ,
        CMD_READ_REG                                     ,
        CMD_READ_FAST_QUAD_O                             ,
        CMD_READ_FAST_QUAD_O_4_BYTE_ADDR                 ,
        CMD_READ_FAST_QUAD_O_DTR                         ,
        CMD_READ_FLAG_SR                                 ,
        CMD_READ_FAST_OCTAL_O_4_BYTE_ADDR                ,
        CMD_READ_VCFG_REG                                ,
        CMD_READ_FAST_OCTAL_O                            ,
        CMD_READ_GPRR                                    ,
        CMD_READ_FAST_OCTAL_O_DTR                        ,
        CMD_READ_ID_9E                                   ,
        CMD_READ_ID_9F                                   ,
        CMD_READ_ID_MIO                                  ,
        CMD_READ_NVCFG_REG                               ,
        CMD_READ_FAST_DUAL_IO                            ,
        CMD_READ_FAST_DUAL_IO_4_BYTE_ADDR                ,
        CMD_READ_FAST_DUAL_IO_DTR                        ,
        CMD_READ_FAST_DUAL_IO_DTR_4_BYTE_ADDR            ,
        CMD_READ_EXTENDED_ADDR_REG                       ,
        CMD_READ_FAST_OCTAL_IO                           ,
        CMD_READ_FAST_OCTAL_IO_4_BYTE_ADDR               ,
        CMD_READ_WORD_QUAD_IO_NO_DTR                     ,
        CMD_READ_FAST_QUAD_IO                            ,
        CMD_READ_FAST_QUAD_IO_4_BYTE_ADDR                ,
        CMD_READ_FAST_QUAD_IO_DTR                        ,
        CMD_READ_FAST_QUAD_IO_DTR_4_BYTE_ADDR            ,
        CMD_TDP_READ                                     ,
        CMD_TDP_READ_DTR                                 ,
        CMD_READ_DIE_SELECT                              ,
        CMD_READ_FAST_OCTAL_IO_DTR                       :
            return 1;
        default:
            return 0;
    endcase
end
endfunction

function [5:0] get_cmd_dcc;
    input [7:0] cmd;
begin
    case(cmd)
        CMD_READ_GPRR                                     :
            return 8;
        CMD_READ_WORD_QUAD_IO_NO_DTR                      :
            return 4;
        CMD_READ_4_BYTE_ADDR                              ,
        CMD_READ_0_LAT                                    :
            return 0;
        CMD_READ_SR                                       ,
        CMD_READ_FLAG_SR                                  ,
        CMD_READ_ID_9E                                    ,
        CMD_READ_ID_9F                                    ,
        CMD_READ_ID_MIO                                   ,
        CMD_READ_NVCFG_REG                                ,
        CMD_READ_VCFG_REG                                :
            return ((mode_cfg.cmd_datawidth == 4 && mode_cfg.addr_dtr) || (mode_cfg.cmd_datawidth == 8) ? 8 : 0);
        default:
            return mode_cfg.dcc;
    endcase
end
endfunction

// DTR commands force DTR mode
// otherwise use the configured mode
function get_cmd_isdtr;
    input [7:0] cmd;
begin
    case(cmd)
        CMD_READ_FAST_DTR                                ,
        CMD_READ_FAST_DTR_4_BYTE_ADDR                    ,
        CMD_READ_FAST_DUAL_O_DTR                         ,
        CMD_READ_FAST_QUAD_O_DTR                         ,
        CMD_READ_FAST_OCTAL_O_DTR                        ,
        CMD_READ_FAST_DUAL_IO_DTR                        ,
        CMD_READ_FAST_DUAL_IO_DTR_4_BYTE_ADDR            ,
        CMD_READ_FAST_QUAD_IO_DTR                        ,
        CMD_READ_FAST_QUAD_IO_DTR_4_BYTE_ADDR            ,
        CMD_TDP_READ_DTR                                 ,
        CMD_READ_FAST_OCTAL_IO_DTR                       :
            return 1;
        CMD_READ_WORD_QUAD_IO_NO_DTR                     :
            return 0;
        default:
            return mode_cfg.addr_dtr;
    endcase
end
endfunction

function get_cmd_isxip;
    input [7:0] cmd;
begin
    case (cmd)
        CMD_READ_FAST                                     :
            return 1;
        default:
            return 0;
    endcase
end
endfunction

function logic [63:00] gen_crc_64 (input logic [63:00] sr_in, input logic [07:00] data); begin
    logic [63:00] sr, poly;
    logic         so;
    poly = 64'h42F0_E1EB_A9EA_3693;
    // shift in data LSB first, init CRC = 0
    sr = sr_in;
    //for (int j=0; j<8; j++) begin
    for (int j=7; j>=0; j=j-1) begin
        so = sr[63];
        for (int i=63; i>0; i=i-1) sr[i] = poly[i] ? sr[i-1] ^ so : sr[i-1];
        sr[0] = poly[0] ? so ^ data[j] : data[j];
    end
    return sr;
end endfunction

always @(negedge CK) begin
    if (!CS_N && cmd_valid) begin
        if (0) begin
        end else if (pkt.cmd == CMD_READ_FLAG_SR) begin
            drive_read_reg(flag_status_reg, mode_cfg.addr_datawidth, mode_cfg.addr_dtr);
        end else if (pkt.cmd == CMD_READ_SR) begin
            drive_read_reg(status_reg, mode_cfg.addr_datawidth, mode_cfg.addr_dtr);
        end else if (pkt.isread && (addr_valid || pkt.addr_bytes==0)) begin
            drive_read_data(pkt.data_datawidth, pkt.dtr);
        end
    end
end

always_comb begin
    if (!rst_n) begin
        nxt_IO_oe = 'b0;
        nxt_IO_o  = 'bz;
        nxt_DS_oe = 'b0;
        nxt_DS_o  = 'bz;
        INT_N_oe  = 'b0;
        INT_N_o   = 'bz;
    end
end

// 6ns = less than tCLQV/tCHQV max of 7ns
always @(nxt_IO_oe) IO_oe <= #6ns nxt_IO_oe;
always @(nxt_IO_o) IO_o <= #6ns nxt_IO_o;
always @(nxt_DS_oe) DS_oe <= #6ns nxt_DS_oe;
always @(nxt_DS_o) DS_o <= #6ns nxt_DS_o;

endmodule

