//-------------------------------------------------------------------------------------------------
// 
//  st_mram_ddr4_tb.vh
//
//  Description: Everspin ST-MRAM DDR4 Behavioral Model Sample Testbench
//
//
//  REQUIRED Definitions:
//  WIDTH:                  X8 or X16
//
//-------------------------------------------------------------------------------------------------
// 
//  This confidential and proprietary software may be used only as authorized
//  by the included licensing agreement from Everspin titled 'EverspinSLA.txt'
//
//                   Copyright 2016 Everspin Technologies, Inc
//
//  The entire notice above must be reproduced and the Everspin SLA included for 
//  all authorized copies.
//
//-------------------------------------------------------------------------------------------------
//  Revision Control Version:   $Revision: 1.24 $
//  Current Revision:           2.5
//
//  Revision History:
//  2.0     2018.08.31      CPM             - first release, sync version to the model
//  2.2     2018.10.02      CPM             - more stimulus
//  2.3     2018.10.29      CPM             - version stamp
//  2.4     2018.11.19      CPM             - version stamp
//  2.5     2019.02.08      CPM             - version stamp
//
//-------------------------------------------------------------------------------------------------
`timescale 1ps / 1ps

`define BIN1333
`define FAST_SIM

module    tb_top;

`include "st_mram_ddr4_parameters_1G.vh"

parameter CLK_PERIOD = 1500;  //must be div by 4
parameter NOMEM      = 14'b10_0000_0000_0000;

wire                    ck_t;
wire                    ck_c;
reg                     reset_n;
reg                     cke;
reg                     cs_n;
reg                     odt;
reg                     act_n;
reg                     ras_n;
reg                     cas_n_a15;
reg                     we_n_a14;
`ifdef X8
wire                    dm_n_tdqs_t;
wire                    dqs_t;
wire                    dqs_c;
wire                    tdqs_c;
`else
wire                    dmu_n;
wire                    dml_n;
wire                    dqsu_t;
wire                    dqsu_c;
wire                    dqsl_t;
wire                    dqsl_c;
`endif
reg   [BG_BITS-1:0]     bg; 
reg   [BA_BITS-1:0]     ba; 
reg   [ADDR_BITS-3:0]   a;  
wire  [DQ_BITS-1:0]     dq; 
reg                     par;
wire                    alert_n;
reg                     ten;

`ifdef X16
wire x8_mode  = 1'b0;
wire x16_mode = 1'b1;
`else
wire x8_mode  = 1'b1;
wire x16_mode = 1'b0;
`endif

integer test_num;
string  test_msg;

reg clock_gate;
reg clock;
reg init_complete;
reg dq_en, dqs_en;
reg [DQ_BITS-1:0] dq_out;


typedef enum logic [3:0] {MR0=4'h0, MR1, MR2, MR3, MR4, MR5, MR6} e_mr;
//CMD - {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]}
typedef enum logic [6:0] {CMD_MRS =7'b1_1000_00, CMD_REF =7'b1_1001_00, CMD_SRE =7'b0_1001_00, 
                          CMD_PRE =7'b1_1010_00, CMD_PREA=7'b1_1010_01,
                          CMD_WR  =7'b1_1100_00, CMD_WRS8=7'b1_1100_10, CMD_WRAS8=7'b1_1100_11,
                          CMD_RD  =7'b1_1101_00, CMD_RDS8=7'b1_1101_10, CMD_RDAS8=7'b1_1101_11,
                          CMD_STO =7'b1_1011_00, CMD_STOA=7'b1_1011_01,
                          CMD_NOP =7'b1_1111_00, CMD_DES =7'bz_zzzz_zz, CMD_ZQCL =7'b1_1110_01, CMD_ZQCS=7'b1_1110_00,
                          CMD_NONE=7'bx_1xxx_xx,
                          CMD_ACT =7'b1_0xxx_xx} e_cmd;

logic [6:0] cmd_list [$] = '{CMD_MRS, CMD_REF, CMD_SRE, CMD_PRE, CMD_PREA, CMD_WR, CMD_WRS8, CMD_WRAS8,
                             CMD_RD, CMD_RDS8, CMD_RDAS8, CMD_STO, CMD_STOA, CMD_NOP, CMD_DES, CMD_ZQCL,
                             CMD_ZQCS, CMD_NONE, CMD_ACT};

integer CL, CWL;

logic [31:0]    write_pipe;
bit   [8*8-1:0] write_fifo[$];
logic [15:0]    data_list [8];
logic [15:0]    dq_wr;
logic           write_op;
integer         write_cnt;
logic           preamble;
logic           postamble;
logic           dqs_enb;
logic           dq_enb;
event           evt_data_shift;
event           evt_release_cs_n;
event           evt_debug;

e_cmd command;

wire [31:0] fifo_size = write_fifo.size(); // debug


`ifdef X16
assign dmu_n  = 1'b1;
assign dml_n  = 1'b1;
assign dqsu_t = dqs_enb ? ck_t  : 'bz;
assign dqsu_c = dqs_enb ? ck_c  : 'bz;
assign dqsl_t = dqs_enb ? ck_t  : 'bz;
assign dqsl_c = dqs_enb ? ck_c  : 'bz;
`else
assign dqs_t = dqs_enb ? ck_t  : 'bz;
assign dqs_c = dqs_enb ? ck_c  : 'bz;
assign dm_n_tdqs_t = 1'b1;
`endif

assign dq    = dq_enb  ? dq_wr : 'bz;

initial begin: CLK_GEN
    clock = 0;
    forever #(CLK_PERIOD/2) clock = ~clock;
end

initial begin: SHM_PROBE
    $shm_open("waves.shm");
    $shm_probe("ASMT");
end

initial begin: RESET_GEN
    init_complete = 0;
    test_msg   = "INIT ...";
    cs_n       = 1'b1;
    dq_wr      = 'bz;
    odt        = 1'b0;
    clock_gate = 0;
    reset_n    = 0;
    ten        = 1'b0;
    CWL        = 9;
    CL         = 10;
    write_pipe = 0;
    test_num   = 0;
    write_cnt  = 0;
    dq_enb     = 0;
    preamble   = 0;
    postamble  = 0;
    cke        = 0;
    /*
    #(TPW_RESET_L - TCKE2RESET);
    cke = 1'b0;
    #TCKE2RESET;
    reset_n = 1;
    #(TRESET2CKE - TCKSRX);
    clock_gate = 1;
    #(TCKSRX - TIS);
    odt = 1'b0;
    */
    #TPW_RESET_L reset_n = 1'b1;
    #(TXPR);
    init_complete = 1;
    wait_clks(10);
    cke = 1'b1;
  end

assign ck_t =  clock;
assign ck_c = ~clock;


initial begin: STIMULUS

    /////////////////////////////////////////
    //                                     //
    //           S T I M U L U S           //
    //                                     //
    /////////////////////////////////////////
    //wait_clks(TXPR_TCK+1);
    @(posedge cke)
    #TXPR;
    do_mrs(MR3, 14'b0000000000000);
    wait_clks(TMRD);
    do_mrs(MR6, 14'b0000000000000);
    wait_clks(TMRD);
    do_mrs(MR5, 14'b0000110000000);
    wait_clks(TMRD);
    do_mrs(MR4, 14'b0000000000000);
    wait_clks(TMRD);
    do_mrs(MR2, 14'b0000000000000);     // CWL=9
    wait_clks(TMRD);
    do_mrs(MR1, 14'b0001100000001);
    wait_clks(TMRD);
    do_mrs(MR0, 14'b00000100000101);    // CL=10
    wait_clks(TMRD);
    wait_clks(TDLLK);                   // wait DLL lock time
    //
    do_act( 'h0, 'h0);                  // bank, row
    #TRCD;
    // running in X8
    data_list = {16'h8000, 16'h8101, 16'h8202, 16'h8303,
                 16'h8404, 16'h8505, 16'h8606, 16'h8707};
    do_write(CMD_WRS8, 'h0, 'h00, data_list);            // addr, data
    wait_clks(20);
    do_read(CMD_RDS8, 'h0, 'h0);
    wait_clks(32);
    do_write(CMD_WRS8, 'h0, 'h10, {16'h0440, 16'h1441, 16'h2442, 16'h3443, 16'h4444, 16'h5445, 16'h6446, 16'h7447});
    wait_clks(TCCD_S+0);
    do_write(CMD_WRS8, 'h0, 'h08, {16'h8448, 16'h9449, 16'hA44A, 16'hB44B, 16'hC44C, 16'hD44D, 16'hE44E, 16'hF44F});
    wait_clks(16);
    do_write(CMD_WRS8, 'h0, 'h08, {16'h0550, 16'h1551, 16'hzz52, 16'h3553, 16'h4554, 16'h5555, 16'h6556, 16'h7557});
    wait_clks(TCCD_S+1);
    do_write(CMD_WRS8, 'h0, 'h10, {16'h8558, 16'h9559, 16'hA55A, 16'hB55B, 16'hC55C, 16'hD55D, 16'hE55E, 16'hF55F});
    wait_clks(8);
    do_write(CMD_WRS8, 'h0, 'h08, {16'h0660, 16'h1661, 16'h2662, 16'h3663, 16'h4664, 16'h5665, 16'h6666, 16'h7667});
    wait_clks(TCCD_S+2);
    do_write(CMD_WRS8, 'h0, 'h10, {16'h8668, 16'h9669, 16'hA66A, 16'hB66B, 16'hC66C, 16'hD66D, 16'hE66E, 16'hF66F});
    wait_clks(2);
    do_pre('h0);
    #TRP
    //do_store(CMD_STOA);   // Store All
    do_store(CMD_STO, 'h0); // Store Bank 0
    //
    #TST
    do_pde;
    wait_clks(1000);
    do_pdx;
    #TXPR;

    wait_clks(10);
    $display("[TB] Enter NoMEM so calibration will not overwrite memory ...");
    do_mrs(MR0, 14'b00_0000_0000_0101 | NOMEM); // CL=10
    wait_clks(TMOD_TCK);
    // Do whatever calibration sequence
    do_cal;
    repeat (1000) @(posedge ck_t);
    do_mrs(MR0, 14'b00_0000_0000_0101);         // CL=10 - exit nomem
    wait_clks(TMOD_TCK);
    do_act( 'h0, 'h0);                          // bank, row
    #TRCD;
    do_read(CMD_RDS8, 'h0, 'h0);
    wait_clks(100);
    do_pre('h0);
    #TRP
    //
    
    -> evt_debug;
    do_act( 'h5, 'h1);                          // bank, row
    #TRCD;
    do_write(CMD_WRAS8, 'h5, 'h20, {16'h1111, 16'h2222, 16'h3333, 16'h4444, 16'h5555, 16'h6666, 16'h7777, 16'h8888});
    #TRP;
    //
    wait_clks(100);
    ram_inst.clear_cache();                     // debug - clear out cache - set to x
    wait_clks(100);
    //
    do_act( 'h5, 'h1);                          // bank, row
    #TRCD;
    do_read(CMD_RDAS8, 'h0, 'h20);              // Read with AP
    
    wait_clks(100);
    do_sre;
    wait_clks(1000);
    do_srx;
    repeat (100) @(posedge ck_t);
    do_zqcs;
    wait_clks(ZQ_ZQCS_CLKS-1);  // violate ZQCS by CKE=0
    #200 cke = 1'b0;
    wait_clks(20);
    #TCKE_MIN;
    wait_clks(1);
    #200 cke = 1'b1;
    wait_clks(1);
    do_nop; // violate tXP
    wait_clks(100);
    dump_errors();
    $finish;
end



always_comb begin
    write_op = write_cnt!=0;
    dqs_enb  = preamble | postamble | dq_enb;
    command = cs_n==1'b0 ? e_cmd'({cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]}) : CMD_NONE;
end

always @(posedge ck_t or negedge ck_t) begin
    if ( preamble || (write_op && (write_cnt>2 || write_pipe[1:0]!=0))) #(CLK_PERIOD/4) -> evt_data_shift;
end

always @(evt_data_shift) begin
    dq_wr = write_fifo.size()==0 ? 'bx : write_fifo.pop_front();
end

always @(evt_release_cs_n) begin
    @(negedge ck_t) cs_n = 1'b1;
end

always @(posedge ck_t) begin
    write_pipe <= {1'b0,write_pipe[31:1]};
    if ( !cs_n && ( {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} == CMD_WR    ||
                    {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} == CMD_WRS8  ||
                    {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} == CMD_WRAS8 )) begin
        write_pipe[CWL] <= 1'b1;
    end
    preamble <= write_pipe[2] & (~write_op | write_op & write_cnt==1);
    if ( write_pipe[2:0]==2 ) begin
        write_cnt <= write_cnt + (write_op ? 7 : 8); // FIX for BC4
        preamble  <= 0;
    end else if ( write_cnt > 0) begin
        write_cnt <= write_cnt - 1;
    end
    postamble <= write_cnt==1 & write_pipe[1:0]==0;
    if ( write_cnt==1 && write_pipe[1:0]==0) dq_enb <= 0;
end

always @(posedge ck_c) begin
    if ( write_cnt > 0) begin
        write_cnt <= write_cnt - 1;
    end
    postamble <= 0;
    if ( write_pipe[2:0]==2 ) dq_enb <= 1;
end

task do_cal();
    begin : TSK_DO_CAL
        // Not a real calibration - just writes and reads
        do_act( 'h0, 'h0);                  // bank, row
        #TRCD;
        // running in X8
        data_list = {16'h5555, 16'hAAAA, 16'h6666, 16'h9999,
                     16'hF0F0, 16'hF0F0, 16'h5A5A, 16'hA5A5};
        do_write(CMD_WRS8, 'h0, 'h00, data_list);            // addr, data
        wait_clks(20);
        do_read(CMD_RDS8, 'h0, 'h0);
        #TRTP;
        do_pre('h0);
        #TRP;
    end
endtask


task wait_clks ( input integer num=1 ); begin : TSK_WAIT_CLKS
        repeat (num) @(posedge ck_t);
end; endtask


task do_mrs (
    input logic [3:0]  mr,
    input logic [13:0] opc
    );
    begin : TSK_DO_MRS
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_MRS;
        a[13:0] =  opc;
        ba = mr[1:0];
        bg = mr[3:2];
        -> evt_release_cs_n;
    end
endtask

task do_ref;
    begin : TSK_DO_REF
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_REF;
        a[13:0] = '0;
        ba = '0;
        bg = '0;
        -> evt_release_cs_n;
    end
endtask


task do_sre;
    begin : TSK_DO_SRE
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_SRE;
        a[13:0] = '0;
        ba = '0;
        bg = '0;
        -> evt_release_cs_n;
    end
endtask



task do_srx;
    begin : TSK_DO_SRX
        @(negedge ck_t);
        cke = 1'b1;
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_NOP;
        a[13:0] = '0;
        ba = '0;
        bg = '0;
        -> evt_release_cs_n;
    end
endtask


task do_pre (
    input logic [3:0] bank
    );
    begin: TSK_DO_PRE
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_PRE;
        ba = bank[1:0];
        bg = x8_mode ? bank[3:2] : {1'bz,bank[2]};
        -> evt_release_cs_n;
    end
endtask

task do_prea;
    begin : TSK_DO_PREA
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_PREA;
        -> evt_release_cs_n;
    end
endtask

task do_act (
    input logic [3:0]  bank,
    input logic [15:0] row_addr
    );
    begin : TSK_DO_ACT
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[13:0]} = {5'h10, row_addr[13:0]};
        ba = bank[1:0];
        bg = x8_mode ? bank[3:2] : {1'bz,bank[2]};
        @(negedge ck_t);
        act_n = 1'b1;
        cs_n  = 1'b1;
    end
endtask


task do_write (
    input e_cmd        cmd,
    input logic [3:0]  bank,
    input logic [6:0]  col_addr,
    input logic [15:0] data [8]
    );
    integer i;
    begin: TSK_DO_WRITE
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = cmd;
        a[6:0]  = col_addr;
        {bg,ba} = bank;
        foreach (data[i]) begin
            write_fifo.push_back(data[i]);
        end
        -> evt_release_cs_n;
    end
endtask


task do_read (
    input e_cmd        cmd,
    input logic [3:0]  bank,
    input logic [6:0]  col_addr
    );
    begin : TSK_DO_READ
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = cmd;
        a[6:0]  = col_addr;
        {bg,ba} = bank;
        -> evt_release_cs_n;
    end
endtask


task do_store (
    input logic [6:0]  cmd,
    input logic [3:0]  bank=0
    );
    begin : TSK_DO_STORE
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = cmd;
        if (cmd==CMD_STO) {bg,ba} = bank;
        -> evt_release_cs_n;
    end
endtask


task do_nop ( input integer num=1 );
    begin : TSK_DO_NOP
        @(negedge ck_t);
        cs_n  = 1'b0;
        a = 'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_NOP;
        -> evt_release_cs_n;
    end
endtask


task do_des ( input integer num=1 );
    begin : TSK_DO_DES
        @(negedge ck_t);
        cs_n  = 1'b1;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_DES;
        -> evt_release_cs_n;
    end
endtask


task do_pde;
    begin : TSK_DO_PDE
        @(negedge ck_t);
        cs_n  = 1'b1;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = 'z;
        cke = 1'b0;
    end
endtask


task do_pdx;
    begin : TSK_DO_PDX
        @(negedge ck_t);
        cs_n  = 1'b1;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = 'z;
        cke = 1'b1;
    end
endtask


task do_zqcl;
    begin : TSK_DO_ZQCL
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_ZQCL;
        -> evt_release_cs_n;
    end
endtask

task do_zqcs;
    begin : TSK_DO_ZQCS
        @(negedge ck_t);
        cs_n  = 1'b0;
        {cke, act_n, ras_n, cas_n_a15, we_n_a14, a[12], a[10]} = CMD_ZQCS;
        @(negedge ck_t);
        cs_n  = 1'b1;
    end
endtask

function string get_param(input string msg, input string delimiter); begin
    for (int i=0; i < msg.len(); i++) if ( msg[i]==delimiter ) return msg.substr(0,i-1);
end endfunction

task dump_errors;
    integer i;
    string  msg_list[$];
    begin : TSK_DUMP_ERRORS
        msg_list = ram_inst.err_list;
        $display("%0d Errors Logged", msg_list.size());
        foreach (msg_list[i]) begin
            $display("%03d: %s <%s>", i, msg_list[i], get_param(msg_list[i], " "));
        end
end endtask

st_mram_ddr4 ram_inst (
    .reset_n    (reset_n),
    .ck_t       (ck_t),
    .ck_c       (ck_c),
    .cke        (cke),
    .cs_n       (cs_n),
    .odt        (odt),
    .act_n      (act_n),
    .ras_n      (ras_n),
    .cas_n_a15  (cas_n_a15),
    .we_n_a14   (we_n_a14),
 `ifdef X8
    .dm_n_tdqs_t(dm_n_tdqs_t),
    .dqs_t      (dqs_t),
    .dqs_c      (dqs_c),
    .tdqs_c     (tdqs_c),
`else
    .dmu_n      (dmu_n),
    .dml_n      (dml_n),
    .dqsu_t     (dqsu_t),
    .dqsu_c     (dqsu_c),
    .dqsl_t     (dqsl_t),
    .dqsl_c     (dqsl_c),
`endif
    .bg         (bg), 
    .ba         (ba), 
    .a          (a),  
    .dq         (dq), 
    .par        (par),
    .alert_n    (alert_n),
    .ten        (ten) 
);


endmodule
