//-------------------------------------------------------------------------------------------------
// 
//  st_mram_ddr4_model.v
//
//  Description: Everspin ST-MRAM DDR4 Behavioral Model
//
//-------------------------------------------------------------------------------------------------
// 
//  This confidential and proprietary software may be used only as authorized
//  by the included licensing agreement from Everspin titled 'EverspinSLA.txt'
//
//                   Copyright 2016 Everspin Technologies, Inc
//
//  The entire notice above must be reproduced and the Everspin SLA included for 
//  all authorized copies.
//
//-------------------------------------------------------------------------------------------------
//  Revision Control Version:   $Revision: 1.140 $
//  Current Revision:           2.5
//
//  Revision History:
//  2.0     2018.08.31      CPM             - implemented user/controller side checks
//  2.1     2018.09.07      CPM             - fixed Precharge All closing of all banks
//                                          - warning message on already active row prints active 
//                                            not not new row
//  2.2     2018.10.02      CPM             - fixed rounding error for tRTP check
//                                          - added ZQCal check: CKE, DES
//  2.3     2018.10.29      CPM             - fixed AP tracking of open banks if DM during last burst
//                                          - syntax changes to remove Vivado warnings
//  2.4     2018.11.19      CPM             - added per bank tracking of STOre command
//  2.5     2019.02.08      CPM             - added tST check for Refresh command
//                                          - warning if DQ or DM is x during write
//                                          - added S1 S2 spec timing
//------------------------------------------------------------------------------------------------- 

// Note - timescale must be set to 1ps/1ps
`timescale 1ps / 1ps

`ifndef RUNSIM
  `define FAST_SIM
  `define BIN1333
  `define X16
`endif

module st_mram_ddr4 (
    reset_n,
    ck_t,
    ck_c,
    cke,
    cs_n,
    odt,
    act_n,
    ras_n,
    cas_n_a15,
    we_n_a14,
`ifdef X8
    dm_n_tdqs_t,
    dqs_t,
    dqs_c,
    tdqs_c,
`else
    dmu_n,
    dml_n,
    dqsu_t,
    dqsu_c,
    dqsl_t,
    dqsl_c,
`endif
    bg,
    ba,
    a,
    dq,
    par,
    alert_n,
    ten
);

`include "st_mram_ddr4_parameters_1G.vh"

// Ports
input           reset_n;
input           ck_t;
input           ck_c;
input           cke;
input           cs_n;
input           odt;
input           act_n;
input           ras_n;
input           cas_n_a15;
input           we_n_a14;
`ifdef X8
inout           dm_n_tdqs_t;
inout           dqs_t;
inout           dqs_c;
output          tdqs_c;
`else
input           dmu_n;
input           dml_n;
inout           dqsu_t;
inout           dqsu_c;
inout           dqsl_t;
inout           dqsl_c;
`endif
input [BG_BITS-1:0]     bg;
input [BA_BITS-1:0]     ba;
input [ADDR_BITS-3:0]   a;    // -3 because a14 and a15 are combined with we_n and cas_n above
inout [DQ_BITS-1:0]     dq;
input           par;
inout           alert_n;
input           ten;

// pragma protect

// pragma protect begin
// let max(a,b) = (a > b) ? a : b;
// convert to a fuction
function real max;
input real  a,b;

assign max = (a > b) ? a : b;
endfunction

parameter // {cs_n, act_n, ras_n, cas_n_a15, we_n_a14}
    MRS = 5'b01000,
    REF = 5'b01001,
    PRE = 5'b01010,
    ACT = 5'b00xxx,   // bottom 3 bits are Row Address
    STO = 5'b01011,
    WR  = 5'b01100,
    RD  = 5'b01101,
    ZQC = 5'b01110,
    NOP = 5'b01111,
    DES = 5'b1xxxx;

logic [4:0] cmd_list[10] = '{MRS, REF, PRE, ACT, STO, WR, RD, ZQC, NOP, DES};
wire  [4:0] cmd_dec     = {cs_n, act_n, ras_n, cas_n_a15, we_n_a14};

reg [127:0] cmd_str [31:0];
reg [12*8-1:0] command;
initial begin
    cmd_str[MRS] = "MRS      ";
    cmd_str[REF] = "Refresh  ";
    cmd_str[PRE] = "Precharge";
    cmd_str[0]   = "Activate ";
    cmd_str[WR ] = "Write    ";
    cmd_str[RD ] = "Read     ";
    cmd_str[NOP] = "NOP      ";  
    cmd_str[ZQC] = "ZQ       ";
    cmd_str[STO] = "Store    ";
end

always_comb begin
    casex (cmd_dec)
        MRS:     command = "MRS";
        REF:     command = "REF";
        PRE:     command = "PRE";
        ACT:     command = "ACT";
        STO:     command = "STO";
        WR:      command = "WR";
        RD:      command = "RD";
        ZQC:     command = "ZQC";
        NOP:     command = "NOP";
        DES:     command = "DES";
        default: command = "???";
    endcase
end

//event evt_write_to_file, read_from_file_evt;

`define BANKS (1<<BG_BITS) * (1<<BA_BITS)

parameter BL_MAX   = 8;     // Maximum Burst Length
parameter BC_BITS  = 4;     // Burst counter bits
parameter AP_BITS  = 1;     // Auto Precharge bits
parameter RFF_BITS = DQ_BITS*BL_MAX;
// %z format uses 8 bytes for every 32 bits or less.
parameter RFF_CHUNK = 16 * (RFF_BITS/32 + (RFF_BITS%32 ? 1 : 0));

// CPM
reg [1024:1] tmp_model_dir;
integer memfd[`BANKS-1:0];
//

// Associative array msg_q, indexed by string, stores queue
//int msg_q[string][$] ;
string info_list [$];
string warn_list [$];
string err_list [$];

// harcoded for 16 McKinley banks
logic [128*8-1:0] cache_mem[16];
logic [128*8-1:0] bank_mem [int][int];
event evt_write_cache;
event evt_write_mem;
event evt_clear_cache;
event evt_load_cache_from_mem;
event evt_do_store;
event evt_get_cache_data;
event evt_check_active_banks;
event evt_check_store_time;
event evt_check_timing;

// bank 0-15 peek into row N
logic [128*8-1:0] bank0_rowN;
logic [128*8-1:0] bank1_rowN;
logic [128*8-1:0] bank2_rowN;
logic [128*8-1:0] bank3_rowN;
logic [128*8-1:0] bank4_rowN;
logic [128*8-1:0] bank5_rowN;
logic [128*8-1:0] bank6_rowN;
logic [128*8-1:0] bank7_rowN;
logic [128*8-1:0] bank8_rowN;
logic [128*8-1:0] bank9_rowN;
logic [128*8-1:0] bankA_rowN;
logic [128*8-1:0] bankB_rowN;
logic [128*8-1:0] bankC_rowN;
logic [128*8-1:0] bankD_rowN;
logic [128*8-1:0] bankE_rowN;
logic [128*8-1:0] bankF_rowN;

integer bankN_active_row [16];      // keep track which row is active in each bank
integer bankN_AP [16];              // keep track that Auto-Precharge has been issued
integer iRowAddr;

initial begin
    iRowAddr = 0;
    for (int i=0; i<16; i++) begin
        bankN_active_row[i] = -1;
        bankN_AP[i] = 0;
    end
end

// For debug - force iRowAddr to peek in the the array
always @(iRowAddr) begin
    bank0_rowN = bank_mem[0][iRowAddr];
    bank1_rowN = bank_mem[1][iRowAddr];
    bank2_rowN = bank_mem[2][iRowAddr];
    bank3_rowN = bank_mem[3][iRowAddr];
    bank4_rowN = bank_mem[4][iRowAddr];
    bank5_rowN = bank_mem[5][iRowAddr];
    bank6_rowN = bank_mem[6][iRowAddr];
    bank7_rowN = bank_mem[7][iRowAddr];
    bank8_rowN = bank_mem[8][iRowAddr];
    bank9_rowN = bank_mem[9][iRowAddr];
    bankA_rowN = bank_mem[10][iRowAddr];
    bankB_rowN = bank_mem[11][iRowAddr];
    bankC_rowN = bank_mem[12][iRowAddr];
    bankD_rowN = bank_mem[13][iRowAddr];
    bankE_rowN = bank_mem[14][iRowAddr];
    bankF_rowN = bank_mem[15][iRowAddr];
end

//
// ODT Termination state parameters
//
parameter HIGHZ   = 2'b00;
parameter RTTNOM  = 2'b01;
parameter RTTPARK = 2'b10;
parameter RTTWR   = 2'b11;
reg         rtt_park_en;    // RTT(PARK) Enable
reg         rtt_nom_en;     // RTT(NOM) Enable
reg         rtt_wr_en;      // RTT(WR) Enable
reg [1:0]   rtt_state;      // ODT Termination state
reg [2:0]   rtt;            // termination resistance
reg [2:0]   rtt_nom_value;
reg [2:0]   rtt_park_value;
reg [2:0]   rtt_wr_value;
reg         odt_static;
parameter TERM_HIGHZ   = 3'b000;
parameter TERM_60_OHM  = 3'b001;
parameter TERM_120_OHM = 3'b010;
parameter TERM_40_OHM  = 3'b011;
parameter TERM_240_OHM = 3'b100;
parameter TERM_48_OHM  = 3'b101;
parameter TERM_80_OHM  = 3'b110;
parameter TERM_34_OHM  = 3'b111;

//
// registers
//
reg [13:0]          mr[0:6];            // the 7 mode registers
reg [13:0]          wr_lvl_mr1;         // the 7 mode registers
reg [7:0]           mpr0[0:3];          // MPR Page 0
reg [7:0]           mpr_rd_mux;         // MPR Read Mux
reg                 pre_cke;            // previous cke state
reg [ADDR_BITS-1:0] wr_ra[0:`BANKS-1];  // read row address
reg [ADDR_BITS-1:0] rd_ra[0:`BANKS-1];  // write row address

reg [4:0]           cmd;                // DDR4 command
reg [4:0]           prev_cmd;           // DDR4 command
reg [4:0]           rl;                 // read latency from MR
reg [4:0]           wl;                 // write latency from MR
reg [MAX_CWL:0]     wr_shifter;         // write latency shifter
reg [MAX_CL:0]      rd_shifter;         // read latency shifter
reg [MAX_CWL-2:0]   odt_wr_shifter;     // ODT write latency shifter
reg [MAX_CL-2:0]    odt_rd_shifter;     // ODT read latency shifter
reg                 write;              // write in progress
reg                 read;               // write in progress
reg                 read_pre;           // read preamble
reg                 dqs_in_en;          // DQS input qualifier 
reg                 dq_in_en;           // DQ input qualifier 
reg                 dqs_out_en;         // DQS output generator 
reg                 dq_out_en;          // DQS output generator 
reg [BC_BITS-1:0]   cmd_wr_bc;          // command write burst value
reg [BC_BITS-1:0]   wr_bc;              // write burst value
reg [BC_BITS-1:0]   cmd_rd_bc;          // command read burst value
reg [BC_BITS-1:0]   rd_bc;              // read burst value
reg                 power_down ;        // power down active
reg                 power_down_q1;      // power down active
reg                 power_down_q2;      // power down active
reg                 self_ref;           // self refresh active
reg [AP_BITS-1:0]   wr_ap;              // write auto precharge value
reg [AP_BITS-1:0]   rd_ap;              // read auto precharge value
reg [AP_BITS+BA_BITS+BG_BITS+CA_BITS+BC_BITS+ADDR_BITS+CA_BITS-1:0]  wr_fifo[4:0];  // write fifo to store write data
reg [AP_BITS+BA_BITS+BG_BITS+CA_BITS+BC_BITS+ADDR_BITS-1:0]          rd_fifo[4:0];  // write fifo to store write data
reg [BG_BITS-1:0]   wr_bg;
reg [BG_BITS-1:0]   rd_bg;
reg [BA_BITS-1:0]   wr_ba;
reg [BA_BITS-1:0]   rd_ba;
reg [CA_BITS-1:0]   cmd_wr_ca;
reg [CA_BITS-1:0]   wr_ca, wr_ca_l, wr_ca_u;
reg [CA_BITS-1:0]   rd_ca;
reg [CA_BITS-1:0]   cmd_rd_ca;
reg [RFF_BITS-1:0]  wr_data;
reg [ADDR_BITS+CA_BITS-1:0] wr_addr;
reg [RFF_BITS-1:0]          rd_data;
reg [ADDR_BITS+CA_BITS-1:0] rd_addr;
reg [ADDR_BITS+CA_BITS-1:0] wr_rd_addr;
reg [ADDR_BITS-1:0]         rd_row;
reg                 rd_ca_en;
reg                 rd_is_bc4;          // current read is BC4
reg                 bc4_a2;             // state of a2 pin during BC4 cmd
reg [2:0]           wr_adr_mask;
reg                 dll_en;
reg                 dll_reset;
reg                 dll_locked;
reg [31:0]          dll_lock_cnt;
reg                 zqcl;               // ZQCL state
reg [9:0]           zqcl_cnt;
reg                 zqcs;               // ZQCS state
reg [8:0]           zqcs_cnt;
reg                 zq_odt_disable;
reg                 odt_en;
reg                 odt_off;
reg [BG_BITS-1:0]   pre_bg;
reg [BA_BITS-1:0]   pre_ba;
integer             mpxlh_cnt;
integer             xmp_cnt;
integer             xmp_dll_cnt;

reg [4:0]           dodtlon;
reg [4:0]           dodtloff;
reg                 rodtl_on;
reg [4:0]           rodtlon;
reg [2:0]           rodtlon_val;
reg [4:0]           rodtloff_calc;
reg [4:0]           odtlcnw_calc;
reg [4:0]           odtlcwn;
reg [2:0]           odtlcwn_val;
reg                 dodt_on;
reg                 dodt_wr_on;
reg                 pre_odt;

wire                write_leveling_mode;    // Write-Leveling Mode (0=disable, 1=enable)
wire                rd_pre_train_mode;      // Read Preamble Training Mode (0=disable, 1=enable)
wire                q_off;                  // Output Buffer Enable(=0), Disable(=1)
wire                mpsm;                   // Maximum Power Savings Mode
wire                odt_int;                // ODT internal logic level
wire                ct_mode;                // connectivity test mode
wire                mt0, mt1, mt2, mt3, mt4, mt5, mt6, mt7, mt8, mt9;
wire                ck_diff_check;
reg                 internal_vref_mon;      // internal vref monitoring


integer wr_fifo_rd_addr;
integer wr_fifo_wr_addr;
integer rd_fifo_rd_addr;
integer rd_fifo_wr_addr;
integer i,j,k;
integer pre_cnt[16];
integer ck_t_cnt;
integer wr_ck_t_cnt;
integer rd_ck_t_cnt;
integer sre_cnt;
logic   sre_pulse;
logic   txp_pulse;
logic   mpsm_entry_pulse;
integer srx_cnt;
integer sr_cnt;
integer cpded_cnt;
integer mped_cnt;               // command path disable delay
integer wr_mpr_cnt;

//command time stamps and counters
real    tck_avg;
real    tck_avg_hist [64];
time    tck_value[TDLLK-1:0];
time    tch_value[TDLLK:0];
time    tcl_value[TDLLK:0];
time    ck_t_width;
time    ck_c_width;
time    tch_width;
time    tcl_width;
real    tch_avg;
real    tcl_avg;
time    ts_ck_t;
time    ts_ck_c;
real    tjit_per;
integer tjit_cc;
real    terr_nper;
reg     txpr_update_en;

real    tjit_ch_rtime;
real    tck_duty_cycle;

time    ts_act_bank[`BANKS];        //timestamp for TFAW check
time    ts_act;
time    ts_act_group[2**BG_BITS];
time    ts_write_group[2**BG_BITS];
time    ts_read_group[2**BG_BITS];
time    ts_write[`BANKS];
time    ts_read[`BANKS];
time    ts_mrs;
time    ts_refresh;
time    ts_zq;
time    ts_pwr_down;
time    ts_pre_bank[`BANKS];
time    ts_pre;
time    ts_self_ref;
time    ts_store;
time    ts_store_bank[`BANKS];
time    ts_txpr;
time    ts_pw_reset;

integer tck_mrs;
integer tck_last_write;
integer tck_read[`BANKS];
integer tck_last_read;
integer tck_refresh;
integer tck_zq;
integer tck_pwr_down;
integer tck_pre;
integer tck_act;
integer tck_act_group[2**BG_BITS];
integer tck_read_group[2**BG_BITS];
integer tck_write_group[2**BG_BITS];
integer tck_write_end_group[2**BG_BITS];
integer tck_self_ref;
integer tck_store;
integer tck_txpr;
integer nWR;                        // WRITE Recovery in clocks

parameter SAME_BANK  = 2'd0;        // same bank, same group
parameter DIFF_BANK  = 2'd1;        // diff bank, same group
parameter DIFF_GROUP = 2'd2;        // diff bank, diff group

integer ck_freq_change;
time    ts_freq_change;
time    ts_sre;
time    ts_srx;

integer speed_bin;
integer cycle;
integer info_count;
integer warning_count;
integer error_count;
event   evt_info, evt_warning, evt_error;
event   evt_mr_write;

typedef enum {INFO, WARN, ERR} e_msg;
typedef enum logic [4:0] {CL_9,  CL_10, CL_11, CL_12, CL_13, CL_14, CL_15, CL_16, CL_18,
                          CL_20, CL_22, CL_24, CL_23, CL_17, CL_19, CL_21, CL_25, CL_26,
                          CL_27, CL_28, CL_29, CL_30, CL_31, CL_32} e_cl;
typedef enum logic [1:0] {BL_8, BL_OTF, BL_BC4, BL_RSVD} e_bl;
typedef enum logic [2:0] {DISABLED, RZQ_4, RZQ_2, RZQ_6, RZQ_1, RZQ_5, RZQ_3, RZQ_7} e_rtt;
typedef enum logic [1:0] {OD_RZQ_7, OD_RZQ_5, OD_RZQ_6, OD_RSVD} e_od;
typedef enum logic [2:0] {WR_OFF, WR_RZQ_2, WR_RZQ_1, WR_HI_Z, WR_RZQ_3, WR_RSVD1, WR_RSVD2, WR_RSVD3} e_rtt_wr;
typedef enum logic [2:0] {CWL_9, CWL_10, CWL_11, CWL12, CWL_14, CWL_16, CWL_18, CWL_20} e_cwl;
typedef enum logic [1:0] {BANKS_2, BANKS_4, BANKS_8, BANKS_16} e_store_stgr;
typedef enum logic       {NORMAL, MPR} e_mpr;
typedef enum logic [1:0] {PAGE_0, PAGE_1, PAGE_2, PAGE_3} e_mpr_page;
typedef enum {WR_NONE, WR_BL8OTF, WR_BL8MRS, WR_BC4OTF, WR_BC4MRS, WRA_BL8OTF, WRA_BL8MRS, WRA_BC4OTF, WRA_BC4MRS} e_write;
typedef enum {DEC_RESET, DEC_MRS, DEC_PRE, DEC_PREA, DEC_ACT, DEC_REF, DEC_SRE, DEC_SRX, DEC_STO, DEC_STOA, 
              DEC_WR, DEC_RD, DEC_ZQCL, DEC_ZQCS, DEC_NOP, DEC_DES} e_decode;
typedef enum {STO_NONE, STO_ONE, STO_ALL} e_store_cmd;
typedef enum {BANK_CLOSED, BANK_OPEN, BANK_DIRTY} e_bank_status;


e_cl          mr_cl;
e_cwl         mr_cwl;
e_bl          mr_bl;
e_rtt         mr_rtt_park, mr_rtt_nom;
e_rtt_wr      mr_rtt_wr;
e_od          mr_od;
e_store_stgr  mr_store;
e_mpr         mr_mpr;
e_mpr_page    mr_mpr_page;
e_write       last_write;
e_store_cmd   last_store;
e_bank_status bank_status [16];

always_comb begin : MR_DECODE
    mr_cl       = e_cl'({mr[0][12],mr[0][6:4],mr[0][2]});
    mr_bl       = e_bl'(mr[0][1:0]);
    mr_rtt_nom  = e_rtt'(mr[1][10:8]);
    mr_rtt_park = e_rtt'(mr[5][8:6]);
    mr_od       = e_od'(mr[1][2:1]);
    mr_rtt_wr   = e_rtt_wr'(mr[2][11:9]);
    mr_cwl      = e_cwl'(mr[2][5:3]);
    mr_store    = e_store_stgr'(mr[3][7:6]);
    mr_mpr      = e_mpr'(mr[3][2]);
    mr_mpr_page = e_mpr_page'(mr[3][1:0]);
end



initial begin
    ck_t_cnt      = 0;
    sre_cnt       = 0;
    sre_pulse     = 0;
    txp_pulse     = 0;
    srx_cnt       = 0;
    sr_cnt        = 0;
    info_count    = 0;
    warning_count = 0;
    error_count   = 0;
    cpded_cnt     = 0;
    mped_cnt      = 0;
    wr_mpr_cnt    = 0;
    mpsm_entry_pulse = 0;
    ts_sre = 0;
    ts_srx = 0;
    ts_zq  = 0;
    tck_zq = 0;
    ts_pw_reset = 0;
    last_write  = WR_NONE;
    ts_store    = 0;
    for (int i=0; i<`BANKS; i++) ts_act_bank[i]   = 0;
    for (int j=0; j<16; j++    ) bank_status[j]   = BANK_CLOSED;
    for (int k=0; k<`BANKS; k++) ts_store_bank[k] = 0; 
    // -----------------
    `ifdef DLL_OFF
        speed_bin = 400;
        log_msg (INFO, "Using DLL_OFF speed bin", -3);
    `elsif BIN800
        speed_bin = 800;
        log_msg (INFO, "Using 800 speed bin", -3);
    `elsif BIN1333
        speed_bin = 1333;
        log_msg (INFO, "Using 1333 speed bin", -3);
    `elsif BIN1600
        speed_bin = 1600;
        log_msg (INFO, "Using 1600 speed bin", -3);
    `elsif BIN1866
        speed_bin = 1866;
        log_msg (ERR, "the 1866 speed bin is not supported!", -1);
    `elsif BIN2133
        speed_bin = 2133;
        log_msg (ERR, "the 2133 speed bin is not supported!", -1);
    `elsif BIN2400
        speed_bin = 2400;
        log_msg (ERR, "the 2400 speed bin is not supported!", -1);
    `endif
end



//
// ODT pin Internal logic value
//
// In Power-Down mode when the ODT Buffer is disabled, the odt_off signal is
// utilized to switch ODT from RTT_NOM to RTT_PARK/HIGHZ
//
assign odt_int             = (mr[5][5] & odt_off) ? 0 : odt;
assign write_leveling_mode = mr[1][7];      // Write-Leveling Mode
assign rd_pre_train_mode   = mr[4][10];     // Read Preamble Training
assign mpsm                = mr[4][1];      // Maximum Power-Saving Mode
wire nomem                 = mr[0][13];
wire nowmem                = mr[1][13];


// Connectivity Test Mode
//
assign ck_diff_check = ten & (ck_t ~^ ck_c);

assign ct_mode = ten & ~cs_n; 
assign mt0 = a[1]^a[6]^par;
assign mt1 = a[8]^a[9]^alert_n;
assign mt2 = a[2]^a[5]^a[13];
assign mt3 = a[0]^a[7]^a[11];
assign mt4 = ck_c^odt^cas_n_a15;
assign mt5 = cke^ras_n^a[10];
assign mt6 = act_n^a[4]^ba[1];
`ifdef X8
assign mt7 = bg[1]^dm_n_tdqs_t^ck_t;
`else
assign mt7 = dmu_n^dml_n^ck_t;
`endif
assign mt8 = we_n_a14^a[12]^ba[0];
assign mt9 = bg[0]^a[3]^(reset_n&ten);
//
// TDQS I/O Buffer
//
wire dm_n_tdqs_t, tdqs_c, dm_n_tdqs_t_in;
assign tdqs_c = 1'bz; 
assign dm_n_tdqs_t = 1'bz;
assign dm_n_tdqs_t_in = ~mr[1][11] ? dm_n_tdqs_t : 1'b1;
//
// DQS I/O buf
//
// Output Buffer Enable/Disable
assign q_off = mr[1][12] | power_down  | mpsm;

initial begin
    internal_vref_mon = 1'b0;
end

always @(posedge ck_diff_check) begin
    log_msg(ERR, "ck_t and ck_c must be complement of each other while test mode is enabled", -1);
end

always @(posedge ck_t) begin
    internal_vref_mon = mr[4][4];
end

`ifdef X8
// x8 strobes
wire dqs_t_in, dqs_c_in;
reg  wl_ck;

// *** TODO *** do we still want to support this feature
//              need testbench to create this condition to check failure mode
//
// The dqs internal clock needs to occur after ck. Since write leveling
// allows for dqs to be slightly before ck, at the posedge of ck, check if 
// check if dqs has transitioned yet. If it has dqs leads ck, and ck_t should
// be used for internal sampling.
reg  dqs_leads_ck;

always @(posedge ck_t) begin
    if ( dqs_t == 1'b1 ) begin
        dqs_leads_ck = 1'b1;
   end else begin
        dqs_leads_ck = 1'b0;
   end
end

assign dqs_t = ten ? ~cs_n ? mt8 : 1'bz : dqs_out_en & ~q_off ? read_pre & rd_pre_train_mode ? 1'b0 : ck_t : rd_pre_train_mode ? 1'b0 : 1'bz;
assign dqs_c = ten ? ~cs_n ? mt9 : 1'bz : dqs_out_en & ~q_off ? read_pre & rd_pre_train_mode ? 1'b1 : ck_c : rd_pre_train_mode ? 1'b1 : 1'bz;

// De-glitch and convert from differential to Single-Ended
assign #1.0 dqs_t_in = (dqs_in_en | write_leveling_mode) &&  dqs_t===1'b1 && dqs_c===1'b0 ? 1'b1 : 1'b0;
assign #1.0 dqs_c_in = (dqs_in_en | write_leveling_mode) &&  dqs_t===1'b0 && dqs_c===1'b1 ? 1'b1 : 1'b0;
`else
// x16 strobes
wire  dqsu_t_in,   dqsu_c_in,   dqsu_hi,     dqsu_lo;
wire  dqsl_t_in,   dqsl_c_in,   dqsl_hi,     dqsl_lo;
event dqsu_hi_evt, dqsu_lo_evt, dqsl_hi_evt, dqsl_lo_evt;

reg  wl_l_ck, wl_u_ck;
assign dqsu_t = ten ? ~cs_n ?  mt9 : 1'bz : dqs_out_en & ~q_off ? read_pre & rd_pre_train_mode ? 1'b0 : ck_t : rd_pre_train_mode ? 1'b0 : 1'bz;
assign dqsu_c = ten ? ~cs_n ? ~mt9 : 1'bz : dqs_out_en & ~q_off ? read_pre & rd_pre_train_mode ? 1'b1 : ck_c : rd_pre_train_mode ? 1'b1 : 1'bz;
assign dqsl_t = ten ? ~cs_n ?  mt8 : 1'bz : dqs_out_en & ~q_off ? read_pre & rd_pre_train_mode ? 1'b0 : ck_t : rd_pre_train_mode ? 1'b0 : 1'bz;
assign dqsl_c = ten ? ~cs_n ? ~mt8 : 1'bz : dqs_out_en & ~q_off ? read_pre & rd_pre_train_mode ? 1'b1 : ck_c : rd_pre_train_mode ? 1'b1 : 1'bz;

// De-glitch and convert from differential to Single-Ended
assign #1.0 dqsu_hi = (dqs_in_en | write_leveling_mode) & dqsu_t===1'b1 & dqsu_c===1'b0 ? 1'b1 : 1'b0;
assign #1.0 dqsu_lo = (dqs_in_en | write_leveling_mode) & dqsu_t===1'b0 & dqsu_c===1'b1 ? 1'b1 : 1'b0;
assign #1.0 dqsl_hi = (dqs_in_en | write_leveling_mode) & dqsl_t===1'b1 & dqsl_c===1'b0 ? 1'b1 : 1'b0;
assign #1.0 dqsl_lo = (dqs_in_en | write_leveling_mode) & dqsl_t===1'b0 & dqsl_c===1'b1 ? 1'b1 : 1'b0;

`endif

//
// DQ I/O buf
//
wire [DQ_BITS-1:0] dq;
wire [DQ_BITS-1:0] dq_in;
reg  [DQ_BITS-1:0] dq_out;


`ifdef X8
assign dq = ten     ? ~cs_n ? { mt7, mt6, mt5, mt4, mt3, mt2, mt1, mt0 } : 8'hz
                    : (q_off)               ? 8'hz
                    : (write_leveling_mode) ? {8{wl_ck}}
                    : (internal_vref_mon)   ? {6'hz,2'b11}
                    : (rd_pre_train_mode)   ? (dq_out_en) ? dq_out : 8'hff
                    : (dq_out_en)           ? dq_out : 8'hz;
`else
assign dq[7:0] = ten ? ~cs_n ? { mt7, mt6, mt5, mt4, mt3, mt2, mt1, mt0 } : 8'hz
                     : (q_off)               ? 'hz
                     : (write_leveling_mode) ? {8{wl_l_ck}}
                     : (internal_vref_mon)   ? {6'hz,2'b11}
                     : (rd_pre_train_mode)   ? (dq_out_en) ? dq_out[7:0]  : 8'hff
                     : (dq_out_en)           ? dq_out[7:0] : 8'hz;
                     
assign dq[15:8] = ten ? ~cs_n ? { ~mt7, ~mt6, ~mt5, ~mt4, ~mt3, ~mt2, ~mt1, ~mt0 } : 8'hz
                      : (q_off)               ? 8'hz
                      : (write_leveling_mode) ? {8{wl_u_ck}}
                      : (internal_vref_mon)   ? 8'hz
                      : (rd_pre_train_mode)   ? (dq_out_en) ? dq_out[15:8] : 8'hff
                      : (dq_out_en)           ? dq_out[15:8] : 8'hz;

`endif
assign dq_in = (dq_in_en) ? dq : 'hx;

always @(negedge reset_n) begin
    ts_pw_reset = $time;
end

always @(posedge reset_n) begin
    if ( ts_pw_reset==0 && ($time-ts_pw_reset) < TPW_RESET_L ) begin
        // reset after power up
        log_msg (ERR, "tPW_RESET_L Violated, Reset deasserted", 0);
    end else if ( ts_pw_reset>0 && ($time-ts_pw_reset) < TPW_RESET_S ) begin
        log_msg (ERR, "tPW_RESET_S Violated, Reset deasserted", 0);
    end
end

function integer get_tst_time; begin : TSK_GET_TST_CNT
    case (mr_store)
        BANKS_2:  return TST2;
        BANKS_4:  return TST4;
        BANKS_8:  return TST8;
        BANKS_16: return TST;
        default:  return TST2;
    endcase
end endfunction


function integer get_tst_cnt; begin : TSK_GET_TST_CNT
    return $ceil(get_tst_time()/tck_avg );
end endfunction


task check_tcksre;
    integer exp_sr_cnt;  // expected sr count
    begin : TSK_CHECK_TCKSRE
    exp_sr_cnt = max($ceil(TCKSRE/tck_avg), TCKSRE_TCK);
    if ( sr_cnt < exp_sr_cnt ) log_msg (ERR, "TCKSRE (min) Violated - Self Refresh", -1);
end endtask

always @(posedge self_ref) begin
    if ( reset_n==1'b1 ) begin
        ts_sre    = $time;
        sre_cnt   = ck_t_cnt;
        sr_cnt    = 1;
        cpded_cnt = TCPDED;
        check_active_banks(DEC_SRE);
        check_store_time;
        fork
            begin
                sre_pulse = 1;
                #(tck_avg) dll_locked  = 0;
            end
            begin
                #(max(TCKSRE, TCKSRE_TCK*tck_avg)) sre_pulse = 0;
                check_tcksre();
            end
        join_any
    end
end

always @(negedge self_ref) begin
    if ( reset_n==1'b1 ) begin
        ts_srx  = $time;
        srx_cnt = ck_t_cnt;
        if ( (ts_srx - ts_sre) < get_tst_time() ) log_msg (ERR, "tCKESR (min) Violated on Exit Self Refresh", 0);
        for (int i=0; i < ( max(TCKSRX_TCK, max(1, $ceil(TCKSRX/tck_avg)))); i++) begin
            if ( $rtoi(tck_avg_hist[i]) != $rtoi(tck_avg) ) begin
                log_msg (ERR, "tCKSRX (min) Violated on Exit Self Refresh", 0);
                break;  // got error - exit loop
            end
        end
    end
    sr_cnt = 0;
end


always @(negedge reset_n) begin : RST_CHECKS
    log_msg (INFO, "RESET_N asserted", 2);
    check_active_banks(DEC_RESET);
    check_store_time;
    if ( (ck_t_cnt - tck_mrs) < TMOD_TCK )
        log_msg (ERR, ("tMOD (min) Violated when asserted reset_n"),-1);
end

function [`BANKS-1:0] active_banks; begin : FNC_ACTIVE_BANKS
    // banks that were open and not precharged
    for (int i=0; i<`BANKS; i++) active_banks[i] = bank_status[i]==BANK_OPEN ? 1'b1 : 1'b0;
    `ifdef X16
    // banks 8-16 are virtual, user does not get direct access
    return active_banks & 16'h00FF;
    `else
    return active_banks;
    `endif
end endfunction

function [`BANKS-1:0] dirty_banks; begin : FNC_DIRTY_BANKS
    // banks that were open precharged but not stored
    for (int i=0; i<16; i++) dirty_banks[i] = bank_status[i]==BANK_DIRTY ? 1'b1 : 1'b0;
end endfunction


task check_active_banks (input e_decode decode);
    begin : TSK_CHK_ACT_BANKS
        -> evt_check_active_banks;
        for(int bank=0; bank<`BANKS; bank++) begin
            // for x16 only give user message for banks only for banks 0-7
            if ( bank_status[bank]==BANK_OPEN ) begin
                if ( decode==DEC_RESET ) log_msg (ERR, $sformatf("Reset asserted with Bank still active [%2d] (possible loss of data)", bank), -1);
                if ( decode==DEC_SRE )   log_msg (ERR, $sformatf("Entered Self Refresh with Bank still active [%2d] (possible loss of data)", bank), -1);
                if ( decode==DEC_REF )   log_msg (ERR, $sformatf("Refresh Command (STO) with Bank still active [%2d] (possible loss of data)", bank), -1);
            end
            if ( decode==DEC_RESET && bank_status[bank]==BANK_DIRTY ) begin
                log_msg (ERR, $sformatf("Reset asserted with Bank still dirty [%2d] (possible loss of data) - need Store Command prior to Reset", bank), -1);
            end
            if ( decode==DEC_RESET && ($time - ts_pre_bank[bank] < TRP-TJIT_PER_TOT)) begin
                log_msg (ERR, "Reset asserted within tRP(min) time (possible loss of data)",-1);
            end
         end
    end
endtask

task check_store_time; begin : TSK_CHK_STORE
    -> evt_check_store_time;
    if ( last_store == STO_ONE ) begin
        //if ( ($time - ts_store) < TST  ) log_msg (ERR, "tST (min) Violated, previous Store command did not finish", -1);
        for (int i=0; i<`BANKS; i++)
            // check Store for the bank that command was issued for
            if ( reset_n==1'b1 && cs_n==1'b0 && i=={bg,ba} && ($time - ts_store_bank[i]) < TST  ) begin
                log_msg (ERR,  "tST (min) Violated, previous Store command did not finish", -1);
            end else begin
                if ( reset_n==1'b0 && ($time - ts_store_bank[i]) < TST  ) begin
                    log_msg (ERR,  "tST (min) Violated, previous Store command did not finish", -1);
                end else begin
                    log_msg (INFO, $sformatf("Bank=%02d Succesfuly Stored",i), -1);
                end
            end
    end else if ( last_store == STO_ALL ) begin
        case (mr_store)
            BANKS_2:    if ( ($time - ts_store) < TST2 ) log_msg (ERR,  "tST (min) Violated, previous Store command did not finish", -1);
                        else if ( !reset_n )             log_msg (INFO, "Bank=All Succesfully Stored", -1);
            BANKS_4:    if ( ($time - ts_store) < TST4 ) log_msg (ERR,  "tST (min) Violated, previous Store command did not finish", -1);
                        else if ( !reset_n )             log_msg (INFO, "Bank=All Succesfully Stored", -1);
            BANKS_8:    if ( ($time - ts_store) < TST8 ) log_msg (ERR,  "tST (min) Violated, previous Store command did not finish", -1);
                        else if ( !reset_n )             log_msg (INFO, "Bank=All Succesfully Stored", -1);
            BANKS_16:   if ( ($time - ts_store) < TST  ) log_msg (ERR,  "tST (min) Violated, previous Store command did not finish", -1);
                        else if ( !reset_n )             log_msg (INFO, "Bank=All Succesfully Stored", -1);
        endcase
    end
end endtask

//
// main loop
//
always @(negedge reset_n or posedge ck_t or posedge ck_c or posedge ten) begin : MAIN_LOOP
    if ( !reset_n || ten ) begin
        do_reset;
        pre_cke = 1'b0;
        for(i=0;i<`BANKS;i=i+1) begin
            wr_ra[i] = 15'hx;
            rd_ra[i] = 15'hx;
        end
        read       = '0;
        write      = '0;
        wr_shifter = '0;
        wr_bc      = '0;
        rd_shifter = '0;
        rd_bc      = '0;
        rd_ca_en   = '0;
        dq_in_en   = '0;
        dqs_in_en  = '0;
        dqs_out_en = '0;
        dq_out_en  = '0;
        power_down      = 0;
        power_down_q1   = 0;
        power_down_q2   = 0;
        wr_fifo_rd_addr = 0;
        wr_fifo_wr_addr = 0;
        rd_fifo_rd_addr = 0;
        rd_fifo_wr_addr = 0;
        odt_en      = 1;
        odt_off     = 0;
        mpxlh_cnt   = 0;
        xmp_cnt     = 0;
        xmp_dll_cnt = 0;
        prev_cmd    = 5'b10000;
        cmd         = 5'b10000;
        self_ref    = 0;
        txpr_update_en = 1'b1;
        ts_txpr = 0;
    end else begin
        if (!self_ref && (((ck_t !== 1'b0) && (ck_t !== 1'b1)) || ((ck_c !== 1'b0) && (ck_c !== 1'b1)))) begin
              log_msg (ERR, "CK and CK_N must be in a known state", 0);
            end
        if ( reset_n==1'b1 && !self_ref && dll_locked && (abs(tjit_per) > TJIT_PER_TOT)) begin
            log_msg (ERR, "Clock Frequency Change can only occur in Self Refresh Mode.", 0);
        end
        
        // Clock frequency change during a self refresh or during a pre power down
        if (self_ref || (power_down  && (!(|active_banks() ) ))) begin
            if (ck_t) begin
                tjit_per = $time - ts_ck_t - tck_avg;
                if ( sre_pulse ) sr_cnt = sr_cnt + 1;
            end else begin
                tjit_per = $time - ts_ck_c - tck_avg;
            end
            if (dll_locked && (abs(tjit_per) > TJIT_PER_TOT)) begin
                log_msg (INFO, "The clock frequency has changed and now the DLL must now be reset.", 2);
                ts_freq_change <= $time;
                ck_freq_change <= ck_t_cnt;
                dll_locked = 0;
            end
        end

        // rising edge of ck_t
        if (ck_t) begin
            // Debug - capture command
            if ( !cs_n ) log_msg (INFO, $sformatf("CK_CNT=%08d CMD=%6s ADDR=%04h BG=%h BA=%h", ck_t_cnt, command, a, bg, ba), 5);
            for (int j=0; j<16; j++) if ( bankN_AP[j] > 1 ) bankN_AP[j] = bankN_AP[j] - 1;
            cmd = cmd_dec;
            if (prev_cmd==REF && pre_cke == 0  && cs_n==1'b0 )      log_msg(ERR, "SRE command not followed by Deselect", 0);
            if (pre_cke==1 && cke==0 && (zqcl || zqcs) )            log_msg(ERR, "CKE should not transition low while ZQ calibration is in progress", 0);
            if (cmd_dec!=DES && (zqcl || zqcs) )                    log_msg(ERR, "Only DESelect is allowed while ZQ calibration is in progress", 0);
            if (pre_cke==0 && cke==1 &&  self_ref &&  cmd_dec<NOP ) log_msg(ERR, "Only a DESelect or NOP command is allowed in the clock following CKE active (SRX)", 0);
            if (pre_cke==0 && cke==1 && !self_ref &&  cs_n==0 )     log_msg(ERR, "Only a DESelect command is allowed in the clock following CKE active (PDX)", 0);
            if ( txp_pulse && cs_n==1'b0 )                          log_msg(ERR, "tXP (min) Violated, expecting only DESelect", 0);
            // cs_n is low on rising edge - indicates a command unless:
            //      - Test Mode is not active or
            //      - Maximum Power Saving Mode is active and the tMPED counter has expired or
            //      - Power-Down Mode is active or 
            //      - Self-Refresh Mode is active
            if (!cs_n && ~cke && cmd==MRS) log_msg (ERR, "Received MRS command with CKE in low state", 0);
            if (~cs_n & ~ten & ~(mpsm && mped_cnt==0) & ~power_down  & ~self_ref) begin 
                if ( cmd < NOP ) begin
                    if ($time - ts_txpr < TXPR) log_msg (ERR, $sformatf("tXPR Violated for Command=%s Spec=%dps Measured=%dps", cmd_str[cmd], TXPR, $time - ts_txpr), 0);
                    //Timing Check - sweep through all posible past commands (j), compared to the current command (cmd) for each grouping (k)
                    foreach ( cmd_list[j] ) begin
                        for (k=0; k<=DIFF_GROUP; k=k+1) begin
                          check_timing(k, {bg,ba}, cmd_list[j], cmd);
                        end
                    end
                end
                if ( pre_cke===1'b1 ) do_cmd;             // do the DDR4 command
            end else begin          // cs_n is high on rising edge - self refresh or power down modes
                casex({pre_cke, cke})
                    2'b00: begin    // Normal Operation
                    end    
                    2'b01: begin    // Self Refresh Exit or Powerdown Exit
                        if (cs_n) begin
                            if (mr[3][2]) log_msg (ERR, "Attempt to exit Self-Refresh or Power-Down while in MPR access mode", 0);
                            if (power_down ) begin
                                log_msg (INFO, "Received Power Down Exit command", 2);
                                if (~mr[5][5] & (odt_int !== 0 & odt_int !== 1)) log_msg (ERR, $sformatf("Attempt to exit Power-Down with ODT Input Buffer enabled and an invalid ODT pin[=%b]", odt_int) , 0);
                                if (mr[1][10:8] !== 3'b000 & (odt_int !== 1 & odt_int !== 0)) log_msg (ERR, $sformatf("Attempt to exit Power-Down with RTT(NOM) Enabled and an invalid ODT pin[=%b]", odt_int) , 0);
                                odt_off = 0;
                                power_down  = 1'b0;        // Signal end of Power-Down
                                exit_power_down();
                            end 
                        end
                        if ( self_ref && (cs_n==1 || cmd_dec==NOP) ) begin
                            log_msg (INFO, "Received SRX command", 0);
                            self_ref = 1'b0;    // Signal end of Self-Refresh
                            dll_en   = 1'b1;    // DLL is enabled on self refresh exit 
                            mr[0][8] = 1'b1;    // Issue a DLL reset
                            odt_en   = 1'b1;    // ODT is enabled on self refresh exit
                        end 
                    end    
                    2'b10: begin    // Power Down Entry
                        if (cs_n & ~mpsm) begin     // cs_n must be high and can not be in Maximum Power Saving Mode
                                                                              log_msg (INFO, "Received Power Down Entry command", 2);
                            if (mr[5][5])                                     log_msg (INFO, "ODT Buffer is Not Active", 3);
                            else                                              log_msg (INFO, "Power-Down Mode: ODT Buffer is Active", 3);
                            if (dll_locked)                                   log_msg (INFO, "Power-Down Mode: DLL is Locked", 3);
                            else                                              log_msg (INFO, "Power-Down Mode: DLL is not Locked", 3);
                            if (|active_banks())                              log_msg (INFO, $sformatf("Power-Down Mode: Active[banks=%016b]", active_banks()), 3);
                            else                                              log_msg (INFO, "Power-Down Mode: Precharge", 3);
                            if (mr[3][2])                                     log_msg (ERR, "Attempt to enter Power Down while in MPR access mode", 0);
                            if (~mr[5][5] & (odt_int !== 0 & odt_int !== 1))  log_msg (ERR, $sformatf("Attempt to enter Power Down with ODT Input Buffer enabled and an invalid ODT pin[=%b]", odt_int),0);
                            if (zqcl | zqcs)                                  log_msg (ERR, "Attempt to enter Power Down with a ZQCAL operation in progress", 0);
                            if (|rd_shifter | read)                           log_msg (ERR, "Attempt to enter Power Down with a Read operation in progress", 0);
                            if (|wr_shifter | write)                          log_msg (ERR, "Attempt to enter Power Down with a Write operation in progress", 0);
                            if ( (ck_t_cnt - tck_mrs) < TMRSPDEN_TCK )        log_msg (ERR, "tMRSPDEN (min) Violated, MRS to PDE", 0);
                            if ( (ck_t_cnt - tck_last_read)  < (TRDPDEN_TCK + rl)) log_msg (ERR, "tRDPDEN (min) Violated, Read to PDE", 0);
                            case ( last_write )
                                WR_BL8OTF:  if ( (ck_t_cnt - tck_last_write) < (TWRPDEN_TCK    + wl + TWR/tck_avg)) log_msg (ERR, "tWRPDEN (min) Violated, Write BL8OTF to PDE", 0);
                                WR_BL8MRS:  if ( (ck_t_cnt - tck_last_write) < (TWRPDEN_TCK    + wl + TWR/tck_avg)) log_msg (ERR, "tWRPDEN (min) Violated, Write BL8MRS to PDE", 0);
                                WR_BC4OTF:  if ( (ck_t_cnt - tck_last_write) < (TWRPDEN_TCK    + wl + TWR/tck_avg)) log_msg (ERR, "tWRPDEN (min) Violated, Write BC4OTF to PDE", 0);
                                WR_BC4MRS:  if ( (ck_t_cnt - tck_last_write) < (TWRPBC4DEN_TCK + wl + TWR/tck_avg)) log_msg (ERR, "tWRPBC4DEN (min) Violated, Write to PDE", 0);
                                WRA_BL8OTF: if ( (ck_t_cnt - tck_last_write) < (TWRAPDEN_TCK   + wl + nWR)) log_msg (ERR, "tWRAPDEN (min) Violated, Write BL8OTF to PDE", 0);
                                WRA_BL8MRS: if ( (ck_t_cnt - tck_last_write) < (TWRAPDEN_TCK   + wl + nWR)) log_msg (ERR, "tWRAPDEN (min) Violated, Write BL8MRS to PDE", 0);
                                WRA_BC4OTF: if ( (ck_t_cnt - tck_last_write) < (TWRAPDEN_TCK   + wl + nWR)) log_msg (ERR, "tWRAPDEN (min) Violated, Write BC4OTF to PDE", 0);
                                WRA_BC4MRS: if ( (ck_t_cnt - tck_last_write) < (TWRPBC4DEN_TCK + wl + nWR)) log_msg (ERR, "tWRPBC4DEN (min) Violated, Write BC4MRS to PDE", 0);
                            endcase
                            power_down  = 1'b1;
                            cpded_cnt   = TCPDED;
                            odt_off     = 1;
                        end
                    end    
                    2'b11: begin    // Normal Operation
                    end    
                endcase
            end
            // 
            // Write Logic
            // 
            if (wr_shifter[0] == 1'b1) begin
                // it's time to grab the write data from the Write FIFO
                {wr_ap, wr_bc, wr_bg, wr_ba, wr_ca, wr_addr} = wr_fifo[wr_fifo_rd_addr];
                wr_ca_l = wr_ca;
                wr_ca_u = wr_ca;
                // if performing a MPR access or the ST-MRAM is in PDA mode, don't read data from external file
                if (~mr[3][2] & ~mr[3][4]) begin
                    // CPM load previously written data so we can replace with new data, maybe only sparse columns will be written
                    //     but - the activate should have already loaded data into the cache
                end
            end
            
            if (write) begin
                if ( wr_bc == 4'b0 && ~wr_shifter[0]) begin
                    write = 1'b0;
                    dqs_in_en = 1'b0;
                    // auto precharge (no AP during MRS write)
                    if (wr_ap && !mr[3][2])  log_msg (INFO, $sformatf("Auto Precharge [WRITE] BG=%d BA=%d RA=%0h", wr_bg, wr_ba, wr_ra[{wr_bg, wr_ba}]), 5);
                end
                dq_in_en = dqs_in_en;     // 1tCK preamble
                if (dq_in_en) wr_bc = wr_bc - 1'b1;
            end
            if (wr_shifter[1] == 1'b1) begin
                dqs_in_en = 1'b1;
            end     
            if (|wr_shifter) begin
                if (wr_shifter[1]) write = 1'b1;
                    wr_shifter = wr_shifter >> 1;
            end
            //
            // Read Logic
            // 
            if (read && dq_out_en) begin
                rd_ca = rd_ca + 1'b1;
            end
                
            if (rd_shifter[0] == 1) begin
                {rd_ap, rd_bc, rd_bg, rd_ba, rd_ca, rd_row} = rd_fifo[rd_fifo_rd_addr];
                // If this is a BC4 read, rd_bc will be equal to 4 and rd_ca[2] will indicate the burst order
                rd_is_bc4 = (rd_bc == 4);
                bc4_a2    = (rd_ca[2]);
                rd_addr   = {rd_row, rd_ca}/BL_MAX;
                if (~mr[3][2]) begin        // this is a memory read, not an MPR read
                    // always read BL-8 aligned, the read mux will pull out the correct date if BC4 or starting at not BL-8 address
                    rd_data = get_cache_data({rd_bg,rd_ba}, {rd_ca[6:3],3'b000});
                    log_msg (INFO, $sformatf("RD: Row: %h Bank: %02d Col: %h Data: %h", rd_row, {rd_bg, rd_ba}, {rd_addr,3'b0}, rd_data), 5);
                end
                if  (rd_fifo_rd_addr == 4) rd_fifo_rd_addr = 0;
                else rd_fifo_rd_addr = rd_fifo_rd_addr + 1;
            end

            if (read) begin
                dq_out_en = dqs_out_en;        // 1tCK preamble
                if (rd_bc==0) begin
                    read       = 0;
                    rd_ca_en   = 0;
                    dqs_out_en = 0;
                    dq_out_en  = 0;
                    if (rd_ap) begin
                        log_msg (INFO, $sformatf("Auto Precharge [READ] Bank=%02d RA=%04h", {rd_bg, rd_ba}, rd_ra[{rd_bg, rd_ba}]),5);
                        //write_cache_to_mem ( {rd_bg, rd_ba} );
                        //bankN_AP[{rd_bg, rd_ba}] = 0;
                        //`ifdef X16
                        //bankN_AP[{1'b1,rd_bg, rd_ba}] = 0;
                        //`endif
                    end
                end
                rd_bc = rd_bc - 1'b1;
            end

            if (rd_shifter[1] == 1'b1) begin
                dqs_out_en = 1'b1;
            end

            if (|rd_shifter) begin
                if (rd_shifter[1]) read = 1'b1;
                rd_shifter = rd_shifter >> 1;
            end
            //
            read_pre <= rd_shifter[1] & ~read;
            //
            // keep track of the last N periods
            for (int i=64; i>0; i--) tck_avg_hist[i] = tck_avg_hist[i-1];
            tck_avg_hist[0] = $time - ts_ck_t;
            // clock period and jitter checks
            if (!self_ref && ts_ck_t ) begin
                tjit_cc     = $time - ts_ck_t - ck_t_width;
                ck_t_width  = $time - ts_ck_t;
                tck_avg     = tck_avg - tck_value[ck_t_cnt%TDLLK]/$itor(TDLLK);
                tck_avg     = tck_avg + ck_t_width/$itor(TDLLK);
                // keep track of the last N TCK periods
                tck_value[ck_t_cnt%TDLLK] = ck_t_width;
                tjit_per = ck_t_width - tck_avg;
                if (dll_locked) begin
                    // check the accumulated clock error
                    terr_nper = 0;
                    for (cycle=0; cycle<12; cycle=cycle+1) begin
                        terr_nper = terr_nper + tck_value[cycle] - tck_avg;
                        terr_nper = abs(terr_nper);
                        case (cycle)
                            0  : ;//same cycle, do nothing
                            1  : if ( (terr_nper - TERR_2PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (2per) Violated by %f ps.", terr_nper - TERR_2PER),6);
                            2  : if ( (terr_nper - TERR_3PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (3per) Violated by %f ps.", terr_nper - TERR_3PER),6);
                            3  : if ( (terr_nper - TERR_4PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (4per) Violated by %f ps.", terr_nper - TERR_4PER),6);
                            4  : if ( (terr_nper - TERR_5PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (5per) Violated by %f ps.", terr_nper - TERR_5PER),6);
                            5  : if ( (terr_nper - TERR_6PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (6per) Violated by %f ps.", terr_nper - TERR_6PER),6);
                            6  : if ( (terr_nper - TERR_7PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (7per) Violated by %f ps.", terr_nper - TERR_7PER),6);
                            7  : if ( (terr_nper - TERR_8PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (8per) Violated by %f ps.", terr_nper - TERR_8PER),6);
                            8  : if ( (terr_nper - TERR_9PER)  >= 1.0) log_msg (INFO, $sformatf("tERR (9per) Violated by %f ps.", terr_nper - TERR_9PER),6);
                            9  : if ( (terr_nper - TERR_10PER) >= 1.0) log_msg (INFO, $sformatf("tERR (10per) Violated by %f ps.", terr_nper - TERR_10PER),6);
                            10 : if ( (terr_nper - TERR_11PER) >= 1.0) log_msg (INFO, $sformatf("tERR (11per) Violated by %f ps.", terr_nper - TERR_11PER),6);
                            11 : if ( (terr_nper - TERR_12PER) >= 1.0) log_msg (INFO, $sformatf("tERR (12per) Violated by %f ps.", terr_nper - TERR_12PER),6);
                        endcase
                    end
                       
                    // max/min tCK jitter
                    if (abs(tjit_per) - TJIT_PER_TOT >= 1.0)         log_msg (INFO, $sformatf("tJIT (per) Violated by %f ps.", abs(tjit_per) - TJIT_PER_TOT),6);
                    if (abs(tjit_cc)  - TJIT_CC_TOT >= 1.0)          log_msg (INFO, $sformatf("tJIT (cc) Violated by %f ps.", abs(tjit_cc) - TJIT_CC_TOT),6);
                    if (TCK_AVG_MIN   - tck_avg >= 1.0)              log_msg (INFO, $sformatf("tCK (avg)MIN Violated by %f ps.", TCK_AVG_MIN - tck_avg),6);
                    if (tck_avg       - TCK_AVG_MAX >= 1.0)          log_msg (INFO, $sformatf("tCK (avg)MAX Violated by %f ps.", tck_avg - TCK_AVG_MAX),6);
                    if (ts_ck_c       - $time < TCL_ABS_MIN*tck_avg) log_msg (INFO, $sformatf("tCL (abs)MIN Violated by %t", TCL_ABS_MIN*tck_avg - ts_ck_c + $time),6);
                    if (tcl_avg       < TCL_AVG_MIN*tck_avg)         log_msg (INFO, $sformatf("tCL (avg)MIN Violated by %t", TCL_AVG_MIN*tck_avg - tcl_avg),6);
                    if (tcl_avg       > TCL_AVG_MAX*tck_avg)         log_msg (INFO, $sformatf("tCL (avg)MAX Violated by %t", tcl_avg - TCL_AVG_MAX*tck_avg),6);
                end
                tch_avg = tch_avg - tch_value[ck_t_cnt%TDLLK]/$itor(TDLLK);
                tch_avg = tch_avg + tch_width/$itor(TDLLK);
                tch_value[ck_t_cnt%TDLLK] = tch_width;
                tjit_ch_rtime = tch_width - tch_avg;
                tck_duty_cycle = tch_avg/tck_avg;
                tcl_width <= $time - ts_ck_c;
            end
            ck_t_cnt <= ck_t_cnt + 1;
            ts_ck_t  = $time;
            pre_cke  <= cke;
            prev_cmd <= cmd;
        end

        if (ck_c) begin
            
            power_down_q1 <= power_down;
            power_down_q2 <= power_down_q1;
            ///////////////////////////
            // Write Logic
            ///////////////////////////
            if (write) begin
                if (dq_in_en) wr_bc = wr_bc - 1'b1; 
            end
            ///////////////////////////
            // Read Logic
            ///////////////////////////
            if (read) begin
                rd_ca_en = read;
                rd_bc    = rd_bc - 1'b1;
                rd_ca    = rd_ca + 1'b1;
            end
            // clk pin is disabled during self refresh
            if (!self_ref) begin
                if (dll_locked) begin
                    if ($time - ts_ck_t < TCH_ABS_MIN*tck_avg) log_msg (INFO, $sformatf("tCH (abs) Violated by %t", TCH_ABS_MIN*tck_avg - $time + ts_ck_t),6);
                    if (tch_avg < TCH_AVG_MIN*tck_avg)         log_msg (INFO, $sformatf("tCH (avg) Violated by %t", TCH_AVG_MIN*tck_avg - tch_avg),6);
                    if (tch_avg > TCH_AVG_MAX*tck_avg)         log_msg (INFO, $sformatf("tCH (avg) Violated by %t", tch_avg - TCH_AVG_MAX*tck_avg),6);
                end
                // calculate the tcl avg jitter
                tcl_avg = tcl_avg - tcl_value[ck_t_cnt%TDLLK]/$itor(TDLLK);
                tcl_avg = tcl_avg + tcl_width/$itor(TDLLK);
                tcl_value[ck_t_cnt%TDLLK] = tcl_width;
                tch_width <= $time - ts_ck_t;
            end
            ts_ck_c = $time;
        end
    end
end

//
// Precharge Down counter before deactivate
//
always @(posedge ck_t or negedge reset_n) begin : PRE_CNT
    if (~reset_n) begin
        pre_bg = 'h0;
        pre_ba = 'h0;
        for (int i=0;i<16;i=i+1) begin
            pre_cnt[i] = 0;
        end
    end else begin
        for (int bank=0; bank<16; bank++) begin
            if (pre_cnt[bank] == 1 && bank_status[bank]!=BANK_CLOSED ) begin
                if ( bank_status[bank]==BANK_OPEN ) write_cache_to_mem(bank);
                bank_status[bank] = (nomem | nowmem) ? BANK_CLOSED : BANK_DIRTY;
                close_bank(bank);
            end
            if ( pre_cnt[bank]>0 ) pre_cnt[bank] = pre_cnt[bank] - 1;
        end
    end
end


always @(evt_mr_write) begin : MR_CHECK
    //////////////////////////////////
    // MR0 Update the read latency
    //////////////////////////////////
    if ( {bg[0],ba}=='h0 ) begin
        case ({a[12], a[6:4], a[2]})
            5'h00: begin rl = 9;    log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h01: begin rl = 10; end
            5'h02: begin rl = 11;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h03: begin rl = 12;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h04: begin rl = 13;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h05: begin rl = 14;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h06: begin rl = 15;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h07: begin rl = 16;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h08: begin rl = 18;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h09: begin rl = 20;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h0a: begin rl = 22;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h0b: begin rl = 24;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h0c: begin rl = 23;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h0d: begin rl = 17;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h0e: begin rl = 19;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h0f: begin rl = 21;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h10: begin rl = 25;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h11: begin rl = 26;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h12: begin rl = 27;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h13: begin rl = 28;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h14: begin rl = 29;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h15: begin rl = 30;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h16: begin rl = 31;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end
            5'h17: begin rl = 32;   log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end 
            default: begin rl = 32; log_msg (ERR, $sformatf("Unsupported CAS Latency [%d]", rl),0); end 
        endcase
        case ({a[13],a[11:9]})
            4'h0:    nWR = 10;
            4'h1:    nWR = 12;
            4'h2:    nWR = 14;
            4'h3:    nWR = 16;
            4'h4:    nWR = 18;
            4'h5:    nWR = 20;
            4'h6:    nWR = 24;
            4'h7:    nWR = 22;
            4'h8:    nWR = 26;
            default: nWR = 1000;
        endcase
    end
    //////////////////////////////////
    // MR2 Update the write latency
    //////////////////////////////////
    if ( {bg[0],ba}=='h2 ) begin
        case (a[5:3])
            3'h0: begin wl = 9; end
            3'h1: begin wl = 10; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
            3'h2: begin wl = 11; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
            3'h3: begin wl = 12; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
            3'h4: begin wl = 14; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
            3'h5: begin wl = 16; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
            3'h6: begin wl = 18; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
            3'h7: begin wl = 20; log_msg (ERR, $sformatf("Unsupported CAS Write Latency [%d]", wl),0); end
        endcase
    end
    if ( {bg[0],ba}=='h6 ) begin
        if (a[12:10] != 0) log_msg (WARN, "tCCD_L was not set to 4 in MR6. Device operation will not be impacted but performance will be!", 2);
    end
end

always @(ck_t) begin
    #(tck_avg/4);
    if (read) begin
        if (dq_out_en) begin
            if (mr[3][2]) log_msg (INFO, $sformatf("MPR Read [Page:%h BA:%h] Data:%h", mr[3][1:0], rd_ba, dq_out),4);
            else log_msg (INFO, $sformatf("Memory Read [BG:%h BA:%h Row:%h Col:%h] Data:%h", rd_bg, rd_ba, rd_ra[{rd_bg, rd_ba}], rd_ca, dq_out),5);
        end
    end
end
// CPM
//always @(ck_c) begin
//    #(tck_avg/4);
//    if (dq_out_en) begin
//        if (mr[3][2]) log_msg (INFO, $sformatf("MPR Read [Page:%h BA:%h] Data:%h", mr[3][1:0], rd_ba, dq_out),4);
//        else          log_msg (INFO, $sformatf("Memory Read [BG:%h BA:%h Row:%h Col:%h] Data:%h", rd_bg, rd_ba, rd_ra[{rd_bg, rd_ba}], rd_ca, dq_out),5);
//    end
//end
///////////////////////////
// Read Data Mux
///////////////////////////
always_comb begin : DQ_OUT_MUX
    if (dq_out_en)  begin
`ifdef X8
        if (mr[3][2]) begin  // MPR reads
            case(rd_bc)
                4'h7: dq_out = {8{mpr_rd_mux[7]}};
                4'h6: dq_out = {8{mpr_rd_mux[6]}};
                4'h5: dq_out = {8{mpr_rd_mux[5]}};
                4'h4: dq_out = {8{mpr_rd_mux[4]}};
                4'h3: dq_out = (rd_is_bc4 & bc4_a2) ? {8{mpr_rd_mux[7]}} : {8{mpr_rd_mux[3]}};
                4'h2: dq_out = (rd_is_bc4 & bc4_a2) ? {8{mpr_rd_mux[6]}} : {8{mpr_rd_mux[2]}};
                4'h1: dq_out = (rd_is_bc4 & bc4_a2) ? {8{mpr_rd_mux[5]}} : {8{mpr_rd_mux[1]}};
                4'h0: dq_out = (rd_is_bc4 & bc4_a2) ? {8{mpr_rd_mux[4]}} : {8{mpr_rd_mux[0]}};
                endcase
        end else begin // Memory Reads
            case(rd_bc)
                4'h7: dq_out = bc4_a2               ? rd_data[32+:8] : rd_data[00+:8];
                4'h6: dq_out = bc4_a2               ? rd_data[40+:8] : rd_data[08+:8];
                4'h5: dq_out = bc4_a2               ? rd_data[48+:8] : rd_data[16+:8];
                4'h4: dq_out = bc4_a2               ? rd_data[56+:8] : rd_data[24+:8];
                4'h3: dq_out = (rd_is_bc4 ^ bc4_a2) ? rd_data[00+:8] : rd_data[32+:8];
                4'h2: dq_out = (rd_is_bc4 ^ bc4_a2) ? rd_data[08+:8] : rd_data[40+:8];
                4'h1: dq_out = (rd_is_bc4 ^ bc4_a2) ? rd_data[16+:8] : rd_data[48+:8];
                4'h0: dq_out = (rd_is_bc4 ^ bc4_a2) ? rd_data[24+:8] : rd_data[56+:8];
            endcase
        end
`else
        if (mr[3][2]) begin  // MPR reads
            case(rd_bc)
                4'h7: dq_out = {16{mpr_rd_mux[7]}};
                4'h6: dq_out = {16{mpr_rd_mux[6]}};
                4'h5: dq_out = {16{mpr_rd_mux[5]}};
                4'h4: dq_out = {16{mpr_rd_mux[4]}};
                4'h3: dq_out = (rd_is_bc4 & bc4_a2) ? {16{mpr_rd_mux[7]}} : {16{mpr_rd_mux[3]}};
                4'h2: dq_out = (rd_is_bc4 & bc4_a2) ? {16{mpr_rd_mux[6]}} : {16{mpr_rd_mux[2]}};
                4'h1: dq_out = (rd_is_bc4 & bc4_a2) ? {16{mpr_rd_mux[5]}} : {16{mpr_rd_mux[1]}};
                4'h0: dq_out = (rd_is_bc4 & bc4_a2) ? {16{mpr_rd_mux[4]}} : {16{mpr_rd_mux[0]}};
            endcase
        end else begin // Memory Reads
            case(rd_bc)
                4'h7: dq_out = bc4_a2               ? {rd_data[096+:8],rd_data[32+:8]} : {rd_data[064+:8],rd_data[00+:8]};
                4'h6: dq_out = bc4_a2               ? {rd_data[104+:8],rd_data[40+:8]} : {rd_data[072+:8],rd_data[08+:8]};
                4'h5: dq_out = bc4_a2               ? {rd_data[112+:8],rd_data[48+:8]} : {rd_data[080+:8],rd_data[16+:8]};
                4'h4: dq_out = bc4_a2               ? {rd_data[120+:8],rd_data[56+:8]} : {rd_data[088+:8],rd_data[24+:8]};
                4'h3: dq_out = (rd_is_bc4 ^ bc4_a2) ? {rd_data[064+:8],rd_data[00+:8]} : {rd_data[096+:8],rd_data[32+:8]};
                4'h2: dq_out = (rd_is_bc4 ^ bc4_a2) ? {rd_data[072+:8],rd_data[08+:8]} : {rd_data[104+:8],rd_data[40+:8]};
                4'h1: dq_out = (rd_is_bc4 ^ bc4_a2) ? {rd_data[080+:8],rd_data[16+:8]} : {rd_data[112+:8],rd_data[48+:8]};
                4'h0: dq_out = (rd_is_bc4 ^ bc4_a2) ? {rd_data[088+:8],rd_data[24+:8]} : {rd_data[120+:8],rd_data[56+:8]};
            endcase
        end
`endif
    end
end
///////////////////////////
// Write-Leveling CK capture
///////////////////////////

`ifdef X8
initial begin
    wl_ck   = 0;
end

always @(posedge dqs_t_in) begin
    wl_ck = write_leveling_mode ? ck_t : 1'b0;
end

always @(posedge write_leveling_mode) begin
    wl_ck = 1'b0;
end

`else
initial begin
    wl_l_ck = 0;
    wl_u_ck = 0;
end

always @(posedge dqsu_hi) begin
    wl_u_ck = write_leveling_mode ? ck_t : 1'b0;
end

always @(posedge dqsl_hi) begin
    wl_l_ck = write_leveling_mode ? ck_t : 1'b0;
end

always @(posedge write_leveling_mode) begin
    wl_u_ck = 1'b0;
    wl_l_ck = 1'b0;
end

`endif
///////////////////////////
// Write Data Capture
///////////////////////////
`ifdef X8
always @ (posedge dqs_t_in or posedge dqs_c_in) begin
    if ( dq_in_en ) begin
        // CPM - writing data to cache every edge instead of end of burst
        write_cache({wr_bg, wr_ba}, wr_ca, dq_in[7:0], {~mr[3][4] & ~mr[1][11] & mr[5][10] & ~mr[5][11] & ~dm_n_tdqs_t_in});
        if (~mr[3][4] & ~mr[1][11] & mr[5][10] & ~mr[5][11] & ~dm_n_tdqs_t_in) begin    // Data Masking is enabled on this transfer, so do nothing
        end else begin
            case (wr_ca[2:0])
                7: wr_data[7:0]   = dq_in;
                6: wr_data[15:8]  = dq_in;
                5: wr_data[23:16] = dq_in;
                4: wr_data[31:24] = dq_in;
                3: wr_data[39:32] = dq_in;
                2: begin                        // PDA Mode Writes occur on the second rising edge if dq[0] is high
                    if (mr[3][4]) begin         // PDA Mode enabled
                        if (~dq_in[0]) begin    // Write is qualified with dq[0]
                            if ({wr_bg[0],wr_ba} == 3'b110) begin
                                if (mr[6][7] & wr_addr[7]) mr[6][5:0] = wr_addr[5:0];        // only set bits 5:0 if [7] is high
                                mr[6][13:6] = wr_addr[13:6];    // set the bits from 13-6
                            end else
                                mr[{wr_bg[0],wr_ba}] = wr_addr[13:0];
                                log_msg (INFO, $sformatf("MRS PDA Write Reg:%h Data:%h", {wr_bg[0], wr_ba}, wr_addr[13:0]),4);
                            end
                            // unload the FIFO entry whether write occurs or not
                            wr_fifo_rd_addr = (wr_fifo_rd_addr == 4) ? 0 : wr_fifo_rd_addr + 1;
                        end else wr_data[47:40] = dq_in;    // PDA Mode not active, do a normal write
                    end
                1: wr_data[55:48] = dq_in;
                0: wr_data[63:56] = dq_in;
            endcase
        end
        if (~mr[3][4]) begin
            //log_msg (INFO, $sformatf("Memory Write [Bank=%02d Row:%h Col:%h] Data:%h", {wr_bg, wr_ba}, wr_ra[{wr_bg, wr_ba}], wr_ca, dq_in),5);
            if (wr_bc == 0) begin
                // CPM Burst Done
                //log_msg (INFO, $sformatf("Write: %d Addr: %h Data: %h", {wr_bg, wr_ba}, wr_addr, wr_data),5);
                wr_fifo_rd_addr = (wr_fifo_rd_addr == 4) ? 0 : wr_fifo_rd_addr + 1;
            end
        end
        wr_ca = wr_ca + 1;
    end
end
`else //x16 write capture
//always @ (posedge dqsl_t_in or posedge dqsl_c_in or posedge dqsu_t_in or posedge dqsu_c_in)
always @(posedge dqsl_hi or posedge dqsl_lo) begin
    if ( dq_in_en ) begin                                               // data window is open
        if ( ^{dml_n,dq_in[07:00]} === 1'bx ) log_msg (WARN, "Lower DM, DQ driven to X", 1);
        write_cache({1'b0,wr_bg[0], wr_ba}, wr_ca_l, dq_in[7:0],{~mr[3][4] & mr[5][10] & ~mr[5][11] & ~dml_n});
        if (~mr[3][4] & mr[5][10] & ~mr[5][11] & ~dml_n) begin
            // Data Masking is enabled on this transfer, so do nothing
        end else begin
            case(wr_ca_l[2:0])
                7: wr_data[7:0]     = dq_in[7:0];
                6: wr_data[23:16]   = dq_in[7:0];
                5: wr_data[39:32]   = dq_in[7:0];
                4: wr_data[55:48]   = dq_in[7:0];
                3: wr_data[71:64]   = dq_in[7:0];
                2: begin
                    if (mr[3][4]) begin                          // PDA Mode enabled
                        if (~dq_in[0]) begin                     // Write is qualified with dq[0]
                            if ({wr_bg[0],wr_ba} == 3'b110) begin
                                if (mr[6][7] & wr_addr[7]) mr[6][5:0] = wr_addr[5:0];   // only set bits 5:0 if [7] is high
                                mr[6][13:6] = wr_addr[13:6];                            // set the bits from 13-6
                            end else
                                mr[{wr_bg[0],wr_ba}] = wr_addr[13:0];
                            log_msg (INFO, $sformatf("MRS PDA Write Reg:%h Data:%h", {wr_bg[0], wr_ba}, wr_addr[13:0]),4);
                        end
                        // unload the FIFO entry whether write occurs or not
                        wr_fifo_rd_addr = (wr_fifo_rd_addr == 4) ? 0 : wr_fifo_rd_addr + 1;
                    end else wr_data[87:80] = dq_in;    // PDA Mode not active, do a normal write
                end
                1: wr_data[103:96]  = dq_in[7:0];
                0: wr_data[119:112] = dq_in[7:0];
            endcase
        end
        if (~mr[3][4]) begin // PDA disabled
            //log_msg (INFO, $sformatf("Lower Memory Write [Bank=%02d Row=%04h Col=%02h] Data=%h", {wr_bg, wr_ba}, wr_ra[{wr_bg, wr_ba}], wr_ca_l, dq_in[7:0]),5);
            if (wr_bc == 0) begin
                // CPM Burst Done
                //log_msg (INFO, $sformatf("Write: %d Addr: %h Data: %h", {wr_bg, wr_ba}, wr_addr, wr_data),5);
            end
        end
        wr_ca_l = wr_ca_l + 1;
    end
end

always @(posedge dqsu_hi or posedge dqsu_lo) begin
    if ( dq_in_en ) begin                                               // data window is open
        if ( ^{dmu_n,dq_in[15:08]} === 1'bx ) log_msg (WARN, "Upper DM, DQ driven to X", 1);
        write_cache({1'b1,wr_bg[0], wr_ba}, wr_ca_u, dq_in[15:08],{~mr[3][4] & mr[5][10] & ~mr[5][11] & ~dmu_n});
        if (~mr[3][4] & mr[5][10] & ~mr[5][11] & ~dmu_n) begin
            // Data Masking is enabled on this transfer, so do nothing
        end else begin
            case(wr_ca_u[2:0])
                7: wr_data[15:8]    = dq_in[15:8];
                6: wr_data[31:24]   = dq_in[15:8];
                5: wr_data[47:40]   = dq_in[15:8];
                4: wr_data[63:56]   = dq_in[15:8];
                3: wr_data[79:72]   = dq_in[15:8];
                2: wr_data[95:88]   = dq_in[15:8];
                1: wr_data[111:104] = dq_in[15:8];
                0: wr_data[127:120] = dq_in[15:8];
            endcase
        end
        //if (~mr[3][4]) log_msg (INFO, $sformatf("Upper Memory Write [Bank=%02d Row=%04h Col=%02h] Data:%h", {1'b1,wr_bg, wr_ba}, wr_ra[{wr_bg, wr_ba}], wr_ca_u, dq_in[15:8]),5);

        if (~mr[3][4]) begin
            if (wr_bc == 0) begin
                // CPM Burst Done
                //log_msg (INFO, $sformatf("Write: %d Addr: %h Data: %h", {wr_bg, wr_ba}, wr_addr, wr_data),5);
                if (wr_fifo_rd_addr == 4) wr_fifo_rd_addr = 0;
                else wr_fifo_rd_addr = wr_fifo_rd_addr + 1;
            end
        end
        wr_ca_u = wr_ca_u + 1;
    end
end


always @(posedge dqsu_hi) begin
    #200.0; -> dqsu_hi_evt;
    if ( dq_in_en && !dqsl_hi ) log_msg (WARN, "Lower DQS did not fire", 1);
end

always @(posedge dqsu_lo) begin
    #200.0; -> dqsu_lo_evt;
    if ( dq_in_en && !dqsl_lo ) log_msg (WARN, "Lower DQS did not fire", 1);
end

always @(posedge dqsl_hi) begin
    #200.0; -> dqsl_hi_evt;
    if ( dq_in_en && !dqsu_hi ) log_msg (WARN, "Upper DQS did not fire", 1);
end

always @(posedge dqsl_lo) begin
    #200.0; -> dqsl_lo_evt;
    if ( dq_in_en && !dqsu_lo ) log_msg (WARN, "Upper DQS did not fire", 1);
end

`endif
//
// MPR Read Mux
//
parameter P_FINE      = 9'h1FF;
parameter P_CDLY      = 6'h04;
parameter P_CDLY_MMIC = 1'b0;
always_comb begin : MPR_RD_MUX
    case({mr[3][1:0], rd_ba})
        4'b00_00: mpr_rd_mux = mpr0[0];                                                     // Page00, MPR0
        4'b00_01: mpr_rd_mux = mpr0[1];                                                     // Page00, MPR1
        4'b00_10: mpr_rd_mux = mpr0[2];                                                     // Page00, MPR2
        4'b00_11: mpr_rd_mux = mpr0[3];                                                     // Page00, MPR3
        4'b01_00: mpr_rd_mux = 8'h0;                                                        // Page01, MPR0
        4'b01_01: mpr_rd_mux = 8'h0;                                                        // Page01, MPR1
        4'b01_10: mpr_rd_mux = 8'h0;                                                        // Page01, MPR2
        4'b01_11: mpr_rd_mux = {1'b0,1'b0,mr[5][2:0],1'b0,1'b0,1'b0};                       // Page01, MPR3
        4'b10_00: mpr_rd_mux = {1'b0, 1'b0, mr[2][11],1'b0, 1'b1, mr[2][12], mr[2][10:9]};  // Page10, MPR0
        4'b10_01: mpr_rd_mux = {mr[6][6:0], mr[3][3]};                                      // Page10, MPR1
        4'b10_10: mpr_rd_mux = {mr[0][6:4], mr[0][2], mr[0][12], mr[2][5:3]};               // Page10, MPR2
        4'b10_11: mpr_rd_mux = {mr[1][10:8], mr[5][8:6], mr[1][2:1]};                       // Page10, MPR3
        4'b11_00: mpr_rd_mux = 8'h0;                                                        // Page11, MPR0 - {pcal, ncal[4:2]}
        4'b11_01: mpr_rd_mux = 8'h0;                                                        // Page11, MPR1 - {ncal[1:0], dll_lock, ncnt_done, ncnt[3:1],~ncnt[0]}
        4'b11_10: mpr_rd_mux = {P_CDLY, P_CDLY_MMIC, P_FINE[8]};                            // Page11, MPR2 - {cdly, cdl_mmic, fine[8]}
        4'b11_11: mpr_rd_mux = P_FINE[7:0];                                                 // Page11, MPR3 - fine[7:0]
    endcase
end
//
// Reset Task
// 
task do_reset;
    integer i;
    begin
        mpr0[0] = 8'h55;    // MPR Page 0, 00 
        mpr0[1] = 8'h33;    // MPR Page 0, 01
        mpr0[2] = 8'h0f;    // MPR Page 0, 10 
        mpr0[3] = 8'h00;    // MPR Page 0, 11 
        mr[0]   = 8'h00;    // Reset the Mode Registers
        mr[1]   = 8'h00;
        mr[2]   = 8'h00;
        mr[3]   = 8'h00;
        mr[4]   = 8'h00;
        mr[5]   = 8'h00;
        mr[6]   = 8'h00;
    end
endtask
//
// Command Interpreter
//
task do_cmd;
    integer i;
    integer tfaw_cnt;
    begin : TSK_DO_CMD
        //
        // Command Sequence Checking
        //
        if (write_leveling_mode && cmd != DES & cmd != MRS)  log_msg (ERR, "Only DES or MRS commands are allowed in Write-Leveling Mode", 0);
        if (mpxlh_cnt != 0 && cmd != NOP)                    log_msg (ERR, "Only NOP commands are allowed during tMPX_LH after Maximum Power-Save Mode exit", 0);
        if (xmp_cnt   != 0 && cmd != DES & cmd != NOP)       log_msg (ERR, "Only DES or NOP commands are allowed during tXMP after Maximum Power-Save Mode exit", 0);
        if (xmp_cnt   == 0 && xmp_dll_cnt != 0 && cmd == RD) log_msg (ERR, "Only commands that do not require a locked DLL are allowed during tXMP_DLL after Maximum Power-Save Mode exit", 0);

        //
        // Command Decoder
        //
        casex(cmd)
            // ================================================================
            MRS: begin
                if ( dirty_banks() != '0 ) log_msg (WARN, $sformatf("Received MRS command without previous STORE, banks not stored [%b]",dirty_banks()), 0);
                if (mr[3][4]) begin    // PDA mode MRS command
                    // signal the start of a write cycle
                    wr_shifter[wl] = 1;
                    // determine write type
                    case(mr[0][1:0])
                        2'b00: begin            // fixed BL8
                            cmd_wr_bc = 8;      // Burst Length + 1 (preamble)
                            wr_adr_mask = 3'b000;
                            log_msg (INFO, $sformatf("Received Fixed BL8 PDA mode MRS command MR%d=%013b", {bg, ba}, a),4); 
                        end
                        2'b01: begin                // on the fly
                            if (a[12]) begin        // BL8
                                log_msg (INFO, $sformatf("Received OTF BL8 PDA mode MRS command MR%d=%013b", {bg, ba}, a),4);
                                cmd_wr_bc = 8;      // Burst Length + 1 (preamble)
                                wr_adr_mask = 3'b000;
                            end else begin          // BC4
                                log_msg (INFO, $sformatf("Received OTF BC4 PDA mode MRS command MR%d=%013b", {bg, ba}, a),4);
                                cmd_wr_bc = 4;      // Burst Length + 1 (preamble)
                                wr_adr_mask = 3'b100;
                            end
                        end
                        2'b10: begin                // BC4
                            log_msg (INFO, $sformatf("Received Fixed BC4 PDA mode MRS command MR%d=%013b", {bg, ba}, a),4);
                            cmd_wr_bc = 4;          // Burst Length + 1 (preamble)
                            wr_adr_mask = 3'b100;
                        end
                        2'b11: begin                // Not supported
                        end
                    endcase
                    // store the write data info
                    wr_rd_addr = a;
                    cmd_wr_ca = a[CA_BITS-1:0];
                    cmd_wr_ca[2:0] = cmd_wr_ca[2:0] & wr_adr_mask;    // mask column address bits based on write: BL8 supports only 000; BC4 supports 100 and 000 modes
                    wr_fifo[wr_fifo_wr_addr] = {a[10], cmd_wr_bc, bg, ba, cmd_wr_ca, {{CA_BITS{1'b0}}, {cas_n_a15, we_n_a14, a}}};
                    if (wr_fifo_wr_addr == 4) wr_fifo_wr_addr = 0;
                    else wr_fifo_wr_addr = wr_fifo_wr_addr + 1;
                end else begin    // normal MRS command
                    log_msg (INFO, $sformatf("Received MRS command MR%d=%013b", {bg,ba}, a),3); 
                    if ({bg[0],ba} == 3'b110) begin                     // if MR6
                      if (mr[6][7] & a[7]) mr[6][5:0]     = a[5:0];    // only set bits 5:0 if [7] is high
                                           mr[6][13:6]    = a[13:6];   // set the bits from 13-6
                    end else begin         mr[{bg[0],ba}] = a[13:0];   // write mode register
                    end
                    -> evt_mr_write;
                end
                if (|active_banks()) log_msg (ERR, $sformatf("Received MRS command while banks=[%016b] are still activate", active_banks()),0);
                if (mr[1][10:8] !== 3'b000 && (odt_int !== 1'b0 || odt_int == 1'b0 && dodtloff !== 'b0) && mr[3][4] !== 1'b1 )
                    log_msg (ERR, $sformatf("Received MRS command with RTT(NOM) enabled and odt(=%b) not in low state", odt_int),0);
                if (write_leveling_mode) begin
                    if ({bg[0],ba} !== 3'b001) begin
                        log_msg (ERR, $sformatf("Only MRS commands to MR1 [bg[0],ba=%d] are allowed during Write-Leveling mode", {bg[0],ba}),0);
                    end else begin
                        if (a[7]) begin // still in write leveling mode so only a[12] can toggle
                            if (((wr_lvl_mr1 ^ a[13:0]) & 14'b10111101111111) !== 14'h0)
                                log_msg (ERR, "Only MR1 A12 can change during Write-Leveling mode",0);
                        end else begin // Exiting write-leveling, so a12-a8 and a2-a1 may change
                            if (((wr_lvl_mr1 ^ a[13:0]) & 14'b10000001111001) !== 14'h0)
                                log_msg (ERR, $sformatf("Only MR1 A12-A8 and A2-A1 can change when exiting  Write-Leveling mode [mr1=%14b a=%14b]", wr_lvl_mr1, a[13:0]),0);
                        end
                    end
                    wr_lvl_mr1 =  a[13:0];
                end
                ts_mrs  <= $time;
                tck_mrs <= ck_t_cnt;
            end
            // ================================================================
            REF: begin
                if (cke == 1) begin
                    if ( mr[3][8] ) begin
                        log_msg (INFO, "Received REF command --> Store All", 2);
                        check_store_time;
                        check_active_banks(DEC_REF);
                        last_store = STO_ALL;
                        -> evt_do_store;
                    end else begin
                        log_msg (INFO, "Received REF command (ignored)", 2);
                    end
                    ts_refresh  <= $time;
                    tck_refresh <= ck_t_cnt;
                end else begin
                    log_msg (INFO, "Received SRE command", 2);
                    //
                    // SRE Error Checking
                    //
                    if (mr[3][2])           log_msg (ERR, "Received SRE command while in MPR access mode", 0);
                    if (|active_banks())    log_msg (ERR, $sformatf("Received SRE command while banks=[%016b] are still active", active_banks()),0);
                    if (prev_cmd != DES)    log_msg (ERR, "Received SRE command and Previous command was not Deselect", 0);
                    //
                    // Set Self Refresh mode and disable the DLL
                    //
                    self_ref = 1'b1;        // Set Self Refresh state
                    dll_en   = 1'b0;        // DLL is disabled on self refresh entry 
                    odt_en   = 1'b0;        // ODT is disabled on self refresh entry
                    last_store = STO_ALL;
                    -> evt_do_store;
                end
                if ( nomem )  log_msg (WARN, "REF/SRE command does not affect the memory Array in NOMEM mode", 1);
                if ( nowmem ) log_msg (WARN, "REF/SRE command does not affect the memory Array in NOWMEM mode", 1);
            end
            // ================================================================
            PRE: begin
                if (mr[3][2]) log_msg (ERR, "Received PRE command while in MPR access mode", 0);
                // Execute a Single Bank Precharge or Precharge All banks
                if (a[10]) log_msg (INFO, "Received PREA command", 4); 
                else       log_msg (INFO, $sformatf("Received PRE command BG=%d BA=%d", bg, ba),4); 
                for (i=0; i<`BANKS; i=i+1) begin
                    if (a[10] || {bg,ba}==i) begin
                        pre_cnt[i] = $ceil(max(TRP/tck_avg, TRP_TCK));
                        `ifdef X16
                        pre_cnt[i+8] = $ceil(max(TRP/tck_avg, TRP_TCK));
                        `endif
                        ts_pre_bank[i] <= $time;
                    end
                end
                pre_bg = bg;
                pre_ba = ba;
                ts_pre  <= $time;
                tck_pre <= ck_t_cnt;
                if ( nomem )  log_msg (INFO, "PRE command does not affect the memory Array in NOMEM mode", 1);
                if ( nowmem ) log_msg (INFO, "PRE command does not affect the memory Array in NOWMEM mode", 1);
            end
            // ================================================================
            ACT: begin
                //
                // ACTIVATE
                //
                bank_status[{bg,ba}] = BANK_OPEN;
                `ifdef X16
                bank_status[{1'b1,bg,ba}] = BANK_OPEN;
                `endif
                ts_act_bank[{bg,ba}] = $time;
                ts_act  = $time;
                tck_act = ck_t_cnt;
                ts_act_group[bg] = $time;
                tck_act_group[bg] = ck_t_cnt;
                wr_ra[{bg,ba}] = {cas_n_a15, we_n_a14, a[13:0]};
                rd_ra[{bg,ba}] = {cas_n_a15, we_n_a14, a[13:0]};
                log_msg (INFO, $sformatf("Received ACT command BG=%d BA=%d RA=%0h", bg, ba, a),4);
                if ( bankN_active_row[{bg,ba}] != -1 ) begin
                    log_msg (ERR, $sformatf("Received ACT: Bank=%d Row=%h, Row Already Active!", {bg,ba}, bankN_active_row[{bg,ba}]),1);
                end else begin
                    -> evt_load_cache_from_mem;
                end
                if ( nomem )  log_msg (INFO, "ACT command does not affect the memory Array in NOMEM mode", 1);
                if ( nowmem ) log_msg (INFO, "ACT command does not affect the memory Array in NOWMEM mode", 1);
                tfaw_cnt = 0;
                for (i=0; i<`BANKS; i=i+1) begin
                    if ($time - ts_act_bank[i] < TFAW) begin
                        tfaw_cnt = tfaw_cnt + 1;
                    end
                end
                if (tfaw_cnt > 4) log_msg (WARN, $sformatf("tFAW Violated when activating BG=%d BA=%d RA=%0h", bg, ba, a),1);
                if (mr[3][2])     log_msg (ERR,  "Received ACT command while in MPR access mode", 0);
                check_store_time;
            end
            // ================================================================
            WR: begin
                if (~mr[3][2] && bank_status[{bg,ba}] != BANK_OPEN ) begin
                    log_msg (ERR,  $sformatf("Received WR command to inactive[%016b] bank[%h]. Write is not executed.", active_banks(), {bg,ba}),0);
                end else begin
                    if (mr[3][2]) begin                 // MPR writes
                        wr_mpr_cnt = ck_t_cnt;          // mark MPR write
                        if (mr[3][1:0] == 2'b00) begin  // MPR writes are valid only to Page 00
                            mpr0[ba] = a[7:0];          // store the data
                            log_msg (INFO, $sformatf("MPR Write [BA:%h Page:%h] Data:%h", ba, mr[3][1:0], a[7:0]),3);
                        end else begin                  // Invalid MPR Write - Issue a warning
                            log_msg (ERR,  $sformatf("MPR Writes to Page:%1h are not supported and operation is ignored.", mr[3][1:0]),0); 
                        end
                    end else begin // Memory writes
                        // signal the start of a write cycle
                        wr_shifter[wl] = 1;
                        // determine write type
                        case(mr[0][1:0])
                            2'b00: begin            // fixed BL8
                                last_write = a[10] ? WRA_BL8MRS : WR_BL8MRS;
                                cmd_wr_bc = 8;      // Burst Length + 1 (preamble)
                                wr_adr_mask = 3'b000;
                                if (mr[3][2]) log_msg (INFO, $sformatf("Received Fixed BL8 MPR WR command Page: %d BA=%d", mr[3][1:0], ba),4);
                                else          log_msg (INFO, $sformatf("Received Fixed BL8 MEM WR command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4);
                            end
                            2'b01: begin            // on the fly
                                if (a[12]) begin    // BL8
                                    last_write = a[10] ? WRA_BL8OTF : WR_BL8OTF;
                                    log_msg (INFO, $sformatf("Received OTF BL8 MEM WR command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                                    cmd_wr_bc = 8;  // Burst Length + 1 (preamble)
                                    wr_adr_mask = 3'b000;
                                end else begin      // BC4
                                    last_write = a[10] ? WRA_BC4OTF : WR_BC4OTF;
                                    log_msg (INFO, $sformatf("Received OTF BC4 MEM WR command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                                    cmd_wr_bc = 4;  // Burst Length + 1 (preamble)
                                    wr_adr_mask = 3'b100;
                                end
                            end
                            2'b10: begin            // BC4
                                last_write = a[10] ? WRA_BC4MRS : WR_BC4MRS;
                                if (mr[3][2])   log_msg (INFO, $sformatf("Received Fixed BC4 MPR WR command Page: %d BA=%d", mr[3][1:0], ba),4);
                                else            log_msg (INFO, $sformatf("Received Fixed BC4 MEM WR command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                                cmd_wr_bc = 4;      // Burst Length + 1 (preamble)
                                wr_adr_mask = 3'b100;
                            end
                            2'b11: begin            // Not supported
                            end
                        endcase
                        // store the write data info
                        wr_rd_addr = {wr_ra[{bg, ba}], a[CA_BITS-1:0]}/BL_MAX;
                        cmd_wr_ca = a[CA_BITS-1:0];
                        cmd_wr_ca[2:0] = cmd_wr_ca[2:0] & wr_adr_mask;    // mask column address bits based on write: BL8 supports only 000; BC4 supports 100 and 000 modes
                        wr_fifo[wr_fifo_wr_addr] = {cmd_wr_bc, bg, ba, cmd_wr_ca, wr_rd_addr};
                        wr_fifo_wr_addr = (wr_fifo_wr_addr == 4) ? 0 : wr_fifo_wr_addr + 1;
                        bankN_AP[{bg,ba}] = (wl+1)*a[10];  // tag if auto-precharge
                        `ifdef X16
                        bankN_AP[{1'b1,bg,ba}] = (wl+1)*a[10];
                        `endif
                        // Auto-precharge
                        tck_write_group[bg]     <= ck_t_cnt;
                        tck_write_end_group[bg] <= ck_t_cnt + wl + cmd_wr_bc/2; // use for tWTR
                        ts_write_group[bg]      <= $time;
                        ts_write[{bg,ba}]       <= $time;
                        tck_last_write          <= ck_t_cnt;
                    end
                end
            end
            // ================================================================
            RD: begin
                //
                // RD Error Checking
                //
                if ( mr[3][2] && (ck_t_cnt - wr_mpr_cnt) < TWR_MPR ) log_msg (ERR, "tWR_MPR (min) Violated, Read MPR", 0);
                if (dll_en && !dll_locked) begin
                    log_msg (ERR, $sformatf("DLL is either not enabled(=%d) or not locked(=%d) before RD cmd issued", dll_en, dll_locked),0); 
                end
                if (~mr[3][2] && bank_status[{bg,ba}] != BANK_OPEN ) begin
                    log_msg (ERR, $sformatf("Received RD command to inactive[%016b] bank[%h]. Read is not executed.", active_banks(), {bg,ba}),0);
                end else begin
                    if ( dll_en ) rd_shifter[rl] = 1;   // DLL On  CL
                    else          rd_shifter[rl-1]=1;   // DLL Off CL-1
                    // determine read type
                    case(mr[0][1:0])
                        2'b00: begin    // BL8
                            cmd_rd_bc = 8;
                            if (mr[3][2])   log_msg (INFO, $sformatf("Received Fixed BL8 MPR RD command Page: %d BA=%d", mr[3][1:0], ba),4);
                            else            log_msg (INFO, $sformatf("Received Fixed BL8 MEM RD command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                        end
                        2'b01: begin    // on the fly
                            if (mr[3][2]) log_msg (ERR, "OTF MPR RD commands are not allowed.", 0);
                            if (a[12]) begin    // BL8
                                log_msg (INFO, $sformatf("Received OTF BL8 MEM RD command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                                cmd_rd_bc = 8;    // Burst Length + 1 (preamble)
                            end else begin    // BC4
                                log_msg (INFO, $sformatf("Received OTF BC4 MEM RD command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                                cmd_rd_bc = 4;    // Burst Length + 1 (preamble)
                            end
                        end
                        2'b10: begin    // BC4
                            if (mr[3][2])
                                 log_msg (INFO, $sformatf("Received Fixed BC4 MPR RD command Page: %d BA=%d", mr[3][1:0], ba),4);
                            else log_msg (INFO, $sformatf("Received Fixed BC4 MEM RD command BG=%d BA=%d CA=%0h", bg, ba, a[CA_BITS-1:0]),4); 
                            cmd_rd_bc = 4;    // Burst Length + 1 (preamble)
                        end
                        2'b11: begin     // Not supported
                        end
                    endcase
                    // store the read data
                    cmd_rd_ca = a[CA_BITS-1:0];
                    cmd_rd_ca[2:0] = cmd_rd_ca[2:0] & 3'b100;        // we only support sequential bursts and modes 100, 000
                    rd_fifo[rd_fifo_wr_addr] = {a[10], cmd_rd_bc, bg, ba, cmd_rd_ca, rd_ra[{bg, ba}]};
                    rd_fifo_wr_addr = (rd_fifo_wr_addr == 4) ? 0 : rd_fifo_wr_addr + 1;
                    if ( a[10] ) begin
                        pre_cnt[{bg,ba}] <= $ceil( max((max(TRTP,TRTP_TCK*tck_avg) + TRP)/tck_avg, TRP_TCK) );
                        `ifdef X16
                        pre_cnt[{1'b1,bg,ba}] <= $ceil( max((max(TRTP,TRTP_TCK*tck_avg) + TRP)/tck_avg, TRP_TCK) );
                        `endif
                        bankN_AP[{bg,ba}] = (rl+1);
                        `ifdef X16
                        bankN_AP[{1'b1,bg,ba}] = (rl+1);
                        `endif
                    end
                    tck_read_group[bg] <= ck_t_cnt;
                    ts_read_group[bg]  <= $time;
                    ts_read[{bg,ba}]   <= $time;
                    tck_read[{bg,ba}]  <= ck_t_cnt;
                    tck_last_read      <= ck_t_cnt;
                end
            end
            // ================================================================
            NOP: begin     // NOP or Max Power Saving Mode Exit
                if (pre_cke==0 && cke==1) begin
                    log_msg (INFO, "Received SRX(NOP) command", 0); 
                    ts_self_ref<=$time;
                    tck_self_ref<=ck_t_cnt;
                    // Exit self refresh mode and enable the DLL
                    self_ref = 1'b0;    // remove self refresh state
                    dll_en = 1'b1;      // DLL is enabled on self refresh exit 
                    odt_en = 1'b1;      // ODT is enabled on self refresh exit
                end else begin
                    if (mr[3][2]) log_msg (ERR, "Received NOP command while in MPR access mode", 0);
                end
            end
            // ================================================================
            DES: begin    // Deselect or Max Power Saving Mode Exit
            end
            // ================================================================
            ZQC: begin  // ZQ Calibration
                if (mr[3][2])        log_msg (ERR, "Received ZQC command while in MPR access mode", 0);
                if (|active_banks()) log_msg (WARN, $sformatf("Some banks still active [%016b] before ZQ command", active_banks()),1);
                if (a[10]) begin
                    log_msg (INFO, "Received ZQCL command", 2);
                    zqcl = 1;
                end else begin        // short calibration
                    log_msg (INFO, "Received ZQCS command", 2);
                    `ifdef REV_S2
                    zqcs = 0;
                    log_msg (INFO, "ZQCS command is ignored", 5);
                    `else
                    zqcs = 1;
                    log_msg (ERR, "ZQCS command not supported", 0);
                    `endif
                end
                if (zqcl==1 || zqcs==1) begin
                    ts_zq<=$time;
                    tck_zq<=ck_t_cnt;
                end
            end
            // ================================================================
            STO: begin
                if (a[10]) begin    // Store All Banks
                    log_msg (INFO, "Received STOA command", 2);
                    if ( !dll_en && mr[3][7:6] != 2'b11 ) log_msg (ERR, "STOA command with DLL Off only supports 16-bank configuration (mr[3][7:6]=3)", 0);
                    check_store_time;
                    last_store = STO_ALL;
                    -> evt_do_store;
                end else begin      // Store one Bank {BG,BA}
                    log_msg (INFO, $sformatf("Received STO command BG=%d BA=%d", bg, ba),2);
                    check_store_time;
                    last_store = STO_ONE;
                    ts_store_bank[{bg,ba}] <= $time;
                    -> evt_do_store;
                end
                if ( nomem )  log_msg (WARN, "STORE command does not affect the memory Array in NOMEM mode", 1);
                if ( nowmem ) log_msg (WARN, "STORE command does not affect the memory Array in NOWMEM mode", 1);
                //ts_store<=$time;
                //tck_store<=ck_t_cnt;
            end
            // ================================================================
        endcase 
    end
endtask
//
// Signal DLL is enabled or disabled
//
always@(mr[1][0]) begin
    dll_en = mr[1][0];
end

always @(dll_en) begin
    if (reset_n) begin
        if (dll_en)
             log_msg (INFO, "DLL is Enabled", 2); 
        else log_msg (INFO, "DLL is Disabled", 2);
    end
end
//
// Signal DLL Reset State
//
always@(mr[0][8]) begin    // DLL Reset
    if (reset_n) begin
        if (mr[0][8]) begin
            if (dll_en) log_msg (INFO, "Received DLL Reset", 2); 
            else        log_msg (ERR,  "Received DLL Reset, but DLL is Disabled", 0); 
        end
        dll_reset = mr[0][8];
    end
end
//
// Model DLL Lock Time
//
//    DLL must be enabled and then reset via the mode registers.
//    If the DLL is disabled, it will lose lock. Lock can not be attained until
//    the DLL is enabled and reset.
//
reg [1:0] dll_state;
always @(negedge reset_n or posedge ck_t) begin
    if (!reset_n) begin
        dll_state    <= 2'b00;
        dll_locked   <= 0;
        dll_lock_cnt <= TDLLK-1;
    end else begin
        case(dll_state)
        2'b00: begin
            dll_locked <= 0;
            dll_lock_cnt <= TDLLK-1;
            if (dll_en & dll_reset) dll_state <= 2'b01;
        end
        2'b01: begin
            mr[0][8] = 0;        // reset the mode register bit
            dll_locked <= 0;
            dll_lock_cnt <= dll_lock_cnt - 1'b1;
            if (dll_lock_cnt == 'h0) dll_state = 2'b10;
            if (~cke) log_msg (ERR, "CKE must be High during tDLLK", 0);
        end
        2'b10: begin
            dll_locked <= 1;
            if (~dll_en | dll_reset) dll_state <= 2'b00;
        end
        endcase
    end
end
//
// Report DLL Status
//
always@(dll_locked) begin
    if (reset_n) begin
        if (dll_locked) log_msg (INFO, "DLL is Locked", 2); 
        else            log_msg (INFO, "DLL lost Lock", 2);
    end
end
//
// Model ZQ Process
//
//  1. All banks must be precharged (checked above)
//  2. tRP must be met
//  3. CKE must be high during the procedure (checked in this process)
//  4. On-die termination must be disabled via the ODT signal or MRS during the calibration procedure or the ST-MRAM will automatically disable RTT
//  5. All devices connected to the DQ bus should be High-Z during the calibration procedure
//
always @(negedge reset_n or posedge ck_t) begin
    if (!reset_n) begin
        zqcl_cnt = ZQ_INIT_CLKS;
        zqcs_cnt = ZQ_ZQCS_CLKS;
        zqcl     = 0;
        zqcs     = 0;
    end else begin
        if (zqcl) begin
            zq_odt_disable = 1;
            if (~cke) log_msg (ERR, "CKE must be High during tZQinit/ZQoper", 0);
            zqcl_cnt = zqcl_cnt - 1'b1;
            if (zqcl_cnt == 0) begin
                zq_odt_disable = 0;
                zqcl = 0;
                zqcl_cnt = ZQ_OPER_CLKS;
                log_msg (INFO, "ZQ Long calibration complete", 2);
                odt_static = 0;
            end
        end
        if (zqcs) begin
            zq_odt_disable = 1;
            if (~cke) log_msg (ERR, "CKE must be High during tZQCS", 0);
            zqcs_cnt = zqcs_cnt - 1'b1;
            if (zqcs_cnt == 0) begin
                zq_odt_disable = 0;
                zqcs = 0;
                zqcs_cnt = ZQ_ZQCS_CLKS;
                log_msg (INFO, "ZQ Short calibration complete", 2);
                odt_static = 0;
            end
        end
    end
end
//
// Model ODT
//
//
// ODT Checking
//
// If RTT(NOM) is to be enabled in MR1, the ODT input signal must be statically held LOW.
//
// when CKE goes high, ODT must be valid
//
always @(posedge cke or negedge reset_n) begin
    if (~reset_n) begin
        odt_static = 0;
    end else begin
        if (odt_int !== 1'b0 & odt_int !== 1'b1)
            log_msg (ERR, $sformatf("ODT(=%b) pin must be valid when CKE goes active", odt_int),0);
        if (~power_down  & ~self_ref) odt_static = 1;
    end
end

always @(odt_int) begin
    if (odt_static) log_msg (ERR, "ODT pin must be static after CKE goes active until ZQ calibration is complete",0);
end
//
// ODT termination values
//
//always_comb begin
always @(evt_mr_write) begin : RTT_DECODE
    case(mr[1][10:8])
        3'b000: rtt_nom_value = TERM_HIGHZ;
        3'b001: rtt_nom_value = TERM_60_OHM;
        3'b010: rtt_nom_value = TERM_120_OHM;
        3'b011: rtt_nom_value = TERM_40_OHM;
        3'b100: rtt_nom_value = TERM_240_OHM;
        3'b101: rtt_nom_value = TERM_48_OHM;
        3'b110: rtt_nom_value = TERM_80_OHM;
        3'b111: rtt_nom_value = TERM_34_OHM;
    endcase
    rtt_nom_en = (mr[1][10:8] == 3'b000) ? 0 : 1;
    case(mr[2][11:9])
        3'b000: begin rtt_wr_value = TERM_HIGHZ;    end
        3'b001: begin rtt_wr_value = TERM_120_OHM; log_msg (ERR, "RTT(WR) NOT supported", 0); end
        3'b010: begin rtt_wr_value = TERM_240_OHM; log_msg (ERR, "RTT(WR) NOT supported", 0); end
        3'b011: begin rtt_wr_value = TERM_HIGHZ;   log_msg (ERR, "RTT(WR) NOT supported", 0); end
        3'b100: begin rtt_wr_value = TERM_80_OHM;  log_msg (ERR, "RTT(WR) NOT supported", 0); end
        3'b101,
        3'b110,    
        3'b111: begin log_msg (ERR, "Attempt to set RTT(WR) to Reserved Value", 0); end
    endcase
    rtt_wr_en = (mr[2][11:9] == 3'b000) ? 0 : 1;
    case(mr[5][8:6])
        3'b000: rtt_park_value = TERM_HIGHZ;
        3'b001: rtt_park_value = TERM_60_OHM;
        3'b010: rtt_park_value = TERM_120_OHM;
        3'b011: rtt_park_value = TERM_40_OHM;
        3'b100: rtt_park_value = TERM_240_OHM;
        3'b101: rtt_park_value = TERM_48_OHM;
        3'b110: rtt_park_value = TERM_80_OHM;
        3'b111: rtt_park_value = TERM_34_OHM;
    endcase
    rtt_park_en = (mr[5][8:6] == 3'b000) ? 0 : 1;
end
//
// Synchronous ODT mode
//
//    1. DLL must be enabled and locked

//
// ODT Latency Calculations
//
always @(posedge ck_t or negedge reset_n) begin
    if (~reset_n) begin
        dodtlon    = 'h0;
        dodtloff   = 'h0;
        rodtl_on   = 'h0;
        rodtlon    = 'h0;
        pre_odt    = 'h0;
        dodt_on    = 0;
        dodt_wr_on = 0;
        odtlcwn    = 0;
    end else begin
        //
        // The odt pin state determines whether rtt is HIGHZ/RTTPARK or RTTMON 
        //
        if (dll_en && odt_en) begin                // ODT and DLL must be enabled
            if (dll_locked & rtt_nom_en) begin        // DLL must be locked and rtt_nom must be enabled 
                if (|dodtlon) begin
                    dodtlon = dodtlon - 1;
                    if (dodtlon == 'h0) dodt_on = 1;
                end
                if (~pre_odt && odt_int) begin         // odt just went high
                    dodtlon = wl - 2;
                end
                if (|dodtloff) begin
                    dodtloff = dodtloff - 1;
                    if (dodtloff == 'h0) dodt_on = 0;
                end
                if (pre_odt && ~odt_int) begin         //odt just went low
                    dodtloff = wl - 2;
                end
            end
        end else    // if the DLL or ODT is disabled and dotd_on is enabled, reset it
            dodt_on = 0;
        //
        // if RTTWR is enabled, these latency counters determine when
        // rtt is switched to RTTWR.
        //
        odtlcnw_calc = wl - 2;
        if (rtt_wr_en) begin    // if RTTWR is enabled
            odt_wr_shifter = {1'b0, odt_wr_shifter[MAX_CWL-2:1]};    // shift out the lowest bit
                        dodt_wr_on = (odt_wr_shifter[0]) ? 1
                                                         : (dodt_wr_on & |odtlcwn) ? 1 
                                                                                   : 0;
            if (~cs_n & act_n & ras_n & ~cas_n_a15 & ~we_n_a14) begin    // A write command is occuring
                case(mr[0][1:0])
                    2'b00: begin     // Fixed BL8
                        odtlcwn_val = 5;
                           end
                    2'b01: begin    // OTF
                        if (a[12]) begin // BL8
                            odtlcwn_val = 5;
                        end else begin    //BC4
                            odtlcwn_val = 3;
                        end
                    end
                    2'b10: begin    // Fixed BC4
                        odtlcwn_val = 3;
                    end
                    2'b11: begin
                    end
                endcase
                // When read command occurs, inject a one at
                // the proper calculated CWL position
                odt_wr_shifter[odtlcnw_calc] = 1;
            end
            if (odt_wr_shifter[0]) odtlcwn = odtlcwn_val;
            else if (|odtlcwn) begin
                odtlcwn = odtlcwn - 1;
            end
        end
        //
        // If RTTPARK is enabled, these latency counters determine
        // when rtt switches between HIGHZ and RTTPARK for reads
        //
        rodtloff_calc = rl - 2;
        odt_rd_shifter = {1'b0, odt_rd_shifter[MAX_CL-2:1]};    // shift out the lowest bit
        rodtl_on = (odt_rd_shifter[0]) ? 1
                                       : (rodtl_on & |rodtlon) ? 1 
                                       : 0;
        if (~cs_n & act_n & ras_n & ~cas_n_a15 & we_n_a14) begin    // A read command is occuring
            case (mr[0][1:0])
                2'b00: begin     // Fixed BL8
                    rodtlon_val = 5;
                       end
                2'b01: begin    // OTF
                    if (a[12]) begin // BL8
                        rodtlon_val = 5;
                    end else begin    //BC4
                        rodtlon_val = 3;
                    end
                end
            
                2'b10: begin    // Fixed BC4
                    rodtlon_val = 3;
                end
                2'b11: begin
                end
            endcase
            // When read command occurs, inject a one at
            // the proper calculated CL position
            odt_rd_shifter[rodtloff_calc] = 1;
        end
        if (odt_rd_shifter[0]) rodtlon = rodtlon_val;
        else if (|rodtlon) begin
            rodtlon = rodtlon - 1;
        end
        pre_odt = odt_int;
    end
end
//
// set the rtt state which determines the rtt value
//
always_comb begin : RTT_DEC
    //
    // If ODT is disabled, the odt state is HIGHZ
    //
    // If dynamic odt is operational, the odt pin controls how the RTT
    // value is set. If RTTPARK is enabled, rtt will default to the RTTPARK
    // value. If RTTPARK is disabled, rtt will default to HIGHZ. During
    // writes, rtt will switch to the RTTNOM value. During reads, rtt
    // will swith to HIGHZ. This switch is per the ODT Latency calculations.
    //
    rtt_state = (odt_en) ? (dll_en & dll_locked) ? (rodtl_on    ? HIGHZ
                                                 : (dodt_wr_on) ? RTTWR
                                                 : (dodt_on)    ? RTTNOM 
                                                 : rtt_park_en  ? RTTPARK
                                                 : HIGHZ)
                                                 : (odt         ? RTTNOM
                                                 : rtt_park_en  ? RTTPARK
                                                 : HIGHZ)
                                                 : HIGHZ; 
    //
    // Set the Termination Impedance
    //
    case(rtt_state)
        HIGHZ:   rtt = TERM_HIGHZ;
        RTTNOM:  rtt = rtt_nom_value; 
        RTTPARK: rtt = rtt_park_value; 
        RTTWR:   rtt = rtt_wr_value; 
    endcase
end
//
// MPR mode
//
always @(mr[3][2]) begin
    if (reset_n) begin
        if (mr[3][2]) begin
                                 log_msg (INFO, "MPR Mode Enabled", 2);
            if (|active_banks()) log_msg (ERR, $sformatf("Banks still active [%016b] when MPR mode enabled", active_banks()),0);
        end else                 log_msg (INFO, "MPR Mode Disabled", 2);
    end
end
//
// Write-Leveling Mode
//
always @(write_leveling_mode) begin : WR_LEVEL
    if (reset_n) begin
        //
        // Entering Write-Leveling Mode
        //
        if (write_leveling_mode) begin
            log_msg (INFO, "Write-Leveling Mode Enabled", 2);
            wr_lvl_mr1 = mr[1];
            if (rtt_wr_en) log_msg (ERR, $sformatf("RTT_WR(=%b) must be disabled when entering write-leveling mode", rtt_wr_value),0);
        end
        if (~write_leveling_mode) log_msg (INFO, "Write-Leveling Mode Disabled", 2);
    end
end
//
// Read Preamble Training Mode
//
always @(rd_pre_train_mode) begin : RD_PRE_TRAIN
    if (reset_n) begin
        if (rd_pre_train_mode) begin
            log_msg (INFO, "Read Preamble Training Mode Enabled", 2);
            if (~mr[3][2]) log_msg (ERR, $sformatf("MPR Operation(=%b) must be enabled when entering Read Preamble Training mode", mr[3][2]),0);
        end
        if (~rd_pre_train_mode) log_msg (INFO, " Read Preamble Training Mode Disabled", 2);
    end
end
//
// Maximum Power Saving Mode
//
always @(posedge mpsm) begin : MPSM_ENTRY
    fork
        begin
            mpsm_entry_pulse = 1;
            mped_cnt = TMPED;
        end
        begin
            // Check for TCKMPE
            #( TMOD_TCK*tck_avg + TCPDED*tck_avg) mpsm_entry_pulse = 0;
            if ( mped_cnt > 1 ) log_msg (ERR, "tCKMPE (min) Violated on MPSM Entry", -1);
        end
    join_any
end

always @(mpsm) begin : MPSM
    if (reset_n) begin
        //
        // Entering Maximum Power Saving Mode
        //
        if ( mpsm ) begin
            log_msg (INFO, "Entering Maximum Power Saving Mode", 2);
            odt_en = 0;                         // Disable ODT
        end
        //
        // Exiting Maximum Power Saving Mode
        //
        if ( ~mpsm ) begin
            log_msg (INFO, "Exiting Maximum Power Saving Mode", 2);
            mpxlh_cnt   = TMPX_LH/tck_avg;              // Load the tMPX_LH counter
            xmp_cnt     = TXMP/tck_avg;                 // Load the tXMP counter
            xmp_dll_cnt = TXMP_DLL/tck_avg + TXSDLL;    // Load the tXMP_DLL counter, TXMP + TXSDLL, TXSDLL is in nCK, XMP is in ns
            odt_en      = 1;                            // Enable ODT
            for (int i=0; i < ( max(TCKSRX_TCK, max(1, $ceil(TCKSRX/tck_avg)))); i++) begin
            if ( $rtoi(tck_avg_hist[i]) != $rtoi(tck_avg) ) begin
                log_msg (ERR, "tCKMPX (min) Violated on Exit MPSM", 0);
                break;  // got error - exit loop
            end
        end
        end
    end
end
//
// Maximum Power Saving Mode Exit
//
//always @(posedge cke && txpr_update_en == 1'b1) begin ---- TODO fix txpr
always @(posedge cke) begin
    if (mpsm & !cs_n) begin
        mr[4][1] = 0;                       // turn off the maximum power saving mode
        if (dll_en) mr[0][8] = 1;           // If the DLL is enabled, issue a DLL reset
    end
    if ( ts_txpr==0 ) ts_txpr  = $time;     // only fire when exiting reset
    tck_txpr = ck_t_cnt;
    //txpr_update_en = 1'b0;                // we only want to check txpr after coming out of reset
end


always @(posedge ck_t) begin
    if ( (power_down || self_ref) && cpded_cnt > 0 ) begin
        if ( cs_n==1'b0 ) log_msg (ERR, "tCPDED (min) Violated on Power Down Entry", 0);
        cpded_cnt <= cpded_cnt - 1;
    end
    if ( mpsm && mped_cnt > 0 ) begin
        if ( cs_n==1'b0 ) log_msg (ERR, "tMPED Violated, Only DES commands are allowed in Maximum Power-Save Mode (MPSM)", -1);
        mped_cnt <= mped_cnt - 1;                           // When mped_cnt is anything but 0, start the tMPED countdown
    end
    if (mpxlh_cnt   != 0) mpxlh_cnt   = mpxlh_cnt   - 1;    // When mpxlh_cnt is anything but 0, start the tMPX_LH countdown
    if (xmp_cnt     != 0) xmp_cnt     = xmp_cnt     - 1;    // When xmp_cnt is anything but 0, start the tXMP countdown
    if (xmp_dll_cnt != 0) xmp_dll_cnt = xmp_dll_cnt - 1;    // When xmp_dll_cnt is anything but 0, start the tXMP_DLL countdown

end

task exit_power_down(); begin : TSK_EXIT_POWER_DOWN
    fork
        begin txp_pulse = 1; end
        begin #(max(TXP_TCK*tck_avg, TXP))  txp_pulse = 0; end
    join_any
end endtask


always @(evt_clear_cache)         clear_cache();
always @(evt_load_cache_from_mem) load_cache_from_mem({bg,ba}, {cas_n_a15, we_n_a14, a[13:0]});

always @(evt_do_store) begin : DO_STORE
    integer start_bank, end_bank, bank, row;
    ts_store  = $time;
    tck_store = ck_t_cnt;
    if ( last_store == STO_ALL ) begin
        start_bank = 0;
        end_bank   = 16;
    end else begin
        start_bank = {bg,ba};
        end_bank   = start_bank + 1;
    end
    if ( nomem || nowmem ) begin
        log_msg (INFO, "Store operation while in NoMem or NoWMem is ignored", 1);
    end else begin
        for (int i=start_bank; i<end_bank; i++) begin
            if ( bank_status[i] == BANK_DIRTY ) begin
                // Bank has an active row, store row
                bank_status[i] = BANK_CLOSED;
                row  = bankN_active_row[i];
                bank = i;
                log_msg (INFO, $sformatf("Store operation, Bank=%2d", bank),2);
            end
        end
    end
end


task automatic clear_cache ();
    // *** This is to be used for debug purposes only ***
    // enhancement - select only one bank or a range
    // clears out cache
    begin : TSK_CLR_CACHE
        log_msg (WARN, "Clearing of cache", 1);
        -> evt_clear_cache;
        for (int i=0; i<16; i++) begin
            cache_mem[i] = {128*8{1'bx}};
        end
    end
endtask

task automatic write_cache (
    input [3:0]     bank,
    input [6:0]     col_addr,
    input [7:0]     data,
    input           mask
    );
    integer         row;
    //write data to respective bank cache to column - col_addr
    begin : TSK_WRITE_CACHE
        row = bankN_active_row[bank];
        if ( !mask ) begin
            -> evt_write_cache;
            cache_mem[bank][col_addr*8+:8] = data;
            log_msg (INFO, $sformatf("Memory Write [Bank=%02d Row=%04h Col=%02h] Data:%h", bank, row, col_addr, data),5);
        end
        if (bankN_AP[bank]==1 && wr_bc==0 ) begin
            log_msg (INFO, $sformatf("Auto Precharge [WRITE] Bank=%02d RA=%0h", bank, row), 3);
            log_msg (INFO, $sformatf("-------> Cache Bank=%02d Data=%h", bank, cache_mem[bank]), 5);
            pre_cnt[bank] <= $ceil(max(TRP/tck_avg, TRP_TCK));
            //write_cache_to_mem (bank);
            bankN_AP[bank] = 0;
            //bankN_active_row[bank] = -1;  // AP - clear row is active
        end
    end
endtask

function [RFF_BITS-1:0] get_cache_data(input [3:0] bank, input [6:0] col_addr);
    integer bank_x16;
    begin : FNC_GET_CACHE
        -> evt_get_cache_data;
        `ifdef X8
        get_cache_data = cache_mem[bank][col_addr*8+:64];
        `else
        bank_x16 = {1'b1,bank[2:0]};
        get_cache_data = {cache_mem[bank_x16][col_addr*8+:64],cache_mem[bank][col_addr*8+:64]};
        `endif
    end
endfunction

task automatic write_cache_to_mem (
    input [3:0] bank
    );
    integer row;
    //integer bank_x16;
    begin : TSK_WRITE_MEM
        // if in nomem  mode - do not write cache to memory
        // if in nowmem mode - write garbage to the memory - memory is corrupted
        // x16 also store the upper bank pair
        if ( nomem==1'b0  ) begin
            -> evt_write_mem;
            //bank_x16 = {1'b1,bank[2:0]};
            row = bankN_active_row[bank];  // get the active row in bank N
            bank_mem[bank][row]     = nowmem ? {128*8{1'bx}} : cache_mem[bank];
            log_msg (INFO, $sformatf("Write Mem Bank=%02d Row=%08h Data=%h", bank, row, cache_mem[bank]), 5);
          //`ifdef X16
          //  bank_mem[bank_x16][row] = nowmem ? {128*8{1'bx}} : cache_mem[bank_x16];
          //  log_msg (INFO, $sformatf("Write Mem Bank=%02d Row=%08h Data=%h", bank_x16, row, cache_mem[bank_x16]), 5);
          //`endif
        end
    end
endtask

task automatic load_cache_from_mem (
    input [3:0]  bg_ba,
    input [15:0] addr
    );
    integer bank, bank_x16, row_addr;
    begin : TSK_LOAD_CACHE
        bank     = bg_ba;
        row_addr = addr;
        bank_x16 = {1'b1,bg_ba[2:0]};
        // re-entrant task so we can spawn off multiple each with its own delay
        //#TRCD // wait TRCD time before mem->cache
        // FIX - CPM is memory corrupted if in nowmem
        if ( nomem==1'b0 && nowmem==1'b0 ) begin
            log_msg (INFO, $sformatf("Load Cache: BG,BA=%h  Addr=%h", bank, row_addr),5);
            -> evt_write_cache;
            bankN_active_row[bank] = row_addr;  // set the active row in bank N
            if ( bank_mem.exists(bank) ) begin
                if ( bank_mem[bg_ba].exists(row_addr)) begin
                    //[128*8-1:0] bank_mem [16][*]
                    cache_mem[bank] = bank_mem[bank][row_addr];
                end else begin
                    cache_mem[bank] = {128*8{1'bx}};
                end
            end else begin
                cache_mem[bank] = {128*8{1'bx}};
            end
            // destructive read array
            bank_mem[bank][row_addr] = {128*8{1'bx}};
            ////////////////////////////////////////////
            `ifdef X16
            ////////////////////////////////////////////
            // in x16 mode bg1 is not used
            bankN_active_row[bank_x16] = row_addr;  // set the active row in bank N
            if ( bank_mem.exists(bank) ) begin
                if ( bank_mem[bg_ba].exists(row_addr)) begin
                    //[128*8-1:0] bank_mem [16][*]
                    cache_mem[bank_x16] = bank_mem[bank_x16][row_addr];
                end else begin
                    cache_mem[bank_x16] = {128*8{1'bx}};
                end
            end else begin
                cache_mem[bank_x16] = {128*8{1'bx}};
            end
            // destructive read array
            bank_mem[bank_x16][row_addr] = {128*8{1'bx}};
            ////////////////////////////////////////////
            `endif
            ////////////////////////////////////////////
        end
    end
endtask

task automatic close_bank(input [3:0] bank); begin : TSK_CLOSE_BANK
    bankN_active_row[bank] = -1; // mark row as inactive
    log_msg (INFO, $sformatf("Closing Bank=%02d", bank),5);
    bankN_AP[bank] = 0;
    //`ifdef X16
    //bankN_active_row[{1'b1,bank[2:0]}] = -1; // mark row as inactive
    //`endif
end endtask


// calculate the absolute value of a real number
function real abs;
input arg;
real arg;
begin
    if (arg < 0.0)  abs = -1.0 * arg;
    else            abs = arg;
end
endfunction


//
// Timing Checks
//
task check_timing;
    input [1:0] grouping;
    input [BG_BITS+BA_BITS-1:0] bank;
    input [4:0] fromcmd;
    input [4:0] cmd;
    //reg err;
    integer twtr_tck;
begin : TSK_CHECK_TIMING
    -> evt_check_timing;
    if ( rd_is_bc4 ) begin
        // BC4 - tWTR_* 6 for all CLs and speed bins
        twtr_tck = TWTR_L_TCK;
    end else begin
        // BL8
        if ( rl == 'd10 ) begin
            twtr_tck = TWTR_L_TCK;
        end else begin
            twtr_tck = TWTR_S_TCK;
        end
    end
    casex ({grouping, fromcmd, cmd})
        // zq cal - TODO
        // power down - TODO
        // self refresh - TODO
        // MRS
        {DIFF_BANK , MRS, MRS} : begin 
            if (ck_t_cnt - tck_mrs < TMRD)
                log_msg (ERR, $sformatf("tMRD (min) Violated for Command= %s", cmd_str[cmd]),0);
        end
        {DIFF_BANK , MRS, RD } ,
        {DIFF_BANK , MRS, REF} ,
        {DIFF_BANK , MRS, PRE} ,
        {DIFF_BANK , MRS, ACT} ,
        {DIFF_BANK , MRS, ZQC} ,
        {SAME_BANK , MRS, RD } ,
        {SAME_BANK , MRS, REF} ,
        {SAME_BANK , MRS, PRE} ,
        {SAME_BANK , MRS, ACT} ,
        {SAME_BANK , MRS, ZQC} : begin 
            if ( (ck_t_cnt - tck_mrs) < TMOD_TCK )
                log_msg (ERR, $sformatf("tMOD (min) Violated for Command= %12s", cmd_str[cmd]),0);
        end
        // refresh
        {DIFF_BANK , REF, MRS} : ;
        {DIFF_BANK , REF, REF} : ;
        {DIFF_BANK , REF, PRE} : ;
        {DIFF_BANK , REF, ACT} : ;
        {DIFF_BANK , REF, ZQC} : ;

        // precharge
        {SAME_BANK , PRE, ACT} : begin 
            if ($time - ts_pre_bank[bank] < TRP-TJIT_PER_TOT)
                log_msg (ERR, $sformatf("tRP (min) Violated for Command= %12s, Bank= %d", cmd_str[0], bank),0);
        end
        {DIFF_BANK , PRE, MRS} ,
        {DIFF_BANK , PRE, REF} ,
        {DIFF_BANK , PRE, ZQC} : begin 
            if (($time - ts_pre) < TRP || (ck_t_cnt - tck_pre) < TRP_TCK )
                log_msg (ERR, $sformatf("tRP (min) Violated for Command= %12s", cmd_str[cmd]),0);
        end
        
        // activate 
        {SAME_BANK , ACT, PRE} : begin 
            if ($time - ts_act_bank[bank] < TRAS-TJIT_PER_TOT)
                log_msg (ERR, $sformatf("tRAS (min) Violated for Command= %12s, Bank= %d", cmd_str[cmd], bank),0);
        end
        {SAME_BANK , ACT, ACT} : begin 
            if ($time - ts_act_bank[bank] < TRC-TJIT_PER_TOT)
                log_msg (ERR, $sformatf("tRC (min) Violated for Command= %12s, Bank= %d  %t", cmd_str[0], bank, $time - ts_act_bank[bank]),0);
        end
        {SAME_BANK , ACT, WR } : begin // tRCD
            if ($time - ts_act_bank[bank] < TRCD-TJIT_PER_TOT)
                log_msg (ERR, $sformatf("tRCD (min) Violated for Command= %12s, Bank= %d  %t", cmd_str[cmd], bank, $time - ts_act_bank[bank]),0);
        end
        {SAME_BANK , ACT, RD } : begin // tRCD
            if ($time - ts_act_bank[bank] < TRCD-TJIT_PER_TOT)
                log_msg (ERR, $sformatf("tRCD (min) Violated for Command= %12s, Bank= %d  %t", cmd_str[cmd], bank, $time - ts_act_bank[bank]),0);
        end
        {DIFF_BANK , ACT, ACT} : begin
            for (int i=0; i<$size(ts_act_group); i++) begin
              if ( i==(bank>>BA_BITS) && (($time - ts_act_group[i] < TRRD_L) || (ck_t_cnt - tck_act_group[i] < TRRD_L_TCK)))
                log_msg (ERR, $sformatf("tRRD_L (min) Violated for Command= %12s, Bank= %d", cmd_str[0], bank),0);
            end
        end
        {DIFF_GROUP, ACT, ACT} : begin 
            for (int i=0; i<$size(ts_act_group); i++) begin
              if (i!=(bank>>BA_BITS) && (($time - ts_act_group[i] < TRRD_S) || (ck_t_cnt - tck_act_group[i] < TRRD_S_TCK)))
                log_msg (ERR, $sformatf("tRRD_S (min) Violated for Command= %12s Bank= %d", cmd_str[0], bank),0);
            end
        end
        {DIFF_BANK , ACT, REF} : begin 
            if ($time - ts_act < TRC-TJIT_PER_TOT)
                log_msg (ERR, $sformatf("tRC (min) Violated for Command= %12s", cmd_str[cmd]),0);
        end
        // writes, reads
        {DIFF_BANK , RD , WR },
        {DIFF_BANK , RD , RD } : begin
            for (int i=0; i<$size(tck_write_group); i++) begin
              if ( i==(bank>>BA_BITS) && ((ck_t_cnt - tck_read_group[i]) < TCCD_L) )
                log_msg (ERR, $sformatf("tCCD_L (min) Violated for Command= %12s, Bank= %d", cmd_str[cmd], bank),0);
            end
        end
        {DIFF_BANK , WR , RD } : begin
            for (int i=0; i<$size(tck_write_group); i++) begin
              if ( i==(bank>>BA_BITS) && ((ck_t_cnt - tck_write_group[i]) < TCCD_L) )
                log_msg (ERR, $sformatf("tCCD_L (min) Violated for Command= %12s, Bank= %d", cmd_str[cmd], bank),0);
              if ( i==(bank>>BA_BITS) && ((ck_t_cnt - tck_write_end_group[i]) <  TWTR_L_TCK) )
                log_msg (ERR, $sformatf("tWTR_L (min) Violated for Command= %12s, Bank=%d", cmd_str[cmd], bank),0);
            end
        end
        {DIFF_BANK , WR , WR } : begin
            for (int i=0; i<$size(tck_write_group); i++) begin
              if ( i==(bank>>BA_BITS) && ((ck_t_cnt - tck_write_group[i]) < TCCD_L) )
                log_msg (ERR, $sformatf("tCCD_L (min) Violated for Command= %12s, Bank= %d", cmd_str[cmd], bank),0);
            end
        end
        {DIFF_GROUP, RD , WR },
        {DIFF_GROUP, RD , RD } : begin
            for (int i=0; i<$size(tck_read_group); i++) begin
              if ( i!=(bank>>BA_BITS) && ((ck_t_cnt - tck_read_group[i]) < TCCD_S) )
                log_msg (ERR, $sformatf("tCCD_S (min) Violated for Command= %12s, Bank=%d", cmd_str[cmd], bank),0);
            end
        end
        {DIFF_GROUP, WR , RD } : begin
            for (int i=0; i<$size(tck_write_group); i++) begin
              if ( i!=(bank>>BA_BITS) && ((ck_t_cnt - tck_write_group[i]) < TCCD_S) )
                log_msg (ERR, $sformatf("tCCD_S (min) Violated for Command= %12s, Bank=%d", cmd_str[cmd], bank),0);
              if ( i!=(bank>>BA_BITS) && ((ck_t_cnt - tck_write_end_group[i]) <  TWTR_S_TCK) )
                log_msg (ERR, $sformatf("tWTR_S (min) Violated for Command= %12s, Bank=%d", cmd_str[cmd], bank),0);
            end
        end
        {DIFF_GROUP, WR , WR } : begin
            for (int i=0; i<$size(tck_write_group); i++) begin
              if ( i!=(bank>>BA_BITS) && ((ck_t_cnt - tck_write_group[i]) < TCCD_S) )
                log_msg (ERR, $sformatf("tCCD_S (min) Violated for Command= %12s, Bank=%d", cmd_str[cmd], bank),0);
            end
        end
        {SAME_BANK , RD , PRE} : begin 
            if (($time - ts_read[bank] < TRTP-TJIT_PER_TOT) || (ck_t_cnt - tck_read[bank] < TRTP_TCK))
                log_msg (ERR, $sformatf("tRTP (min) Violated for Command= %12s, Bank %d", cmd_str[cmd], bank),0);
        end

    endcase
end
endtask


task log_msg (input e_msg flag, input string msg, input integer int_level);
    string key;
    integer level;
    begin : TSK_log_msg
        string msg_str;
        if ( VERBOSE_LEVEL!=999 && (reset_n !== 1'b0 && ten===1'b0 && mpsm===1'b0 || int_level<0) ) begin
            level = abs(int_level);
            if ( flag == INFO ) begin
                -> evt_info;
                info_count = info_count + 1;
                if ( level <= VERBOSE_LEVEL ) $display("[CM] %m INFO   : Time: %t, %s", $time, msg);
                info_list.push_back(msg);
            end else if ( flag == WARN ) begin
                -> evt_warning;
                warning_count = warning_count + 1;
                if ( level <= VERBOSE_LEVEL || STOP_ON_WARN ) $display("[CM] %m WARNING: Time: %t, %s", $time, msg);
                warn_list.push_back(msg);
                if ( STOP_ON_WARN ) $finish;
            end else begin
                -> evt_error;
                error_count = error_count + 1;
                $display("[CM] %m ERROR  : Time: %t, %s", $time, msg);
                err_list.push_back(msg);
                if ( STOP_ON_ERROR ) $finish;
            end
        end
    end
endtask



`ifdef SKIP_MODEL_SPECIFY

`else
specify
`ifdef DLL_OFF
    specparam
        tSetupAddrCmd = 115,
        tHoldAddrCmd  = 140;
`elsif BIN800
    specparam 
        tSetupAddrCmd = 115,
        tHoldAddrCmd  = 140;
`elsif BIN1333
    specparam 
        tSetupAddrCmd = 115,
        tHoldAddrCmd  = 140;
`elsif BIN1600
    specparam 
        tSetupAddrCmd = 115,
        tHoldAddrCmd  = 140;
`elsif BIN1866
    specparam 
        tSetupAddrCmd = 100,
        tHoldAddrCmd  = 125;
`elsif BIN2133
    specparam 
        tSetupAddrCmd = 80,
        tHoldAddrCmd  = 105;
`elsif BIN2400
    specparam 
        tSetupAddrCmd = 62,
        tHoldAddrCmd  = 87;
`else
    specparam 
        tSetupAddrCmd = 115,
        tHoldAddrCmd  = 140;
`endif
    //
    // Reset Timing
    //
`ifdef FAST_SIM
    $setuphold(posedge reset_n, cke, 5_000, 5_000);             // cke -> reset_n
`else
     $setuphold(posedge reset_n, cke, 10_000, 500_000_000);     // cke -> reset_n 
`endif
    $width(negedge reset_n, 200_000_000, 200_000_000);
    //
    // Command/Address Setup and Hold Timing
    //
    $setuphold(posedge ck_t &&& reset_n, cke,       tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, act_n,     tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, ras_n,     tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, cas_n_a15, tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, we_n_a14,  tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, odt,       tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, ba,        tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, a,         tSetupAddrCmd, tHoldAddrCmd);
    $setuphold(posedge ck_t &&& reset_n, bg,        tSetupAddrCmd, tHoldAddrCmd);
endspecify
`endif

// pragma protect end

endmodule

