# //---------------------------------------------------------------------------------
# // 
# //  patch_ddr4mig_mram.tcl
# //  $Revision: 1.1.2.73 $
# //  $Date: 2019/09/21 18:54:49 $
# //  Description: Everspin ST-MRAM DDR4 Patch Script for Xilinx MIG 
# //
# //---------------------------------------------------------------------------------
# // 
# //  This confidential and proprietary software may be used only as authorized 
# //  by the included licensing agreement from Everspin titled 'EverspinSLA.txt'.     
# //                                                                               
# //                   Copyright 2017 Everspin Technologies, Inc
# //                                                                               
# //  The entire notice above must be reproduced and the Everspin SLA included for 
# //  all authorized copies.                                                                 
# //
# //---------------------------------------------------------------------------------
# //
# //
# //--------------------------------------------------------------------------------- 
# //--------------------------------------------------------------------------------- 
# // 
# // USER REQUIRED SETTINGS 
# //
# // It is recommended that user does NOT Change the following settings


set mig_ip_name "ddr4_0"
# Internal test
#set lola 1
set lola 0
#set device "DRAM"
set device "MRAM"
set ddr "ddr4"
# Internal Debugging
set MR_update 1
# X16
set ddr_part_width x16
# Recommended mem order
set mem_addr_order "ROW_BANK_COLUMN"
# For Debugging 
set verbose_patching 0
set Regression_test 0

set Ecc 1
set Ecc_Autocorrect 1


# // Set mig_no_mem_en=1 to keep the device in NOMEM mode. If set to 0, nomem will only be used for calibration. 
set mig_no_mem_en 0

# // mig_timing_set = 0 ----> First Silicon /w DRAM timing merge
# // mig_timing_set = 1 ----> First Silicon timing  
# // mig_timing_set = 2 ----> ES / CS Spec timing
# // mig_timing_set = 3 ----> S2
set mig_timing_set 3
# // NOTE: BE SURE THE DEVICE SELECTED IN THE CUSTOM PARTS FILE MATCHES THE TIMING SET.  WS for WS, ES for ES.

set vivado_version [version -short]



variable script_file
set script_file "patch_ddr4mig_mram.tcl"

# Help information for this script
# It is recommended that user does not change the default settings
proc help {} {
  variable script_file
  puts "Syntax:\n"
  puts "$script_file"
  puts "It is recommended that user does not change the default settings\n"
  puts "$script_file -tclargs \[--help\]\n"            
  puts "$script_file -tclargs \[--USE_Lola\]\n"
  puts "$script_file -tclargs \[--No_Lola\]\n"
  puts "$script_file -tclargs \[--Verbose_patching\]\n"
  puts "$script_file -tclargs \[--USE_DRAM\]\n"        
  puts "$script_file -tclargs \[--USE_MRAM\]\n"
  puts "$script_file -tclargs \[--USE_DDR4\]\n"
  puts "$script_file -tclargs \[--USE_DDR3\]\n"
  puts "$script_file -tclargs \[--USE_x8\]\n"
  puts "$script_file -tclargs \[--USE_x16\]\n"
  puts "$script_file -tclargs \[--USE_ROW_BANK_COLUMN\]\n"
  puts "$script_file -tclargs \[--USE_ROW_BANK_COLUMN_BANK\]\n"
  puts "$script_file -tclargs \[--No_mem_en\]\n"  
  puts "$script_file -tclargs \[--MR_update\]\n"
  puts "$script_file -tclargs \[--Regression_test\]\n"
  puts "$script_file -tclargs \[--USE_ECC\]\n"
  puts "$script_file -tclargs \[--USE_ECC_AUTOCORRECT\]\n"
  puts "-------------------------------------------------------------------------\n"
  exit 0
}



foreach i $::argc {puts $i}
if { $::argc > 0 } {
  for {set i 0} {$i < [llength $::argv]} {incr i} {
    set option [string trim [lindex $::argv $i]]
    #puts "$i $option"
    switch -exact -- $option {
      "--help"                { help }
      "--USE_Lola"            { set lola 1 }
      "--No_Lola"             { set lola 0 }
      "--Verbose_patching"    { set verbose_patching 1 }
      "--USE_DRAM"            { set device "DRAM"}
      "--USE_MRAM"            { set device "MRAM"}
      "--USE_DDR4"            { set ddr    "ddr4"}
      "--USE_DDR3"            { set ddr    "ddr3"}
      "--USE_x8"              { set ddr_part_width     "x8"}
      "--USE_x16"             { set ddr_part_width     "x16"}
      "--USE_ROW_BANK_COLUMN"       { set mem_addr_order "ROW_BANK_COLUMN"}
      "--USE_ROW_BANK_COLUMN_BANK"  { set mem_addr_order "ROW_BANK_COLUMN_BANK"}
      "--MR_update"           { set MR__update 1}
      "--No_mem_en"           { set mig_no_mem_en 1}
      "--Regression_test"     { set Regression_test 1}
      "--USE_ECC"             { set Ecc 1}
      "--USE_ECC_AUTOCORRECT" { set Ecc_Autocorrect 1}
        default {
        if { [regexp {^-} $option] } {
          puts "ERROR: Unknown option '$option' specified, please type '$script_file -tclargs --help' for usage info.\n"
          return 1
        }
      }
    }
  }
}

switch ${vivado_version} {
    "2016.4" {
        switch ${ddr} {
            "ddr3" {
                set mig_ver  "1.4"
                set file_ver "1_4"
                set mk_col_width 6
            }
            "ddr4" {
                set mig_ver  "2.1"
                set file_ver "2_1"
                set mk_col_width 7
            }
            default {
                set mig_ver  0
                set file_ver 0
                set mk_col_width 0
            }
        }
    }   
    "2017.2" - "2017.4" - "2018.1" - "2018.3" {
        switch ${ddr} {
            "ddr3" {
                set mig_ver  "1.4"
                set file_ver "1_4"
                set mk_col_width 6
            }
            "ddr4" {
                set mig_ver  "2.2"
                set file_ver "2_2"
                set mk_col_width 7
            }
            default {
                set mig_ver  0
                set file_ver 0
                set mk_col_width 0
            }
        }
    }
    default {
        set mig_ver  0
        set file_ver 0
        set mk_col_width 0
    }
}




if {$Regression_test} {
    set mig_ip_name "${ddr}_0"
    open_project ./fork/fork.xpr
}

puts "INFO: \$mig_ip_name = $mig_ip_name"




# //--------------------------------------------------------------------------------- 


proc puts_line_patched {this_line this_out_file} {
      global ddr
      global verbose_patching
      if { $verbose_patching } {
	      if {$ddr == "ddr3"} {
                 puts $this_out_file "   // Parameter patched to support Everspin 256Mb DDR3 ST-MRAM."
              }
	      if {$ddr == "ddr4"} {
                 puts $this_out_file "   // Parameter patched to support Everspin 1Gb DDR4 ST-MRAM."
              }
	      puts $this_out_file "   //${this_line}"
      }
}


proc puts_line_added {this_line this_out_file} {
      global verbose_patching
      puts $this_out_file "${this_line}"
      if { $verbose_patching } {
              puts $this_out_file "   // patched to support Performance "
      }
}


proc puts_line_removed {this_line this_out_file} {
      global ddr
      global verbose_patching
      if { $verbose_patching } {
	      if {$ddr == "ddr3"} {
	         puts $this_out_file "   // Line removed  to support Everspin 256Mb DDR3 ST-MRAM."
              }
	      if {$ddr == "ddr4"} {
	         puts $this_out_file "   // Line removed  to support Everspin 1Gb DDR4 ST-MRAM."
              }
	      puts $this_out_file " "
      }
}





proc patch_mig { mig_name no_mem_en timing_set lola device MR_update } {
  global ddr
  global ddr_part_width
  global mem_addr_order
  global mig_ver
  global file_ver
  global vivado_version
  global mk_col_width
  global Ecc
  global Ecc_Autocorrect
  puts "=======================================================================";
  puts "||  Everspin ST-MRAM DDR Patch Script for Xilinx MIG v${mig_ver} (${vivado_version})  ||";
  puts "=======================================================================";
  puts "  Target Device:  EMD4E001G08AG1-125-ES  WS1                           ";

  set this_script [info script]
  puts "INFO: ddr type is $ddr , timing set is $timing_set this is $this_script"

  #set mig_ver "2.2"

# // New parameters
    #set mk_col_width 	{7}
    set mk_mr0sub_nomem	{14'b1}
    # set mk_mr0sub 		{14'b0}
    set mk_mr0sub 		{13'b0000000000100}
    #  updated per IBM DLL reset issue 

    switch ${ddr} {

      "ddr3" {
	switch $timing_set {
		
		2 {
			set timing_set_name "ES / CS Spec"
			set mk_tfaw_ns		{190.0}
			set mk_trp_ns		{72.0}
			if {$ddr_part_width == "x8"} {
				set mk_trrd_ns		{36.0}
                        }
			if {$ddr_part_width == "x16"} {
			        set mk_trrd_l_ns	{36.0}
			        set mk_trrd_s_ns	{36.0}
                        }
			set mk_tras_ns		{111}
			set mk_trcd_ns		{102}
			set mk_trfc_ns		{261}
			set mk_twtr_l_ck	{6}
			set mk_twtr_s_ck	{6}
		}

		default {
				error "ERROR: Please select a timing set ($ddr $timing_set) at the top of this script: $this_script"
		}
	}
      }
      "ddr4" {
	switch $timing_set {
		
		0 {
			set timing_set_name "First Silicon / DRAM merged timing"
			set mk_tfaw_ns		{310.0}
			#set mk_trp_ns		{15.0}
			set mk_trp_ns		{40.0}
			if {$ddr_part_width == "x8"} {
				set mk_trrd_ns		{10.0}
                        }
			if {$ddr_part_width == "x16"} {
			        set mk_trrd_l_ns	{10.0}
			        set mk_trrd_s_ns	{10.0}
                        }
			set mk_tras_ns		{114.5}
			set mk_trcd_ns		{107.0}
			set mk_trfc_ns		{351}
			set mk_twtr_l_ck	{6}
			set mk_twtr_s_ck	{6}
		}

		1 {
			set timing_set_name "First Silicon"
			set mk_tfaw_ns		{310.0}
			#set mk_trp_ns		{5.0}
			set mk_trp_ns		{40.0}
			if {$ddr_part_width == "x8"} {
				set mk_trrd_ns		{10.0}
                        }
			if {$ddr_part_width == "x16"} {
			        set mk_trrd_l_ns	{10.0}
			        set mk_trrd_s_ns	{10.0}
                        }
			set mk_tras_ns		{114.5}
			set mk_trcd_ns		{107.0}
			set mk_trfc_ns		{351}
			set mk_twtr_l_ck	{6}
			set mk_twtr_s_ck	{4}
		}

		2 {
			set timing_set_name "ES / CS Spec"
			set mk_tfaw_ns		{240.0}
			#set mk_trp_ns		{5.0}
                        # trc must be 150ns but Xilinx does not have a trc parameter.
                        # First try would be: trp = trc - tras = 150ns - 103ns = 47ns
                        # But we can also use trp = trc - trcd - min(taa,twr) = 150ns - 95ns - 15ns = 40ns
			set mk_trp_ns		{40.0}
			if {$ddr_part_width == "x8"} {
				set mk_trrd_ns		{10.0}
                        }
			if {$ddr_part_width == "x16"} {
			        set mk_trrd_l_ns	{10.0}
			        set mk_trrd_s_ns	{10.0}
                        }
			set mk_tras_ns		{103}
			set mk_trcd_ns		{95}
			set mk_trfc_ns		{351}
			set mk_twtr_l_ck	{6}
			set mk_twtr_s_ck	{6}
		}

              	3 {
			set timing_set_name "ES / CS Spec"
			set mk_tfaw_ns		{240.0}
			#set mk_trp_ns		{5.0}
                        # trc must be 150ns but Xilinx does not have a trc parameter.
                        # First try would be: trp = trc - tras = 190ns - 103ns = 47ns
                        # But we can also use trp = trc - trcd - min(taa,twr) = 190ns - 95ns - 15ns = 40ns
			set mk_trp_ns		{80.0}
			if {$ddr_part_width == "x8"} {
				set mk_trrd_ns		{10.0}
                        }
			if {$ddr_part_width == "x16"} {
			        set mk_trrd_l_ns	{10.0}
			        set mk_trrd_s_ns	{10.0}
                        }
			set mk_tras_ns		{103}
			set mk_trcd_ns		{135}
			set mk_trfc_ns		{351}
			set mk_twtr_l_ck	{6}
			set mk_twtr_s_ck	{6}
		}














		default {
				error "ERROR: Please select a timing set ($ddr $timing_set) at the top of this script: $this_script"
		}
	  }
	}
	default {
			error "ERROR: Please select a timing set ($ddr $timing_set) at the top of this script: $this_script"
	}
    }


  set mig_ip [get_ips $mig_name]

  set mig_ip_filename [get_property {IP_FILE} $mig_ip]
  set mig_ip_file [get_files $mig_ip_filename]

  #==============================================================
  # Step 1: Disable synthesis checkpoint.
  #==============================================================
 
# save_project_as project_test ./project_test -force
  
  puts "debug ${mig_ip_file}"

 set_property IS_LOCKED false [get_files "${mig_name}.xci"] 

 set_property {GENERATE_SYNTH_CHECKPOINT} {0} $mig_ip_file
  
  #==============================================================
  # Step 2: Reset the IP outputs and generate hdl
  #==============================================================
  reset_target all $mig_ip
  generate_target all $mig_ip
  
  #====================================================================================================================================
  # Step 3: Verify the MIG version.
  #====================================================================================================================================
  set file [get_files "${mig_name}.sv"]
  set filename [get_property {NAME} $file]

  if {$ddr == "ddr3"} {
     set match_sdram "DDR3_SDRAM"
     set ap_col_match_addr "6'h38"
  }
  if {$ddr == "ddr4"} {
     set match_sdram "DDR4_SDRAM"
     set ap_col_match_addr "7'h78"
  }

  set in_file  [open $filename r]

  # line-by-line, read the original file
  set ver_n 0
  while {[gets $in_file line] != -1} {
    set new_line $line
    
    # Look for mig version
    if {[string match -nocase "*CORE_GENERATION_INFO*" $new_line] && [string match -nocase "*ipVersion=${mig_ver}*" $new_line] && [string match "*${match_sdram}*" $new_line]} {
      incr ver_n 1
    }

  } 

  close $in_file

  if {$ver_n == 0} {
	error "*** ERROR: MIG version check failed.  Please ensure MIG v${mig_ver} is used."
  } elseif {$ver_n == 1} {
	puts "*** MIG version check passed!"
  } elseif {$ver_n > 1} {
  	puts $ver_n
	error "*** ERROR: Unable to determine MIG version"
  }

  #====================================================================================================================================
  # Step 3: Find MIG DDR period (tCK) and calculate the timing values (nCK) for the controller
  #====================================================================================================================================
    set target_file "${mig_name}_${ddr}.sv"
    puts "--> ${target_file}"
	set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	set filename [get_property {NAME} $file]

	set in_file  [open $filename r]

	# line-by-line, read the original file
	set got_tck 0
	while {[gets $in_file line] != -1} {
		set new_line $line

		# Look for mig tCK
	  	if {[string match {*parameter*} $new_line] && [string match {*tCK*} $new_line] && [string match -nocase "*// Memory clock period*" $new_line]} {
	 		set linesplit [split $new_line {"=",","}]
 			lassign $linesplit junk mig_tck morejunk
	  		incr got_tck 1
	  	}
	} 

	close $in_file

	if {$got_tck == 0} {
	error "*** ERROR: MIG tCK was not found."
	} elseif {$got_tck == 1} {
	puts "*** Found MIG tCK = $mig_tck ps"
	} elseif {$got_tck > 1} {
		puts $got_tck
	error "*** ERROR: Unable to determine MIG tCK."
	}

	set tfaw_ck [ expr { int(($mk_tfaw_ns * 1000.0 / $mig_tck) + 0.99 ) }]
	set trp_ck [ expr { int(($mk_trp_ns * 1000.0 / $mig_tck) + 0.99 ) }]
	if {$ddr_part_width == "x8"} {
		set trrd_ck [ expr { int(($mk_trrd_ns * 1000.0 / $mig_tck) + 0.99 ) }]
        }
	if {$ddr_part_width == "x16"} {
		set trrd_l_ck [ expr { int(($mk_trrd_l_ns * 1000.0 / $mig_tck) + 0.99 ) }]
		set trrd_s_ck [ expr { int(($mk_trrd_s_ns * 1000.0 / $mig_tck) + 0.99 ) }]
        }
	set tras_ck [ expr { int(($mk_tras_ns * 1000.0 / $mig_tck) + 0.99 ) }]
	set trcd_ck [ expr { int(($mk_trcd_ns * 1000.0 / $mig_tck) + 0.99 ) }]
	set trfc_ck [ expr { int(($mk_trfc_ns * 1000.0 / $mig_tck) + 0.99 ) }]
	set twtr_l_ck $mk_twtr_l_ck
	set twtr_s_ck $mk_twtr_s_ck

	puts "*** Using timing set = $timing_set_name"
	puts "*** tFAW   = $mk_tfaw_ns ns to $tfaw_ck nCK"
	puts "*** tRP    = $mk_trp_ns ns   to $trp_ck nCK"
	if {$ddr_part_width == "x8"} {
		puts "*** tRRD = $mk_trrd_ns ns  to $trrd_ck nCK"
        }
	if {$ddr_part_width == "x16"} {
		puts "*** tRRD_L = $mk_trrd_l_ns ns  to $trrd_l_ck nCK"
		puts "*** tRRD_S = $mk_trrd_s_ns ns  to $trrd_s_ck nCK"
        }
	puts "*** tRAS   = $mk_tras_ns ns to $tras_ck nCK"
	puts "*** tRCD   = $mk_trcd_ns ns to $trcd_ck nCK"
	puts "*** tRFC   = $mk_trfc_ns ns to $trfc_ck nCK"
	puts "*** tWTR_L =           $twtr_l_ck nCK"
	puts "*** tWTR_S =           $twtr_s_ck nCK"


 #====================================================================================================================================
  # Step 3.5: Examine if AXI is used or not in the xci file
  #====================================================================================================================================
set_property IS_LOCKED false [get_files "${mig_name}.xci"]  

 #set target_file "MRAM_DDR4_x16_AXI_v0.xci"
 #     puts "--> ${target_file}"
#	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
#	  set filename [get_property {NAME} $file]
         
 #set file [get_files MRAM_DDR4_x16_AXI_v0.xci]

#	  set in_file  [open $filename r]
#	  set out_file [open $tmp_filename w]


set axi_protocol 0
set memory_part_axi        "EMD4E001GAS1"
set memory_part_default    "MT40A512M16HA-075E"

set file [get_files "${mig_name}.xci"]
set filename [get_property {NAME} $file]

set in_file  [open $filename r]

 while {[gets $in_file line] != -1} {
 	    set new_line $line
	   	    
	    # Patch col_width
	    if {[string match {*\"BUSIFPARAM_VALUE.C0_DDR*_S_AXI.PROTOCOL\">AXI4<*} $new_line]} {
	     set axi_protocol 1}
	    }
            puts "\$axi_protocol = "
            puts  $axi_protocol
	  close $in_file








  #====================================================================================================================================
  # Step 4: Patch the controller files with the new values
  #====================================================================================================================================

	  #====================================================================================================================================
	  #rtl/ip_top/${mig_ip}_${ddr}.sv
	  #====================================================================================================================================
      set target_file "${mig_name}_${ddr}.sv"
      puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]

	  set mr0_oldwidth			{13'b}

	  # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
	    set line_patched 0
            set line_added 0
            



  	    if {[string match {*parameter*MEMORY_PART*} $new_line]} {
              set new_line "    parameter MEMORY_PART                       = \"EMD4E001GAS1\","     
              # puts $new_line 
              incr n 1
	      set line_patched 1
            }
  	    if {[string match {*parameter*MEMORY_DENSITY*} $new_line]} {
              if {$ddr=="ddr3"} {
                set new_line "    parameter MEMORY_DENSITY                    = \"0.25Gb\","
              }
              if {$ddr=="ddr4"} {
                set new_line "    parameter MEMORY_DENSITY                    = \"1Gb\","
              }
              # puts $new_line 
              incr n 1
	      set line_patched 1
            }

  	    if {[string match {*parameter*MEMORY_SPEED_GRADE*} $new_line]} {
              set new_line "    parameter MEMORY_SPEED_GRADE                = \"125\","     
              # puts $new_line 
              incr n 1
	      set line_patched 1
            }

            if {[string match {*parameter*MEM_ADDR_ORDER*=*} $new_line]} {
              set new_line "    parameter         MEM_ADDR_ORDER          = \"$mem_addr_order\","
              # puts $new_line 
              incr n 1
              set line_patched 1
            }

  	    if {[string match {*parameter*tRFC*=*} $new_line]} {
              if {$device=="DRAM" || $ddr=="ddr3"} {
                set new_line "    parameter         tRFC                    = $trfc_ck, //In DDR clock cycles"
              }     
              if {$device=="MRAM" && $ddr=="ddr4"} {
                set new_line "    parameter         tRFC                    = 830, //In DDR clock cycles"
              }
              # puts $new_line 
              incr n 1
	      set line_patched 1
            }





#Add status and control input output 
 	    if {[string match {*input*} $new_line]  && [string match {*sys_rst,*} $new_line]} {
              set new_line "
   // begin Status and control inputs and outputs
   input power_fail_has_scramed,
   output  cntr_power_fail_complete,
   output inflight_writes,
   // end of the patch"     
              # puts $new_line 
              incr n 1
	      set line_added 1
            }

# AXI, wire definition
            if {[string match {*wire*} $new_line]  && [string match {*c0_div_clk;*} $new_line]} {
	      if {$axi_protocol==1}  {
                set new_line "  wire  inflight_writes_axi;\n  wire inflight_writes_intfc;"     
                # puts $new_line 
                incr n 1
	        set line_added 1
	        }
            }

#AXI, assignment  
            if {[string match {*assign*} $new_line]  && [string match {*aclk*} $new_line] && [string match {*=*} $new_line] && [string match {*c0_div_clk;*} $new_line]} {
	      if {$axi_protocol==1}  {
                set new_line "   assign inflight_writes = inflight_writes_axi || inflight_writes_intfc;"     
                # puts $new_line 
                incr n 1
	        set line_added 1
	        }
            }

#tREFI  
            if {[string match {*parameter*} $new_line]  && [string match {*=*} $new_line] && [string match {*tREFI*} $new_line]} {
	        if {$device=="DRAM"} {
                set new_line "    parameter         tREFI                   = 5200, //In DDR clock cycles"
                }
                if {$device=="MRAM"} {
                set new_line "    parameter         tREFI                   = 17'h1ffff, //In DDR clock cycles"
                }
                incr n 1
	        set line_patched 1
            }



#AXI, port connection
            if {[string match {*.dbg_out*} $new_line]  && [string match {*(),*} $new_line]} {
	      if {$axi_protocol==1}  {
                set new_line "
   // begin Status and control inputs and outputs
   .power_fail_has_scramed   (power_fail_has_scramed),
   .cntr_power_fail_complete (cntr_power_fail_complete),
   .inflight_writes          (inflight_writes_intfc),
   .inflight_writes_axi      (inflight_writes_axi),
   // end of the patch"
                # puts $new_line 
	    } else {
              set new_line "
   // begin Status and control inputs and outputs
   .power_fail_has_scramed   (power_fail_has_scramed),
   .cntr_power_fail_complete (cntr_power_fail_complete),
   .inflight_writes          (inflight_writes),
   // end of the patch" 
                   }
            incr n 1
	    set line_added 1
            }

#AXI, port connection
           if {[string match {*.s_axi_rready*} $new_line]  && [string match {*(c0_ddr4_s_axi_rready),*} $new_line]} {
	     if {$axi_protocol==1}  {
               set new_line " .inflight_writes                        (inflight_writes_axi),"
               # puts $new_line 
               incr n 1
	       set line_added 1
	       }
            }


           if {[string match {*parameter*MR1*} $new_line]} {
               if {$ddr=="ddr3"} {
                  set new_line "    parameter         MR1                       = 13'b0_0000_0100_0100,"
               }
               if {$ddr=="ddr4" && $MR_update == 0} {
                # Old value, Functional  for Lola
                set new_line "    parameter         MR1                       = 13'b0_0011_0000_0001,"
               }
               if {$ddr=="ddr4" && $MR_update == 1} {
                # for lola regbank needs to be changed as well
                set new_line "    parameter         MR1                       = 13'b0_0000_0000_0101,"
               }

               # puts $new_line 
               incr n 1
	       set line_patched 1
            }

           
           if {[string match {*parameter*MR2*} $new_line]} {
               if {$ddr=="ddr3"} {
                  set new_line "    parameter         MR2                       = 13'b0_0000_0001_0000,"
               }
               if {$ddr=="ddr4"} {
                  set new_line "    parameter         MR2                       = 13'b0_0000_0000_0000,"
               }
               # puts $new_line 
               incr n 1
	       set line_patched 1
            }

           if {[string match {*parameter*MR3*} $new_line]} {
               if {$ddr=="ddr3"} {
                  set new_line "    parameter         MR3                       = 13'b0_0000_0000_0000,"
               }
               if {$ddr=="ddr4" && $device=="DRAM"} {
                  set new_line "    parameter         MR3                       = 13'b0_0000_1000_0000,"
               }
               if {$ddr=="ddr4" && $device=="MRAM" && $MR_update == 0} {
                  # A8 = Refresh command executes Store all
                  # A7:6 2'b10 = Store all bank stagger: 8 bank (official recommendation)
                  # A7:6 2'b00 = Store all bank stagger: 2 bank (grandfather exception)
                  set new_line "    parameter         MR3                       = 13'b0_0001_0000_0000,"
               }
              if {$ddr=="ddr4" && $device=="MRAM" && $MR_update == 1} {
                  # A8 = Refresh command executes Store all
                  # A7:6 2'b10 = Store all bank stagger: 8 bank (official recommendation)
                  # A7:6 2'b00 = Store all bank stagger: 2 bank (grandfather exception)
                  set new_line "    parameter         MR3                       = 13'b0_0001_1000_0000,"
               }


               # puts $new_line 
               incr n 1
	       set line_patched 1
            }

           if {[string match {*parameter*MR4*} $new_line]} {
               set new_line "    parameter         MR4                       = 13'b0_0000_0000_0000,"
               # puts $new_line 
               incr n 1
	       set line_patched 1
            }

           if {[string match {*parameter*MR5*} $new_line]} {
               if {$MR_update ==0} {
               set new_line "    parameter         MR5                       = 13'b0_0100_0100_0000,"
               }
               if {$MR_update ==1} {
               set new_line "    parameter         MR5                       = 13'b0_0100_1110_0000,"
               }
               # puts $new_line 
               incr n 1
	       set line_patched 1
            }
           if {[string match {*parameter*MR6*} $new_line]} {
               set new_line "    parameter         MR6                       = 13'b0_0000_0010_0010,"
               # puts $new_line 
               incr n 1
	       set line_patched 1
            }

    	    # Patch col_width
	    if {[string match {*parameter*} $new_line] && [string match {*integer*} $new_line] && [string match {*COL_WIDTH*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter integer COL_WIDTH               = $mk_col_width,"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    }
	    
	    # Patch tfaw
	    if {[string match {*parameter*} $new_line] && [string match {*tFAW*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter         tFAW                    = $tfaw_ck, //In DDR clock cycles"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    }

	    # # Patch twtr_l
	    if {[string match {*parameter*} $new_line] && [string match {*tWTR_L*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter         tWTR_L                  = $twtr_l_ck, //In DDR clock cycles"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    }
	    
	    # # Patch twtr_s
	    if {[string match {*parameter*} $new_line] && [string match {*tWTR_S*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter         tWTR_S                  = $twtr_s_ck, //In DDR clock cycles"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    } 

	    # # Patch trp
	    if {[string match {*parameter*} $new_line] && [string match {*tRP*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter         tRP                     = $trp_ck, //In DDR clock cycles"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    }
	    
	    if {$ddr_part_width == "x8"} {
	      # # Patch trrd
	      if {[string match {*parameter*} $new_line] && [string match {*tRRD*} $new_line] && [string match {*=*} $new_line]} {
	        set new_line "    parameter         tRRD                    = $trrd_ck, //In DDR clock cycles"
	        #puts $new_line
	        incr n 1
	        set line_patched 1
	      }
            }
	    if {$ddr_part_width == "x16"} {
	      # # Patch trrd_l
	      if {[string match {*parameter*} $new_line] && [string match {*tRRD_L*} $new_line] && [string match {*=*} $new_line]} {
	        set new_line "    parameter         tRRD_L                  = $trrd_l_ck, //In DDR clock cycles"
	        #puts $new_line
	        incr n 1
	        set line_patched 1
	      }

	      # # Patch trrd_s
	      if {[string match {*parameter*} $new_line] && [string match {*tRRD_S*} $new_line] && [string match {*=*} $new_line]} {
	        set new_line "    parameter         tRRD_S                  = $trrd_s_ck, //In DDR clock cycles"
	        #puts $new_line
	        incr n 1
	        set line_patched 1
	      }
            }
	    
	    # # Patch tras
	    if {[string match {*parameter*} $new_line] && [string match {*tRAS*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter         tRAS                    = $tras_ck, //In DDR clock cycles"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    }

	    # # Patch trcd
	    if {[string match {*parameter*} $new_line] && [string match {*tRCD*} $new_line] && [string match {*=*} $new_line]} {
	      set new_line "    parameter         tRCD                    = $trcd_ck, //In DDR clock cycles"
	      #puts $new_line
	      incr n 1
	      set line_patched 1
	    } 

	    # Patch mr0.  We need to expand the parm width to 14b to support the no mem bit.  nomem mode is persistant past cal if nomem_en is set at the top of this script.
	    if {[string match -nocase {*parameter*} $new_line] && [string match -nocase {*MR0*} $new_line] && [string match -nocase "*${mr0_oldwidth}*" $new_line]} {
	      if { $no_mem_en == 1 } {
	      	incr n [regsub -nocase -- $mr0_oldwidth $new_line $mk_mr0sub_nomem new_line]
	      } else {
                        set new_line "   parameter         MR0                       = 13'b0000000000100,"
                        incr n 1
                       # incr n [regsub -nocase -- $mr0_oldwidth $new_line $mk_mr0sub new_line]
	      }
	      #puts $new_line
	      set line_patched 1
	    }    

	    if { $line_patched } {
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }
            puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { ($n != 18 && $ddr=="ddr3" && $axi_protocol==0)} {
	    puts "*** ERROR: Failed to patch $filename as expected. Patched $n of expected 18 "
	  	error "ERROR!"
	  }
	  if { ($n != 24 && $ddr=="ddr4" && $axi_protocol==0)} {
	    puts "*** ERROR: Failed to patch $filename as expected. Patched $n of expected 24 "
	  	error "ERROR!"
	  }
	  if { ($n != 27 && $ddr=="ddr4" && $axi_protocol==1)} {
	    puts "*** ERROR: Failed to patch $filename as expected. Patched $n of expected 27 line(s)."
	  	error "ERROR!"
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."

 	  #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_mc.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_mc.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]

	  #============================================
	  #	Old settings generated by the MIG to search
	  #============================================
	  set trasf_width_old				{3}	

	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  	  set trasf_width_new				{6} 

	  # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
            

	    
	    # Patch trasf_width
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*tRASF*} $new_line] && [string match -nocase "*${trasf_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trasf_width_old $new_line $trasf_width_new new_line]
	      set line_patched 1
	    }

# Performance
             if {[string match -nocase {*,input*calDone*} $new_line]} {
                set new_line "
   // begin Status and control inputs and outputs
   ,input                   power_fail_has_scramed
   ,output reg              cntr_power_fail_complete
   ,output reg              inflight_writes          // status output
   ,input                   inflight_writes_ui       // status input"
               if {$Ecc_Autocorrect==1} {
                 append new_line "
   ,input                   enable_autocorrect_sync  // insync with clk_mem"
               }
               if {$axi_protocol==1}  {
                 append new_line "
   ,input                   inflight_writes_axi      // status input"
               }
               append new_line "
   // end Status and control inputs and outputs"
                incr n 1
                set line_added 1
              
             }
  
             if {$Ecc_Autocorrect==1 && [string match -nocase {*,output * eccSingle} $new_line]} {
               set new_line "   ,output     \[2*nCK_PER_CLK-1:0\]          eccDouble"
               incr n 1
               set line_added 1
             }

             if {[string match -nocase {*,input*fi_xor_wrdata_en} $new_line]} {
              set new_line "   ,input      \[DBAW-1:0\]                   fi_xor_wrdata_bufaddr"
              incr n 1
              set line_added 1
	    }


             if {[string match  {*synopsys translate_off*} $new_line]} {
              set new_line "
reg \[2:0\]          cmd_saved;

always @ (posedge clk) cmd_saved <= cmd;
wire a_mc_003 =    useAdr
    && accept
    && ((cmd_saved == 3'h0) || (cmd_saved == 3'h3))
    && |clks_counter 
    && (dBufAdr_value_seen == dBufAdr);


 always @ (posedge clk) begin
   if (rst_r1) begin
     clks_counter <= 'b0;
     dBufAdr_value_seen <= 'b0;
   end
   else if(    useAdr
            && accept
            && (cmd_saved == 3'h3)) begin
     assert property (~a_mc_003);
     clks_counter <= 4'b1;
     dBufAdr_value_seen <= dBufAdr;
   end
   else if (clks_counter > 4'h4) begin
     clks_counter <= 'b0;
   end
   else if (|clks_counter) begin
     clks_counter <= clks_counter + 1'b1;
   end
 end"
              incr n 1
              set line_added 1
	    }




             if {[string match -nocase {*wire*int_srxIss;*} $new_line]} {
              set new_line "reg        scram_writing_done_pulse;
reg        scram_writing_done_pulsed;
reg        scram_refReq_started;
reg        per_state_idle;
reg \[3:0\]          clks_counter;
reg \[DBAW-1:0\]     dBufAdr_value_seen;"
              incr n 1
              set line_added 1
	    }

           
           if {[string match  {*assign wrCAS = winWrite & tranSentC;*} $new_line] } {
              if {$axi_protocol==1}  {
               set new_line "

// begin Status and control outputs

 always @(posedge clk) begin
   if(rst_r1) begin
     inflight_writes               <= 'b0;
     scram_writing_done_pulse      <= 'b0;
     scram_writing_done_pulsed     <= 'b0;
     scram_refReq_started          <= 'b0;
     cntr_power_fail_complete      <= 'b0;
   end
   else begin
     inflight_writes          <=  (    ~&txn_fifo_empty
                                    || ~&cas_fifo_empty
                                    || |pages_open);
     if (    power_fail_has_scramed
          && !(    ~&txn_fifo_empty
                || ~&cas_fifo_empty
                || inflight_writes_ui
                || inflight_writes_axi)
          && !scram_writing_done_pulse
          && !scram_writing_done_pulsed
          && !scram_refReq_started
          && per_state_idle
          && !cntr_power_fail_complete) begin
       scram_writing_done_pulse      <= 1'b1;
     end
     else if (scram_writing_done_pulse) begin
       scram_writing_done_pulse      <= 'b0;
       scram_writing_done_pulsed     <= 1'b1;
     end
     else if (    scram_writing_done_pulsed
               && !cntr_power_fail_complete
               && refReq) begin
         scram_writing_done_pulsed   <= 'b0;
         scram_refReq_started        <= 'b1;
     end
     else if (    scram_refReq_started
               && !cntr_power_fail_complete
               && !refReq) begin
         scram_refReq_started        <= 'b0;
         cntr_power_fail_complete    <= 1'b1;
     end
     else if (    cntr_power_fail_complete
               && !power_fail_has_scramed) begin
         cntr_power_fail_complete     <= 'b0;
     end
   end
 end
 
// end Status and control outputs

assign apgr = ap || (col == $ap_col_match_addr);"
    } else {
           set new_line "

// begin Status and control outputs

 always @(posedge clk) begin
   if(rst_r1) begin
     inflight_writes               <= 'b0;
     scram_writing_done_pulse      <= 'b0;
     scram_writing_done_pulsed     <= 'b0;
     scram_refReq_started          <= 'b0;
     cntr_power_fail_complete      <= 'b0;
   end
   else begin
     inflight_writes          <=  (    ~&txn_fifo_empty
                                    || ~&cas_fifo_empty
                                    || |pages_open);
     if (    power_fail_has_scramed
          && !(    ~&txn_fifo_empty
                || ~&cas_fifo_empty
                || inflight_writes_ui)
          && !scram_writing_done_pulse
          && !scram_writing_done_pulsed
          && !scram_refReq_started
          && per_state_idle
          && !cntr_power_fail_complete) begin
       scram_writing_done_pulse      <= 1'b1;
     end
     else if (scram_writing_done_pulse) begin
       scram_writing_done_pulse      <= 'b0;
       scram_writing_done_pulsed     <= 1'b1;
     end
     else if (    scram_writing_done_pulsed
               && !cntr_power_fail_complete
               && refReq) begin
         scram_writing_done_pulsed   <= 'b0;
         scram_refReq_started        <= 'b1;
     end
     else if (    scram_refReq_started
               && !cntr_power_fail_complete
               && !refReq) begin
         scram_refReq_started        <= 'b0;
         cntr_power_fail_complete    <= 1'b1;
     end
     else if (    cntr_power_fail_complete
               && !power_fail_has_scramed) begin
         cntr_power_fail_complete     <= 'b0;
     end
   end
 end
 
// end Status and control outputs

assign apgr = ap || (col == $ap_col_match_addr);"
    }
              incr n 1
              set line_added 1
	    }

#periodic_config defined differently when for lola (lola==1)

           if {[string match  {*wire*periodic_config*=*} $new_line] && $lola==0 } {
              set new_line "wire \[31:0\] periodic_config = (( PER_RD_INTVL == 0 )
                                 || cntr_power_fail_complete
                                 || power_fail_has_scramed ) ? 32'b0 : { 30'b0, calDone, calDone };"
              incr n 1
              set line_patched 1
	    }



             
             if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*bagr;*} $new_line]} {
              set new_line "wire        apgr;\nwire \[3:0\]           pages_open;"
              incr n 1
              set line_added 1
	    }


             if {[string match -nocase {*,.ap*(ap)*} $new_line]} {
              set new_line "         ,.ap       (apgr)"
              incr n 1
              set line_patched 1
	    }


             if {[string match -nocase {*,.mcCKt*(mcCKt)*} $new_line]} {
              set new_line "   ,.scram_writing_done_pulse (scram_writing_done_pulse)"
              incr n 1
              set line_added 1
	    }

             if {[string match -nocase {*,.cas_fifo_full*} $new_line]} {
               set new_line "         ,.pages_open          (pages_open\[bg\])"
               if {$Ecc_Autocorrect==1} {
                 append new_line "
         // to autocorrect     
         ,.cas_rd_txn2autocor (cas_rd_txn2autocor\[bg*TXN_FIFO_MAX_WIDTH+:TXN_FIFO_MAX_WIDTH\])
         ,.cas_rd_Buf2autocor (cas_rd_Buf2autocor\[bg*DBAW+:DBAW\])
         ,.cas_rd_cmd2autocor (cas_rd_cmd2autocor\[bg\])
         ,.cas_ac_dis2autocor (cas_ac_dis2autocor\[bg\])
         ,.autocor2wr_txn     (autocor2wr_txn)
         ,.autocor2wr_Buf     (autocor2wr_Buf)
         ,.autocor2wr_ac_dis  (autocor2wr_ac_dis)
         ,.autocor2wr_req     (autocor2wr_req\[bg\])
         ,.wr_ack2autocor     (grWr_ack2autocor\[bg\])"
               }
               incr n 1
               set line_added 1
	     }

             if {[string match -nocase {*,.gt_data_ready*} $new_line]} {
              set new_line "     ,.per_state_idle     (per_state_idle)"
              incr n 1
              set line_added 1
	    }

             if {$Ecc_Autocorrect==1 && [string match -nocase {wire * cmd_l_rank_cas*} $new_line]} {
               set new_line "
// already declared above
//wire \[4*CSBITS-1:0\]   pages_open;
// cke related, not needed
//wire \[4*CSBITS-1:0\]   rank_fifo_empty_all;
//wire \[4*CSBITS-1:0\]   rank_activity;
//wire                  ctl_block_req;

localparam  TXN_FIFO_MAX_WIDTH = 53;
       // to autocorrect
wire \[4*TXN_FIFO_MAX_WIDTH-1:0\]  cas_rd_txn2autocor;
wire               \[4*DBAW-1:0\]  cas_rd_Buf2autocor;
wire                      \[3:0\]  cas_rd_cmd2autocor;
wire                      \[3:0\]  cas_ac_dis2autocor;
wire   \[TXN_FIFO_MAX_WIDTH-1:0\]  autocor2wr_txn;
wire                 \[DBAW-1:0\]  autocor2wr_Buf;
wire                             autocor2wr_ac_dis;
wire                      \[3:0\]  autocor2wr_req;
wire                      \[3:0\]  grWr_ack2autocor;
wire                             wr_ack2autocor;
wire      \[PAYLOAD_WIDTH*8-1:0\]  autocor_wr_data;

wire      \[PAYLOAD_WIDTH*8-1:0\]  muxed_wr_data;
wire   \[PAYLOAD_DM_WIDTH*8-1:0\]  muxed_wr_data_mask;
wire                             rd_outstanding_fifo_empty;
wire                             autocorrecting_fifo_empty;
wire                             rd_outstanding_fifo_full;
wire                             autocorrecting_fifo_full;

assign wr_ack2autocor = |grWr_ack2autocor;

ddr4_v2_2_mc_autocorrect #(
    .PAYLOAD_WIDTH       (PAYLOAD_WIDTH)
   ,.ADDR_FIFO_WIDTH     (ADDR_FIFO_WIDTH)
   ,.DBAW                (DBAW)
   ,.MEM                 (MEM)
   ,.BGBITS              (BGBITS)
   ,.S_HEIGHT            (S_HEIGHT)
   ,.TXN_FIFO_MAX_WIDTH  (53)
   ,.ABITS               (ABITS)
   ,.COLBITS             (COLBITS)
   ,.nCK_PER_CLK         (nCK_PER_CLK)
   ,.TCQ                 (TCQ)
)u_ddr_mc_autocorrect(
    .clk                            (clk)
   ,.rst                            (rst)

   ,.calDone                        (calDone)
   ,.enable_autocorrect_sync        (enable_autocorrect_sync)

   ,.autocorrecting_rd_txn2autocor  (cas_rd_txn2autocor)
   ,.autocorrecting_rd_Buf2autocor  (cas_rd_Buf2autocor)
   ,.autocorrecting_rd_cmd2autocor  (cas_rd_cmd2autocor)
   ,.autocorrecting_ac_dis2autocor  (cas_ac_dis2autocor)
   ,.autocor2wr_txn                 (autocor2wr_txn)
   ,.autocor2wr_Buf                 (autocor2wr_Buf)
   ,.autocor2wr_ac_dis              (autocor2wr_ac_dis)
   ,.autocor2wr_req                 (autocor2wr_req)
   ,.wr_ack2autocor                 (wr_ack2autocor)

   ,.winPortC                       (winPortC)
   ,.tranSentC                      (tranSentC)

   ,.rd_data_mc2ni                  (rd_data_mc2ni)
   ,.rd_data_addr_mc2ni             (rd_data_addr_mc2ni)
   ,.rd_data_en_mc2ni               (rd_data_en_mc2ni)
   ,.rd_data_end_mc2ni              (rd_data_end_mc2ni)
   ,.ecc_err_addr                   (ecc_err_addr)
   ,.eccSingle                      (eccSingle)
   ,.eccDouble                      (eccDouble)
   ,.eccMultiple                    (eccMultiple)

   ,.fi_xor_wrdata_bufaddr          (fi_xor_wrdata_bufaddr)
   ,.fi_xor_wrdata_en               (fi_xor_wrdata_en)
   ,.autocor_wr_data                (autocor_wr_data)
   ,.autocor_wr_data_en             (autocor_wr_data_en)

   ,.cmd                    (cmd)
   ,.row                    (row)
   ,.rank                   (rank)
   ,.group                  (bagr)
   ,.bank                   (ba)
   ,.col                    (col)
   ,.useAdr                 (useAdr)
   ,.accept                 (accept)

   ,.rd_out_ac_dis_fifo_output      (rd_out_ac_dis_fifo_output)
   ,.autocorrecting_dis_fifo_output (autocorrecting_dis_fifo_output)
   ,.rd_outstanding_fifo_empty      (rd_outstanding_fifo_empty)
   ,.autocorrecting_fifo_empty      (autocorrecting_fifo_empty)
   ,.rd_outstanding_fifo_full       (rd_outstanding_fifo_full)
   ,.autocorrecting_fifo_full       (autocorrecting_fifo_full)
);

assign muxed_wr_data      = (autocor_wr_data_en)? autocor_wr_data : wr_data_ni2mc;
assign muxed_wr_data_mask = (autocor_wr_data_en)? {PAYLOAD_DM_WIDTH{1'b0}} : wr_data_mask_ni2mc;
"

               incr n 1
               set line_added 1
             }

             if {$Ecc_Autocorrect==1 && [string match -nocase {* ddr4_v2_2_6_mc_group #*} $new_line]} {
               set new_line "         .TXN_FIFO_MAX_WIDTH(53)
         ,.nCK_PER_CLK     (nCK_PER_CLK)
         ,"

               incr n 1
               set line_added 1
             }


             if {$Ecc_Autocorrect==1 && [string match -nocase {*)u_ddr_mc_ecc(*} $new_line]} {
               set new_line "   .enable_autocorrect_sync        (enable_autocorrect_sync)
   ,.enable_ERR_syndrome_recording (1'b0)
   ,.eccDouble           (eccDouble)
   ,"

               incr n 1
               set line_added 1
             }

             if {$Ecc_Autocorrect==1 && [string match -nocase {*endmodule*} $new_line]} {
               set new_line "
/******************************************************************************
// (c) Copyright 2013 - 2014 Xilinx, Inc. All rights reserved.
//
// This file contains confidential and proprietary information
// of Xilinx, Inc. and is protected under U.S. and
// international copyright and other intellectual property
// laws.
//
// DISCLAIMER
// This disclaimer is not a license and does not grant any
// rights to the materials distributed herewith. Except as
// otherwise provided in a valid license issued to you by
// Xilinx, and to the maximum extent permitted by applicable
// law: (1) THESE MATERIALS ARE MADE AVAILABLE \"AS IS\" AND
// WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
// AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
// BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
// INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
// (2) Xilinx shall not be liable (whether in contract or tort,
// including negligence, or under any other theory of
// liability) for any loss or damage of any kind or nature
// related to, arising under or in connection with these
// materials, including for any direct, or any indirect,
// special, incidental, or consequential loss or damage
// (including loss of data, profits, goodwill, or any type of
// loss or damage suffered as a result of any action brought
// by a third party) even if such damage or loss was
// reasonably foreseeable or Xilinx had been advised of the
// possibility of the same.
//
// CRITICAL APPLICATIONS
// Xilinx products are not designed or intended to be fail-
// safe, or for use in any application requiring fail-safe
// performance, such as life-support or safety devices or
// systems, Class III medical devices, nuclear facilities,
// applications related to the deployment of airbags, or any
// other applications that could lead to death, personal
// injury, or severe property or environmental damage
// (individually and collectively, \"Critical
// Applications\"). Customer assumes the sole risk and
// liability of any use of Xilinx products in Critical
// Applications, subject only to applicable laws and
// regulations governing limitations on product liability.
//
// THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
// PART OF THIS FILE AT ALL TIMES.
******************************************************************************/
//   ____  ____
//  /   /\/   /
// /___/  \  /    Vendor             : Xilinx
// \   \   \/     Version            : 1.1
//  \   \         Application        : MIG
//  /   /         Filename           : ddr3_v1_3_1_mc_autocorrect.sv
// \   \  /  \    Date Created       : Thu Apr 18 2013
//  \___\/\___\
//
// Device           : UltraScale
// Design Name      : DDR4 SDRAM & DDR3 SDRAM
// Purpose          :
//                   ddr3_v1_3_1_mc_autocorrect.sv module
// Reference        :
// Revision History :
//*****************************************************************************

`timescale 1ns/100ps

module ddr4_v2_2_mc_autocorrect # (parameter
    PAYLOAD_WIDTH              = 64
   ,ADDR_FIFO_WIDTH            = 30
   ,DBAW                       = 5
   ,MEM                        = \"DDR4\"
   ,BGBITS                     = 2
   ,S_HEIGHT                   = 1
   ,TXN_FIFO_MAX_WIDTH         = 53
   ,AUTOCORRECTING_FIFO_BYPASS = \"ON\"
   ,ABITS                      = 18
   ,COLBITS                    = 10
   ,nCK_PER_CLK                = 4
   ,TCQ                        = 0.1
)(
    input                               clk
   ,input                               rst

   ,input                               calDone
   ,input                               enable_autocorrect_sync

   ,input  \[4*TXN_FIFO_MAX_WIDTH-1:0\]   autocorrecting_rd_txn2autocor
   ,input  \[4*DBAW-1:0\]                 autocorrecting_rd_Buf2autocor
   ,input  \[3:0\]                        autocorrecting_rd_cmd2autocor
   ,input  \[3:0\]                        autocorrecting_ac_dis2autocor
   ,output \[TXN_FIFO_MAX_WIDTH-1:0\]     autocor2wr_txn
   ,output \[DBAW-1:0\]                   autocor2wr_Buf
   ,output                              autocor2wr_ac_dis
   ,output reg \[3:0\]                    autocor2wr_req
   ,input                               wr_ack2autocor

   ,input  \[3:0\]                        winPortC
   ,input                               tranSentC

   ,input  \[PAYLOAD_WIDTH*8-1:0\]        rd_data_mc2ni
   ,input  \[DBAW-1:0\]                   rd_data_addr_mc2ni
   ,input                               rd_data_en_mc2ni
   ,input                               rd_data_end_mc2ni
   ,input  \[ADDR_FIFO_WIDTH-1:0\]        ecc_err_addr
   ,input  \[2*nCK_PER_CLK-1:0\]          eccSingle
   ,input  \[2*nCK_PER_CLK-1:0\]          eccDouble
   ,input  \[2*nCK_PER_CLK-1:0\]          eccMultiple

   ,input  \[DBAW-1:0\]                   fi_xor_wrdata_bufaddr
   ,input                               fi_xor_wrdata_en
   ,output reg \[PAYLOAD_WIDTH*8-1:0\]    autocor_wr_data
   ,output reg                          autocor_wr_data_en

   ,input  \[2:0\]                        cmd
   ,input  \[ABITS-1:0\]                  row
   ,input  \[1:0\]                        rank
   ,input  \[1:0\]                        group
   ,input  \[1:0\]                        bank
   ,input  \[COLBITS-1:0\]                col
   ,input                               useAdr
   ,input                               accept

   ,output reg                          rd_out_ac_dis_fifo_output
   ,output reg                          autocorrecting_dis_fifo_output
   ,output reg                          rd_outstanding_fifo_empty
   ,output reg                          autocorrecting_fifo_empty
   ,output reg                          rd_outstanding_fifo_full
   ,output reg                          autocorrecting_fifo_full
);


// Pad fifo width to nibble align ap + cmd + LR_WIDTH + ABITS + COLBITS + rank + group + bank;
localparam TXN_FIFO_WIDTH = (S_HEIGHT > 1) ? 53 : 49;
localparam PER_RD_FIELD   = (S_HEIGHT > 1) ? 52 : 48;
localparam AP_FIELD       = (S_HEIGHT > 1) ? 51 : 47;
localparam CMD_MSB        = (S_HEIGHT > 1) ? 50 : 46;
localparam CMD_LSB        = (S_HEIGHT > 1) ? 48 : 44;
localparam LR_UNUSED_MSB  = 47;
localparam LR_MSB         = 46;
localparam LR_LSB         = 44;
localparam ROW_MSB        = 43;
localparam ROW_LSB        = 24;
localparam COL_MSB        = 23;
localparam COL_LSB        =  8;
localparam WSPACE_MSB     =  7;
localparam RANK_MSB       =  5;
localparam RANK_LSB       =  4;
localparam GROUP_MSB      =  3;
localparam GROUP_LSB      =  2;
localparam BANK_MSB       =  1;
localparam BANK_LSB       =  0;

localparam TXN_ADDR_WIDTH = ABITS + COLBITS + 3*2;  // 2 bits each of rank group bank

localparam RD_OUTSTANDING_PWIDTH = 4;
localparam RD_OUTSTANDING_DEPTH  = 32'b1 << RD_OUTSTANDING_PWIDTH;
localparam RD_OUTSTANDING_FULL_THRESHOLD = RD_OUTSTANDING_DEPTH - 1;
localparam AUTOCORRECTING_FIFO_PWIDTH = 3;
localparam AUTOCORRECTING_FIFO_DEPTH = 32'b1 << AUTOCORRECTING_FIFO_PWIDTH;
localparam AUTOCORRECTING_FIFO_FULL_THRESHOLD = AUTOCORRECTING_FIFO_DEPTH - 1;

localparam
    NATRD          = 3'b001
   ,NATWR          = 3'b000
   ,NATRMW         = 3'b011
;

reg                                   local_enable_autocorrect;
wire \[TXN_ADDR_WIDTH+3-1:0\]           txn_input;
reg \[TXN_ADDR_WIDTH+3-1:0\]            txn_input_r;
wire                                  wruseAdr_cycle;

// reads outstanding fifo signals
reg  \[TXN_FIFO_WIDTH-1:0\]             rd_outstanding_fifo    \[RD_OUTSTANDING_DEPTH-1:0\];
reg  \[TXN_FIFO_WIDTH-1:0\]             rd_outstanding_fifo_nxt\[RD_OUTSTANDING_DEPTH-1:0\];
reg  \[TXN_FIFO_WIDTH-1:0\]             rd_outstanding_fifo_output;
reg  \[RD_OUTSTANDING_PWIDTH-1:0\]      rd_outstanding_fifo_wptr;
reg  \[RD_OUTSTANDING_PWIDTH-1:0\]      rd_outstanding_fifo_rptr;
reg  \[DBAW-1:0\]                       buf_fifo    \[RD_OUTSTANDING_DEPTH-1:0\];
reg  \[DBAW-1:0\]                       buf_fifo_nxt\[RD_OUTSTANDING_DEPTH-1:0\];
reg  \[DBAW-1:0\]                       buf_fifo_output;
reg  \[RD_OUTSTANDING_DEPTH-1:0\]       rd_out_ac_dis_fifo;
reg  \[RD_OUTSTANDING_DEPTH-1:0\]       rd_out_ac_dis_fifo_nxt;

// autocorrecting fifo signals
reg  \[TXN_FIFO_WIDTH-1:0\]             autocorrecting_fifo    \[AUTOCORRECTING_FIFO_DEPTH-1:0\];
reg  \[TXN_FIFO_WIDTH-1:0\]             autocorrecting_fifo_nxt\[AUTOCORRECTING_FIFO_DEPTH-1:0\];
reg  \[TXN_FIFO_WIDTH-1:0\]             autocorrecting_fifo_output;
wire \[TXN_FIFO_WIDTH-1:0\]             autocorrecting_fifo_output_nxt;
reg  \[AUTOCORRECTING_FIFO_PWIDTH-1:0\] autocorrecting_fifo_wptr;
reg  \[AUTOCORRECTING_FIFO_PWIDTH-1:0\] autocorrecting_fifo_rptr;
reg  \[DBAW-1:0\]                       autocorrecting_dbuf_fifo    \[AUTOCORRECTING_FIFO_DEPTH-1:0\];
reg  \[DBAW-1:0\]                       autocorrecting_dbuf_fifo_nxt\[AUTOCORRECTING_FIFO_DEPTH-1:0\];
reg  \[DBAW-1:0\]                       autocorrecting_dbuf_fifo_output;
reg  \[AUTOCORRECTING_FIFO_DEPTH-1:0\]  autocorrecting_dis_fifo;
reg  \[AUTOCORRECTING_FIFO_DEPTH-1:0\]  autocorrecting_dis_fifo_nxt;
reg  \[PAYLOAD_WIDTH*8-1:0\]            autocorrecting_wrdata_buffer \[AUTOCORRECTING_FIFO_DEPTH-1:0\];
reg  \[PAYLOAD_WIDTH*8-1:0\]            autocorrecting_wrdata_buffer_nxt \[AUTOCORRECTING_FIFO_DEPTH-1:0\];
wire \[PAYLOAD_WIDTH*8-1:0\]            autocorrecting_wrdata_buffer_input;
reg  \[AUTOCORRECTING_FIFO_DEPTH-1:0\]  autocorrecting_wrdata_buffer_valid;
wire \[AUTOCORRECTING_FIFO_DEPTH-1:0\]  autocorrecting_wrdata_buffer_valid_nxt;
wire \[PAYLOAD_WIDTH*8-1:0\]            autocor_wr_data_nxt;
wire                                  autocor_wr_data_en_nxt;

wire \[1:0\]                            incoming_group_fsm;
wire \[1:0\]                            outgoing_group_fsm_nxt;

function \[TXN_ADDR_WIDTH-1:0\] convert_wide_address;
  input \[TXN_FIFO_MAX_WIDTH-1:0\] wide_address_in;
  begin
    convert_wide_address = {wide_address_in\[ABITS+ROW_LSB-1:ROW_LSB\],
                            wide_address_in\[COLBITS+COL_LSB-1:COL_LSB\],
                            wide_address_in\[RANK_MSB:BANK_LSB\]};
  end
endfunction

(* keep = \"TRUE\" *) reg rst_r1;

always @(posedge clk)
  rst_r1 <= rst;

always @(posedge clk)
  if (rst_r1) local_enable_autocorrect <= 1'b0;
  else local_enable_autocorrect <= calDone & enable_autocorrect_sync;

assign txn_input = {cmd,                   // 30:28
                    row,                   // 27:12
                    col,                   // 11: 6
                    rank,                  //  5: 4
                    group,                 //  3: 2
                    bank                   //  1: 0 
                   };

always @(posedge clk) txn_input_r <= txn_input;

assign wruseAdr_cycle =    useAdr
                        && accept
                        && (    (txn_input_r\[TXN_ADDR_WIDTH+3-1:TXN_ADDR_WIDTH\] == NATWR)
                             || (txn_input_r\[TXN_ADDR_WIDTH+3-1:TXN_ADDR_WIDTH\] == NATRMW));


// Map Bank and Group address bits to the Group FSMs (group_fsm).
wire \[TXN_FIFO_WIDTH-1:0\] rd_outstanding_fifo_input;

generate
   if (MEM == \"DDR3\") begin
      assign incoming_group_fsm = rd_outstanding_fifo_input\[GROUP_LSB:BANK_MSB\];
      assign outgoing_group_fsm_nxt = autocorrecting_fifo_empty? 2'b0 : autocorrecting_fifo_output_nxt\[GROUP_LSB:BANK_MSB\];
   end else begin
      if (BGBITS == 2) begin
         assign incoming_group_fsm = rd_outstanding_fifo_input\[GROUP_MSB:GROUP_LSB\];
         assign outgoing_group_fsm_nxt = autocorrecting_fifo_empty? 2'b0 : autocorrecting_fifo_output_nxt\[GROUP_MSB:GROUP_LSB\];
      end else begin
         assign incoming_group_fsm = {rd_outstanding_fifo_input\[GROUP_LSB\],rd_outstanding_fifo_input\[BANK_LSB\]};
         assign outgoing_group_fsm_nxt = autocorrecting_fifo_empty? 2'b0
                                     : {autocorrecting_fifo_output_nxt\[GROUP_LSB\],autocorrecting_fifo_output_nxt\[BANK_LSB\]};
      end
   end
endgenerate


// ===========================================
// reads outstanding fifo begin
// ===========================================

// which group added a read outstanding
wire \[3:0\]   which_bank_group_is_adding_txn =    autocorrecting_rd_cmd2autocor
                                               & winPortC
                                               & {4{(tranSentC && local_enable_autocorrect)}};
wire \[1:0\]   bank_group_txn_encoded         =  {    which_bank_group_is_adding_txn\[3\]
                                                 || which_bank_group_is_adding_txn\[2\],
                                                    which_bank_group_is_adding_txn\[3\]
                                                 || which_bank_group_is_adding_txn\[1\]};

// Select the txn fifo input
wire \[TXN_FIFO_WIDTH-1:0\] selected_rd_txn2autocor
      = autocorrecting_rd_txn2autocor\[bank_group_txn_encoded*TXN_FIFO_MAX_WIDTH+:TXN_FIFO_WIDTH\];

assign rd_outstanding_fifo_input
      =  {1'b0,  // PER_RD_FIELD
          1'b0,  // AP_FIELD
          NATWR, // CMD_MSB:CMD_LSB
          selected_rd_txn2autocor\[CMD_LSB-1:0\]};

// Increment the txn wptr if we see a cas_won
wire         inc_rd_outstanding_fifo_wptr  =   (|which_bank_group_is_adding_txn)
                                             & !selected_rd_txn2autocor\[PER_RD_FIELD\]; // get the right bit

// Store rd_outstanding_fifo_input to the reads outstanding_fifo
wire       rd_outstanding_fifo_push     = inc_rd_outstanding_fifo_wptr;

// Select the buf fifo input
wire \[DBAW-1:0\]           buf_fifo_input = autocorrecting_rd_Buf2autocor\[bank_group_txn_encoded*DBAW+:DBAW\];

// Select the autocorrecting disable fifo input
wire                      ac_dis_fifo_input = autocorrecting_ac_dis2autocor\[bank_group_txn_encoded\];

// Store buf_fifo_input to the buf_fifo
wire       buf_fifo_push     = inc_rd_outstanding_fifo_wptr;

wire \[RD_OUTSTANDING_PWIDTH-1:0\] rd_outstanding_fifo_wptr_nxt  = rd_outstanding_fifo_wptr + { { RD_OUTSTANDING_PWIDTH-1 { 1'b0 } }, inc_rd_outstanding_fifo_wptr }; // spyglass disable W164a

 
always @(*) begin
  rd_outstanding_fifo_nxt                           = rd_outstanding_fifo;
  rd_outstanding_fifo_nxt\[rd_outstanding_fifo_wptr\] = rd_outstanding_fifo_push ? rd_outstanding_fifo_input
						      : rd_outstanding_fifo\[rd_outstanding_fifo_wptr\];
  buf_fifo_nxt                                      = buf_fifo;
  buf_fifo_nxt\[rd_outstanding_fifo_wptr\]            = buf_fifo_push ? buf_fifo_input
                                                                    : buf_fifo\[rd_outstanding_fifo_wptr\];
end

// Issue accept_ns to the UI when the FIFO is not full and the group_fsm_select is a match
wire rd_outstanding_fifo_full_nxt  = (rd_outstanding_fifo_wptr - rd_outstanding_fifo_rptr) >= RD_OUTSTANDING_FULL_THRESHOLD\[RD_OUTSTANDING_PWIDTH-1:0\];

// Increment the txn rptr on .
wire         inc_rd_outstanding_fifo_rptr =    rd_data_en_mc2ni
                                            && !rd_outstanding_fifo_empty;
wire         rd_outstanding_fifo_pop      = inc_rd_outstanding_fifo_rptr;

wire \[RD_OUTSTANDING_PWIDTH-1:0\] rd_outstanding_fifo_rptr_nxt   = rd_outstanding_fifo_rptr + { { RD_OUTSTANDING_PWIDTH-1 { 1'b0 } }, inc_rd_outstanding_fifo_rptr }; // spyglass disable W164a

// Generate fifo empty signal and flop for timing.
wire                       rd_outstanding_fifo_empty_nxt  = rd_outstanding_fifo_wptr_nxt == rd_outstanding_fifo_rptr_nxt;

// Flop txn fifo output.
wire \[TXN_FIFO_WIDTH-1:0\]  rd_outstanding_fifo_output_nxt = rd_outstanding_fifo_nxt\[rd_outstanding_fifo_rptr_nxt\];

// Read FIFO
wire \[DBAW-1:0\]    buf_fifo_output_nxt = buf_fifo_nxt\[rd_outstanding_fifo_rptr_nxt\];


// ===========================================
// reads outstanding fifo end
// ===========================================


// ===========================================
// Autocorrecting pending FIFO. begin
// ===========================================


// Autocorrecting fifo input signals are the txn fifo output ports
wire \[TXN_FIFO_WIDTH-1:0\]  autocorrecting_fifo_input
      =  {1'b0,  // PER_RD_FIELD
          1'b0,  // AP_FIELD
          NATWR, // CMD_MSB:CMD_LSB
          rd_outstanding_fifo_output\[CMD_LSB-1:0\]};

wire \[DBAW-1:0\]            autocorrecting_dbuf_fifo_input = {1'b1,{DBAW-1{1'b0}}} | autocorrecting_fifo_wptr;

wire                       autocorrecting_dis_fifo_input
      =     wruseAdr_cycle
        && (txn_input_r\[TXN_ADDR_WIDTH-1:0\] == convert_wide_address(autocorrecting_fifo_input));


// Write into the fifo at the write pointer on every cycle
always @(*) begin
  autocorrecting_fifo_nxt = autocorrecting_fifo;
  autocorrecting_fifo_nxt\[autocorrecting_fifo_wptr\] = autocorrecting_fifo_input;
  autocorrecting_dbuf_fifo_nxt = autocorrecting_dbuf_fifo;
  autocorrecting_dbuf_fifo_nxt\[autocorrecting_fifo_wptr\] = autocorrecting_dbuf_fifo_input;
  autocorrecting_wrdata_buffer_nxt = autocorrecting_wrdata_buffer;
  autocorrecting_wrdata_buffer_nxt\[autocorrecting_fifo_wptr\] = autocorrecting_wrdata_buffer_input;
end

// Increment the write pointer when the txn fifo read pointer increments
wire                       autocorrecting_fifo_push =    rd_data_en_mc2ni
                                                      && |(eccSingle | eccDouble)
                                                      && ~|eccMultiple
                                                      && ~rd_out_ac_dis_fifo_output
                                                      && ~rd_outstanding_fifo_empty
                                                      && ~autocorrecting_fifo_full
                                                      && local_enable_autocorrect;
wire \[AUTOCORRECTING_FIFO_PWIDTH-1:0\] autocorrecting_fifo_wptr_nxt = autocorrecting_fifo_wptr                                              // spyglass disable W164a
                                               + { { AUTOCORRECTING_FIFO_PWIDTH-1 { 1'b0 }}, autocorrecting_fifo_push };

// Increment the read pointer when a Autocorrecting wins, unless it is an underfill read
wire                       autocorrecting_fifo_pop = |autocor2wr_req & wr_ack2autocor; // autocorrecting_won;
wire \[AUTOCORRECTING_FIFO_PWIDTH-1:0\] autocorrecting_fifo_rptr_nxt = autocorrecting_fifo_rptr                                              // spyglass disable W164a
                                               + { { AUTOCORRECTING_FIFO_PWIDTH-1 { 1'b0 }}, autocorrecting_fifo_pop };

// Full before all entries are used
wire   autocorrecting_fifo_empty_nxt = autocorrecting_fifo_wptr_nxt == autocorrecting_fifo_rptr_nxt;
wire   autocorrecting_fifo_full_nxt
         =   (    ( autocorrecting_fifo_wptr_nxt - autocorrecting_fifo_rptr_nxt )
                >= AUTOCORRECTING_FIFO_FULL_THRESHOLD\[AUTOCORRECTING_FIFO_PWIDTH-1:0\])
           | autocorrecting_wrdata_buffer_valid\[autocorrecting_fifo_wptr_nxt\];

// Read Autocorrecting FIFO
//wire \[TXN_FIFO_WIDTH-1:0\]  autocorrecting_fifo_output    = ( autocorrecting_fifo_empty & ( AUTOCORRECTING_FIFO_BYPASS == \"ON\" ) ) ? autocorrecting_fifo_input : autocorrecting_fifo\[autocorrecting_fifo_rptr\];
assign  autocorrecting_fifo_output_nxt    = autocorrecting_fifo\[autocorrecting_fifo_rptr\];

wire \[DBAW-1:0\]    autocorrecting_dbuf_fifo_output_nxt  = autocorrecting_dbuf_fifo\[autocorrecting_fifo_rptr\];


// ===========================================
// Autocorrecting pending FIFO. end
// ===========================================


// ===========================================
// Autocorrecting write data BUFFER. begin
// ===========================================

wire \[AUTOCORRECTING_FIFO_DEPTH-1:0\]  autocorrecting_wrdata_buffer_valid_set
       =   ({{AUTOCORRECTING_FIFO_DEPTH-1{1'b0}},1'b1} << autocorrecting_fifo_wptr)
         &  { AUTOCORRECTING_FIFO_DEPTH { autocorrecting_fifo_push } };
wire \[AUTOCORRECTING_FIFO_DEPTH-1:0\]  autocorrecting_wrdata_buffer_valid_clr
       =   (   ({{AUTOCORRECTING_FIFO_DEPTH-1{1'b0}},1'b1} << fi_xor_wrdata_bufaddr\[AUTOCORRECTING_FIFO_PWIDTH-1:0\] )
             &  { AUTOCORRECTING_FIFO_DEPTH { autocor_wr_data_en_nxt  } })
         | (   ({{AUTOCORRECTING_FIFO_DEPTH-1{1'b0}},1'b1} << autocor2wr_Buf\[AUTOCORRECTING_FIFO_PWIDTH-1:0\] )
             &  { AUTOCORRECTING_FIFO_DEPTH { (|autocor2wr_req && wr_ack2autocor && autocor2wr_ac_dis) } });

assign autocorrecting_wrdata_buffer_valid_nxt
       =   ~autocorrecting_wrdata_buffer_valid_clr
         & ( autocorrecting_wrdata_buffer_valid_set | autocorrecting_wrdata_buffer_valid );

assign autocor_wr_data_en_nxt
       =   fi_xor_wrdata_en
         & fi_xor_wrdata_bufaddr\[DBAW-1\]
         & autocorrecting_wrdata_buffer_valid\[fi_xor_wrdata_bufaddr\[AUTOCORRECTING_FIFO_PWIDTH-1:0\]\];

assign autocorrecting_wrdata_buffer_input = autocorrecting_fifo_push? rd_data_mc2ni
                                            : autocorrecting_wrdata_buffer\[autocorrecting_fifo_wptr\];

assign autocor_wr_data_nxt
       = autocor_wr_data_en_nxt? autocorrecting_wrdata_buffer\[fi_xor_wrdata_bufaddr\[AUTOCORRECTING_FIFO_PWIDTH-1:0\]\]
                               : {PAYLOAD_WIDTH*8{1'b0}};

// ===========================================
// Autocorrecting write data BUFFER. end
// ===========================================


// ===========================================
//  FIFO autocorrect disable. start
// ===========================================

integer                         i,j,k,l;

reg \[RD_OUTSTANDING_PWIDTH-1:0\]       rd_outstanding_fifo_index;
reg \[RD_OUTSTANDING_DEPTH-1:0\]        rd_outstanding_fifo_entry_valid;
reg                                   rd_outstanding_fifo_entry_valid_end;
reg \[TXN_ADDR_WIDTH-1:0\]              converted_rd_out_fifo_address \[RD_OUTSTANDING_DEPTH-1:0\];
reg \[AUTOCORRECTING_FIFO_PWIDTH-1:0\]  autocorrecting_fifo_index;
reg \[AUTOCORRECTING_FIFO_DEPTH-1:0\]   autocorrecting_fifo_entry_valid;
reg                                   autocorrecting_fifo_entry_valid_end;
reg \[TXN_ADDR_WIDTH-1:0\]              converted_autocorrecting_fifo_address \[AUTOCORRECTING_FIFO_DEPTH-1:0\];


always @(*) begin
  if (rst_r1 || rd_outstanding_fifo_empty) begin
    rd_out_ac_dis_fifo_nxt = 'b0;
  end
  else begin
    rd_out_ac_dis_fifo_nxt = rd_out_ac_dis_fifo;
  end
  begin
    rd_outstanding_fifo_entry_valid = 'b0;
    rd_outstanding_fifo_entry_valid_end = 1'b0;
    for (i=0;i<RD_OUTSTANDING_DEPTH;i=i+1) begin
      rd_outstanding_fifo_index = rd_outstanding_fifo_rptr + i;
      rd_outstanding_fifo_entry_valid\[rd_outstanding_fifo_index\] = !rd_outstanding_fifo_entry_valid_end;
      if (rd_outstanding_fifo_index == rd_outstanding_fifo_wptr) begin
        rd_outstanding_fifo_entry_valid_end = 1'b1;
      end
    end
    for (j=0;j<RD_OUTSTANDING_DEPTH;j=j+1) begin
      if (rd_outstanding_fifo_entry_valid\[j\]) begin
        converted_rd_out_fifo_address\[j\] = convert_wide_address(rd_outstanding_fifo\[j\]);
        if (    wruseAdr_cycle
             && (txn_input_r\[TXN_ADDR_WIDTH-1:0\] == converted_rd_out_fifo_address\[j\])) begin
          rd_out_ac_dis_fifo_nxt\[j\] = 1'b1;
        end
      end
      else begin
        rd_out_ac_dis_fifo_nxt\[j\] = 1'b0;
      end
    end
  end
  if (rd_outstanding_fifo_push) begin
    rd_out_ac_dis_fifo_nxt\[rd_outstanding_fifo_wptr\] = ac_dis_fifo_input;
  end
end

wire rd_out_ac_dis_fifo_output_nxt = rd_out_ac_dis_fifo_nxt\[rd_outstanding_fifo_rptr_nxt\];

always @(*) begin
  if (rst_r1 || autocorrecting_fifo_empty) begin
    autocorrecting_dis_fifo_nxt = 'b0;
  end
  else begin
    autocorrecting_dis_fifo_nxt = autocorrecting_dis_fifo;
  end
  begin
    autocorrecting_fifo_entry_valid = 'b0;
    autocorrecting_fifo_entry_valid_end = 1'b0;
    for (k=0;k<AUTOCORRECTING_FIFO_DEPTH;k=k+1) begin
      autocorrecting_fifo_index = autocorrecting_fifo_rptr + k;
      autocorrecting_fifo_entry_valid\[autocorrecting_fifo_index\] = !autocorrecting_fifo_entry_valid_end;
      if (autocorrecting_fifo_index == autocorrecting_fifo_wptr) begin
        autocorrecting_fifo_entry_valid_end = 1'b1;
      end
    end
    for (l=0;l<AUTOCORRECTING_FIFO_DEPTH;l=l+1) begin
      if (autocorrecting_fifo_entry_valid\[l\]) begin
        converted_autocorrecting_fifo_address\[l\] = convert_wide_address(autocorrecting_fifo\[l\]);
        if (    wruseAdr_cycle
             && (txn_input_r\[TXN_ADDR_WIDTH-1:0\] == converted_autocorrecting_fifo_address\[l\])) begin
          autocorrecting_dis_fifo_nxt\[l\] = 1'b1;
        end
      end
      else begin
        autocorrecting_dis_fifo_nxt\[l\] = 1'b0;
      end
    end
  end
  if (autocorrecting_fifo_push) begin
    autocorrecting_dis_fifo_nxt\[autocorrecting_fifo_wptr\] = autocorrecting_dis_fifo_input;
  end
end

wire autocorrecting_dis_fifo_output_nxt = autocorrecting_dis_fifo_nxt\[autocorrecting_fifo_rptr_nxt\];


// ===========================================
//  FIFO autocorrect disable. end
// ===========================================


always @(posedge clk) begin
  if (rst_r1) begin
    rd_outstanding_fifo_wptr           <= #TCQ '0;
    rd_outstanding_fifo_rptr           <= #TCQ '0;
    autocorrecting_fifo_wptr           <= #TCQ '0;
    autocorrecting_fifo_rptr           <= #TCQ '0;
    rd_outstanding_fifo_full           <= #TCQ '0;
    rd_outstanding_fifo_empty          <= #TCQ '1;
    rd_out_ac_dis_fifo_output          <= #TCQ '0;
    autocorrecting_fifo_empty          <= #TCQ '1;
    autocorrecting_fifo_full           <= #TCQ '0;
    autocorrecting_dis_fifo_output     <= #TCQ '0;
    autocor2wr_req                     <= #TCQ '0;
    autocorrecting_wrdata_buffer_valid <= #TCQ '0;
    autocor_wr_data                    <= #TCQ '0;
    autocor_wr_data_en                 <= #TCQ '0;
  end
  else begin
    rd_outstanding_fifo_wptr           <= #TCQ rd_outstanding_fifo_wptr_nxt;
    rd_outstanding_fifo_rptr           <= #TCQ rd_outstanding_fifo_rptr_nxt;
    autocorrecting_fifo_wptr           <= #TCQ autocorrecting_fifo_wptr_nxt;
    autocorrecting_fifo_rptr           <= #TCQ autocorrecting_fifo_rptr_nxt;
    rd_outstanding_fifo_full           <= #TCQ rd_outstanding_fifo_full_nxt;
    rd_outstanding_fifo_empty          <= #TCQ rd_outstanding_fifo_empty_nxt;
    rd_outstanding_fifo                <= #TCQ rd_outstanding_fifo_nxt;
    rd_outstanding_fifo_output         <= #TCQ rd_outstanding_fifo_output_nxt;
    buf_fifo                           <= #TCQ buf_fifo_nxt;
    buf_fifo_output                    <= #TCQ buf_fifo_output_nxt;
    rd_out_ac_dis_fifo                 <= #TCQ rd_out_ac_dis_fifo_nxt;
    rd_out_ac_dis_fifo_output          <= #TCQ rd_out_ac_dis_fifo_output_nxt;
    autocorrecting_fifo_empty          <= #TCQ autocorrecting_fifo_empty_nxt;
    autocorrecting_fifo_full           <= #TCQ autocorrecting_fifo_full_nxt;
    autocorrecting_fifo                <= #TCQ autocorrecting_fifo_nxt;
    autocorrecting_fifo_output         <= #TCQ autocorrecting_fifo_output_nxt;
    autocorrecting_dbuf_fifo           <= #TCQ autocorrecting_dbuf_fifo_nxt;
    autocorrecting_dbuf_fifo_output    <= #TCQ autocorrecting_dbuf_fifo_output_nxt;
    autocorrecting_dis_fifo            <= #TCQ autocorrecting_dis_fifo_nxt; 
    autocorrecting_dis_fifo_output     <= #TCQ autocorrecting_dis_fifo_output_nxt;
    autocor2wr_req                     <= #TCQ {3'b0,!wr_ack2autocor & !autocorrecting_fifo_empty}
                                                << outgoing_group_fsm_nxt ;
    autocorrecting_wrdata_buffer       <= #TCQ autocorrecting_wrdata_buffer_nxt;
    autocorrecting_wrdata_buffer_valid <= #TCQ autocorrecting_wrdata_buffer_valid_nxt;
    autocor_wr_data                    <= #TCQ autocor_wr_data_nxt;
    autocor_wr_data_en                 <= #TCQ autocor_wr_data_en_nxt;
  end
end

assign     autocor2wr_txn    = autocorrecting_fifo_output;
assign     autocor2wr_Buf    = autocorrecting_dbuf_fifo_output;
assign     autocor2wr_ac_dis = autocorrecting_dis_fifo_output;

//synopsys translate_off
`define MEM_AUTOCOR_INTERNAL

`ifdef MEM_AUTOCOR_INTERNAL
// Events - When asserted high in a test that passes all verification checks, these coverage
//          properties indicate that a functional coverage event has been hit.
// ---------------------------------------------------------------------------------------------

// Both FIFOs full at the same time
wire   e_mc_autocorrect_000_fifos_full = rd_outstanding_fifo_full & autocorrecting_fifo_full;
always @(posedge clk) mc_autocorrect_000: if (~rst_r1) cover property (e_mc_autocorrect_000_fifos_full);

// Asserts - When asserted high, an illegal condition has been detected and the test has failed.
// ---------------------------------------------------------------------------------------------

// Reads Outstanding fifo in use (current) depth
wire \[RD_OUTSTANDING_PWIDTH:0\] rd_outstanding_fifo_current_depth
      =  ({1'b1,rd_outstanding_fifo_wptr} - rd_outstanding_fifo_rptr) & {1'b0,{RD_OUTSTANDING_PWIDTH{1'b1}}};

// Autocorrecting fifo in use (current) depth
wire \[AUTOCORRECTING_FIFO_PWIDTH-1:0\]  autocorrecting_fifo_current_depth
      = ({1'b1,autocorrecting_fifo_wptr} - autocorrecting_fifo_rptr) & {1'b0,{AUTOCORRECTING_FIFO_PWIDTH{1'b1}}};

// Reads Outstanding fifo overflow. Maximum allowed fifo pointer separation is depth - 1
wire   a_mc_autocorrect_000_rdouts_of      = (rd_outstanding_fifo_wptr - rd_outstanding_fifo_rptr) == (RD_OUTSTANDING_DEPTH\[RD_OUTSTANDING_PWIDTH-1:0\] -1) & inc_rd_outstanding_fifo_wptr;
always @(posedge clk) if (~rst_r1) assert property (~a_mc_autocorrect_000_rdouts_of);

// Reads Outstanding fifo underflow.
wire   a_mc_autocorrect_001_rdouts_uf      = rd_outstanding_fifo_empty & inc_rd_outstanding_fifo_rptr & ~inc_rd_outstanding_fifo_wptr;
always @(posedge clk) if (~rst_r1) assert property (~a_mc_autocorrect_001_rdouts_uf);

// Autocorrecting fifo overflow.
wire   a_mc_autocorrect_002_autocorrecting_of      = autocorrecting_fifo_full & autocorrecting_fifo_push;
always @(posedge clk) if (~rst_r1) assert property (~a_mc_autocorrect_002_autocorrecting_of);

// Autocorrecting fifo underflow.
wire   a_mc_autocorrect_003_autocorrecting_uf      = autocorrecting_fifo_empty & autocorrecting_fifo_pop;
always @(posedge clk) if (~rst_r1) assert property (~a_mc_autocorrect_003_autocorrecting_uf);

// pushng a Reads Outstanding transaction from an incorrect bank_group
wire   a_mc_autocorrect_004_rdouts_pusherror
        =   (bank_group_txn_encoded != incoming_group_fsm)
          & inc_rd_outstanding_fifo_wptr;
always @(posedge clk) if (~rst_r1) assert property (~a_mc_autocorrect_004_rdouts_pusherror);

// poping a Reads Outstanding transaction from an incorrect bank_group 
wire   a_mc_autocorrect_005_rdouts_poperror
        =   (rd_data_addr_mc2ni != buf_fifo_output)
          & inc_rd_outstanding_fifo_rptr;
always @(posedge clk) if (~rst_r1) assert property (~a_mc_autocorrect_004_rdouts_pusherror);




`endif

//synopsys translate_on

endmodule
"
               incr n 1
               set line_added 1
             }


	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }
	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if {$Ecc_Autocorrect==1 && $n != 17 && $lola==0} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 17 line(s)."
	  }
	  if {$Ecc_Autocorrect==0 && $n != 12 && $lola==0} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 12 line(s)."
	  }
	  if {$n != 11 && $lola==1} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 11 line(s)."
	  }


	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


        #=========================================================================================================
        #rtl/controller/${ddr}_v2_2_mc_periodic.sv
        #=========================================================================================================
        set line_patched 0
        set line_added 0
        set target_file "${ddr}_v${file_ver}_mc_periodic.sv"
        puts "--> ${target_file}"
        set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
        set filename [get_property {NAME} $file]
        set tmp_filename "${filename}.tmp"
        set backup_filename "${filename}.bak"
        set in_file  [open $filename r]
        set out_file [open $tmp_filename w]

        # line-by-line, read the original file
        set n 0
        set flag 0
        while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0

            if {[string match {*output * gt_data_ready*} $new_line]} {
                set new_line "    output           per_state_idle,"
                incr n 1
                set line_added 1
            }

            if {[string match {*assign * gt_data_ready = periodic_state_update_status*} $new_line]} {
                set new_line "assign per_state_idle = periodic_state == IDLE;"
                incr n 1
                set line_added 1
            }

            if {[string match {*if ( interval_expired & ~read_accum ) begin*} $new_line]} {
                set new_line "
  if (!periodic_fsm_start) begin
    // Abort waiting if the enable goes away.
    // This cuts the majority of the waiting time.
    // In subsequent states don't try to abort because
    // we might leave ourselves in a bad state.
    periodic_state_nxt = IDLE;
  end else if ( interval_expired & ~read_accum ) begin"
                incr n 1
                set line_patched 1
            }

            if { $line_patched } {    
              puts_line_patched $line $out_file
            }
            if { $line_added } {    
              puts_line_added $line $out_file
            }
            puts $out_file $new_line
        }

        close $in_file
        close $out_file

        if { $n != 3 } {
          error "*** Failed to patch $filename as expected. Patched $n of expected 3 line(s)."
        }
        # Rename the original HDL file to have .bak extension.
        file rename -force $filename $backup_filename
        # Move the temporary file to have the filename of the original HDL file.
        file rename -force $tmp_filename $filename
        
        puts "Successfully patched \"${filename}\"."


        #=========================================================================================================
        #ddr3_v1_4_mc_act_rank.sv
        #=========================================================================================================
      if {$ddr == "ddr3"} {
      	    set line_patched 0
            set line_added 0

      set target_file "ddr3_v${file_ver}_mc_act_rank.sv"
         puts "--> ${target_file}"

        set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]


 # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            

# Status and control inputs and outputs	   
             if {[string match {*reg*rrdS*;*} $new_line]} {
              
              set new_line "reg \[2:0\] rrdS\[0:3\];"                
              incr n 1
	      set line_patched 1
             }

            
            if {[string match {*reg*rrdL*;*} $new_line]} {
              
              set new_line "reg \[2:0\] rrdL\[0:3\];"   
              incr n 1
	      set line_patched 1
             }


           if {[string match {*reg*rrdDLR*;*} $new_line]} {
              
              set new_line "reg \[2:0\] rrdDLR;" 
              incr n 1
	      set line_patched 1
             }




	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }

            puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 3 }  {
	    puts "*** ERROR: Failed to patch $filename as expected. Patched $n of expected 3 line(s)."
	  	error "ERROR!"
	  }

         
         # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."

        }

	  #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_intfc.sv
	  #====================================================================================================================================
	  set target_file "${mig_name}_${ddr}_mem_intfc.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]


 # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
	    set line_patched 0
            set line_added 0
            

# Status and control inputs and outputs	   
             if {[string match {*,output*} $new_line]  && [string match {*calDone*} $new_line]} {
               if {$axi_protocol==1} {
                 set new_line "
   // begin Status and control inputs and outputs
   ,input power_fail_has_scramed
   ,output  cntr_power_fail_complete
   ,output  inflight_writes
   ,input  inflight_writes_axi
   // end of patch"
                # puts $new_line 
               } else {                  
                 set new_line "
   // begin Status and control inputs and outputs
   ,input power_fail_has_scramed
   ,output  cntr_power_fail_complete
   ,output  inflight_writes
   // end of patch"
                # puts $new_line 
                      }
              incr n 1
	      set line_added 1
	    }

# wire defintion
# 
             if {[string match {*wire*} $new_line]  && [string match {*[7:0]*} $new_line] && [string match {*cal_RESET_n;*} $new_line]}  {
               set new_line "  wire inflight_writes_mc;\n  wire  inflight_writes_ui;\n  assign inflight_writes = inflight_writes_mc | inflight_writes_ui;"
             #  puts $new_line 
               incr n 1
	       set line_added 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*,.fi_xor_wrdata *} $new_line]} {
               set new_line "   ,.fi_xor_wrdata_bufaddr  (wrDataAddr)
   ,.enable_autocorrect_sync  (1'b1)"
               incr n 1
               set line_added 1
            }

#  ports mapping for status and control inputs and outputs 

             if {[string match {*,.wr_data_addr_phy2mc*} $new_line]  && [string match {*(wrDataAddr)*} $new_line]} {
               if {$axi_protocol==1} {
                 set new_line "
   // begin Status and control inputs and outputs
   ,.power_fail_has_scramed   (power_fail_has_scramed)
   ,.cntr_power_fail_complete (cntr_power_fail_complete)
   ,.inflight_writes          (inflight_writes_mc)
   ,.inflight_writes_ui       (inflight_writes_ui)
   ,.inflight_writes_axi      (inflight_writes_axi)
   // end of patch"
              #   puts $new_line 
               } else {                  
                 set new_line "
   // begin Status and control inputs and outputs
   ,.power_fail_has_scramed   (power_fail_has_scramed)
   ,.cntr_power_fail_complete (cntr_power_fail_complete)
   ,.inflight_writes          (inflight_writes_mc)
   ,.inflight_writes_ui       (inflight_writes_ui)
   // end of patch"
              #   puts $new_line 
                      }
              incr n 1
	      set line_added 1
	    }

# ports mapping for status and control inputs and outputs
# 
             if {[string match {*.autoprecharge*} $new_line] && [string match {*(autoprecharge),} $new_line]}  {
               set new_line "    .inflight_writes    (inflight_writes_ui),"
            #   puts $new_line 
               incr n 1
	       set line_added 1
	    }


             if {[string match {*,parameter*integer*DATA_BUF_ADDR_WIDTH*} $new_line]}  {
               set new_line "   ,parameter         integer DATA_BUF_ADDR_WIDTH = 6"
            #   puts $new_line 
               incr n 1
	       set line_patched 1
	    }

	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }

            puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $Ecc_Autocorrect==1 && $n != 6 } {
	    puts "*** ERROR: Failed to patch $filename as expected. Patched $n of expected 10 line(s)."
	  	error "ERROR!"
	  }
	  if { $Ecc_Autocorrect==0 && $n != 5 } {
	    puts "*** ERROR: Failed to patch $filename as expected. Patched $n of expected 10 line(s)."
	  	error "ERROR!"
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


	  #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_mc_group.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_mc_group.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]

	  #============================================
	  #	Old settings generated by the MIG to search
	  #============================================
  		# set trasf_width_old				{}	#already declared
	  	set trcd_cntr_width_old 			{3}
	  	set trcd_cntr_nxt_width_old 		{3}
	  	set trp_cntr_width_old 			{4}
	  	set trp_cntr_nxt_width_old 		{4}
	  	set tras_cntr_rb_width_old			{3}
	  	set tras_cntr_rb_nxt_width_old		{3}
	  	set tras_cntr_extend_width_old		{4}
	  	set e_trcd_cntr_0_0_width_old		{3}
	  	set e_trcd_cntr_0_1_width_old		{3}
	  	set e_trcd_cntr_0_2_width_old		{3}
	  	set e_trcd_cntr_0_3_width_old		{3}

	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  		# set trasf_width_new			{}	#already declared
	  	set trcd_cntr_width_new 		{6}
	  	set trcd_cntr_nxt_width_new 	{6}
	  	set trp_cntr_width_new 		{6}
	  	set trp_cntr_nxt_width_new 	{6}
	  	set tras_cntr_rb_width_new		{6}
	  	set tras_cntr_rb_nxt_width_new	{6}
	  	set tras_cntr_extend_width_new	{7}
	  	set e_trcd_cntr_0_0_width_new	{7}
	  	set e_trcd_cntr_0_1_width_new	{7}
	  	set e_trcd_cntr_0_2_width_new	{7}
	  	set e_trcd_cntr_0_3_width_new	{7}

	  # line-by-line, read the original file
	  set n 0
          set flag 0
          set flag1 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
	    set line_patched 0
            set line_added 0

            if {$Ecc_Autocorrect==1 && [string match {*,COLBITS = 10*} $new_line]} {
              set new_line "   ,CSBITS = 4
   ,TXN_FIFO_MAX_WIDTH = 53
   ,ACTIVE_PD_CKE_USE = \"OFF\"
   ,nCK_PER_CLK     = 4"
              incr n 1
	      set line_added 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*,output reg * cmdSize*} $new_line]} {
              set new_line "
   // to autocorrect
   ,output reg \[TXN_FIFO_MAX_WIDTH-1:0\] cas_rd_txn2autocor
   ,output reg               \[DBAW-1:0\] cas_rd_Buf2autocor
   ,output reg                          cas_rd_cmd2autocor
   ,output reg                          cas_ac_dis2autocor
   ,input      \[TXN_FIFO_MAX_WIDTH-1:0\] autocor2wr_txn
   ,input                    \[DBAW-1:0\] autocor2wr_Buf
   ,input                               autocor2wr_ac_dis
   ,input                               autocor2wr_req
   ,output reg                          wr_ack2autocor"
              incr n 1
	      set line_added 1
	    }
	    
	    # Performance improvement
            if {[string match {*localparam TXN_FIFO_DEPTH*} $new_line]  && [string match {*=*} $new_line] && [string match {*4;*} $new_line]}  {
              set new_line "localparam TXN_FIFO_DEPTH = 16;" 
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*localparam TXN_FIFO_PWIDTH*} $new_line]  && [string match {*=*} $new_line] && [string match {*2;*} $new_line]}  {
              set new_line "localparam TXN_FIFO_PWIDTH = 4;" 
              incr n 1
	      set line_patched 1
	    }
            
            if {[string match {*localparam CAS_FIFO_DEPTH*} $new_line]  && [string match {*=*} $new_line] && [string match {*4;*} $new_line]}  {
              set new_line "localparam CAS_FIFO_DEPTH = 16;" 
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*localparam CAS_FIFO_PWIDTH*} $new_line]  && [string match {*=*} $new_line] && [string match {*2;*} $new_line]}  {
              set new_line "localparam CAS_FIFO_PWIDTH = 4;" 
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wire*} $new_line] && [string match {*buf_fifo_push*} $new_line] && [string match {*=*} $new_line] && [string match {*inc_txn_fifo_wptr;*} $new_line]} {
              if {$Ecc_Autocorrect==0} {
                set new_line "
wire              txn_fifo_full_nxt;
wire              buf_fifo_push  =   inc_txn_fifo_wptr_accept_useadr | select_periodic_read;
wire \[DBAW-1:0\] buf_fifo_input = buf_fifo_push ? dBufAdr : buf_fifo\[txn_fifo_wptr\];"
              } else {
                # New buf_fifo_push requires some signals that haven't been defined yet.
                # Move down below
                set new_line ""
              }
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*wire * txn_fifo_input =*} $new_line]} {
              set new_line "wire \[TXN_FIFO_WIDTH-1:0\] txn_fifo_input =
           (autocor_req_to_this_group_at_right_time)? autocor2wr_txn\[TXN_FIFO_WIDTH-1:0\] :
               (PER_RD_PERF ? txn_out_or_per : txn_inp_or_per);

// Store dBufAdr the cycle after the address is stored
wire       buf_fifo_push  =   inc_txn_fifo_wptr_accept_useadr | select_periodic_read;

wire \[DBAW-1:0\] buf_fifo_input = 
                    autocor_req_to_this_group_at_right_time? autocor2wr_Buf :
                                  (buf_fifo_push ? dBufAdr : buf_fifo\[txn_fifo_wptr\]);"
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*assign accept_ns * =*} $new_line]} {
              set new_line "assign accept_ns        =    txn_fifo_accept_ns
                           //& ~ctl_block_req  // cke
                           & ~periodic_read_or_ref_block
                           & ~autocor_req_to_this_group_at_right_time;"
              incr n 1
	      set line_patched 1
	    }


            if {[string match {*wire*} $new_line] && [string match {*txn_fifo_full_nxt*} $new_line] && [string match {*txn_fifo_wptr*} $new_line] && [string match {*txn_fifo_rptr*} $new_line] && [string match {*>=*} $new_line] && [string match {*TXN_FIFO_FULL_THRESHOLD*} $new_line]} {
              set new_line "assign txn_fifo_full_nxt  = (txn_fifo_wptr - txn_fifo_rptr) >= TXN_FIFO_FULL_THRESHOLD\[TXN_FIFO_PWIDTH-1:0\];"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wire rd_req_nxt                = ( rdReqR | set_rd_req ) & ~cas_won;*} $new_line]} {
              set new_line "wire rd_req_nxt                = ( rdReqR & ~cas_won ) | set_rd_req;"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wire*wr_req_nxt*=*} $new_line]} {
              set new_line "wire wr_req_nxt                = ( wrReqR & ~cas_won ) | set_wr_req;"
              incr n 1
	      set line_patched 1
	    }

            if {$Ecc_Autocorrect==1 && [string match {*reg * txn_fifo_rptr;*} $new_line]} {
              set new_line "reg  \[TXN_FIFO_DEPTH-1:0\]  txn_fifo_entry_valid;
reg  \[TXN_FIFO_DEPTH-1:0\]  txn_fifo_autocor_disable;
reg  \[TXN_FIFO_DEPTH-1:0\]  txn_fifo_autocor_disable_nxt;
wire                       select_periodic_read_nxt;"
              incr n 1
	      set line_added 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*reg * cas_fifo_rptr;*} $new_line]} {
              set new_line "reg  \[CAS_FIFO_DEPTH-1:0\]  cas_fifo_entry_valid;
reg  \[CAS_FIFO_DEPTH-1:0\]  cas_fifo_autocor_disable;
reg  \[CAS_FIFO_DEPTH-1:0\]  cas_fifo_autocor_disable_nxt;"
              incr n 1
	      set line_added 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*Use a delayed*} $new_line]} {
              set new_line "wire           txn_fifo_full_nxt;

// See what the Autocorrector wants
wire        autocor_req_to_this_group_at_right_time
               =   autocor2wr_req
                  & !wr_ack2autocor
                  & !( accept & useAdr )
                  & !txn_fifo_full
                  & !txn_fifo_full_nxt
                  & !select_periodic_read
                  & !select_periodic_read_nxt
                  //& !ctl_block_req  // cke
                  & calDone;

$new_line"
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*wire * select_periodic_read_nxt =*} $new_line]} {
              set new_line "assign      select_periodic_read_nxt = ~periodic_read_push_safe & periodic_read_push_safe_nxt;"
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*wire * inc_txn_fifo_wptr *=*} $new_line]} {
              set new_line "wire        inc_txn_fifo_wptr               =   inc_txn_fifo_wptr_accept_useadr
                                              | select_periodic_read
                                              | (autocor2wr_req & wr_ack2autocor & !autocor2wr_ac_dis);"
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*wire * select_periodic_read_nxt =*} $new_line]} {
              set new_line "assign      select_periodic_read_nxt = ~periodic_read_push_safe & periodic_read_push_safe_nxt;"
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*wire * select_periodic_read_nxt =*} $new_line]} {
              set new_line "assign      select_periodic_read_nxt = ~periodic_read_push_safe & periodic_read_push_safe_nxt;"
              incr n 1
	      set line_patched 1
	    }


            if {[string match {*wire*} $new_line] && [string match {*cas_pend_fifo_output*} $new_line] && [string match {*=*} $new_line] && [string match {*cas_fifo_empty*} $new_line] && [string match {*CAS_FIFO_BYPASS*} $new_line] && [string match {*cas_pend_fifo_input*} $new_line] && [string match {*cas_fifo_rptr*} $new_line]} {
              set new_line "wire \[TXN_FIFO_WIDTH-1:0\]  cas_pend_fifo_output;\n 
assign cas_pend_fifo_output  = ( cas_fifo_empty && ( CAS_FIFO_BYPASS == \"ON\" ) )
         ? cas_pend_fifo_input
         : ( ( !cas_fifo_empty && cas_pend_fifo_pop && cas_fifo_valid\[cas_fifo_rptr_nxt\] )
           ? cas_pend_fifo\[cas_fifo_rptr_nxt\]
           : cas_pend_fifo\[cas_fifo_rptr\] );
              
assign cas_pend_output_valid = ( cas_fifo_empty && ( CAS_FIFO_BYPASS == \"ON\" ) )
         ? cas_pend_fifo_push
         : ( ( !cas_fifo_empty && cas_pend_fifo_pop && cas_fifo_valid\[cas_fifo_rptr_nxt\] )
           ? cas_fifo_valid\[cas_fifo_rptr_nxt\]
           : cas_fifo_valid\[cas_fifo_rptr\] && !cas_won );"
              
              incr n 1
	      set line_patched 1
	    }
            
            if {[string match {*wire*} $new_line] && [string match {*DBAW*} $new_line] && [string match {*cas_dbuf_output*} $new_line] && [string match {*cas_dbuf_fifo*} $new_line] && [string match {*cas_fifo_rptr*} $new_line]} {
              set new_line "wire \[DBAW-1:0\]            cas_dbuf_output;\n assign cas_dbuf_output =  ( !cas_fifo_empty && cas_pend_fifo_pop && cas_fifo_valid\[cas_fifo_rptr_nxt\])
                ? cas_dbuf_fifo\[cas_fifo_rptr_nxt\]
                : cas_dbuf_fifo\[cas_fifo_rptr\];"
              incr n 1
	      set line_patched 1
            }

           if {[string match {*RMW_WRWAIT: begin*} $new_line]} {
              set flag1 1
              incr n 1
	      set line_patched 1
	    }
           
           if {[string match {*gr_cas_state_nxt*} $new_line] && [string match {*=*} $new_line] && [string match {*CAS_IDLE;*} $new_line] && $flag1==0} {
              set new_line "          if (    ~cas_fifo_empty
                && cas_fifo_valid\[cas_fifo_rptr_nxt\]
                && trcd_cntr_is_zero\[cmd_rank_cas_nxt\]\[cmd_l_rank_cas_nxt_3ds\]\[cmd_bank_cas_nxt\]
                && (    ( cmd_cmd_nxt == NATRD )
                     || ( cmd_cmd_nxt == NATWR ) ) ) begin
             set_rd_req       = ( cmd_cmd_nxt == NATRD );
             set_wr_req       = ( cmd_cmd_nxt == NATWR );
             gr_cas_state_nxt = CAS_WAIT;
           end
           else begin
             gr_cas_state_nxt = CAS_IDLE;
             set_rd_req       = 1'b0;
             set_wr_req       = 1'b0;
           end"
              incr n 1
	      set line_patched 1
              set flag1 0
            }
           
           if {[string match {*Track page open and CAS with autoprecharge to Activate time on a per Rank per Bank basis*} $new_line]} {
              set flag 1
            }

           if {[string match {*always*} $new_line] && $flag ==1} {
              set new_line "  pages_open_nxt = 1'b0;"
              incr n 1
	      set line_added 1
              set flag 0
            }
           
            if {[string match {*cmd_lrank_row_cas*} $new_line] && [string match {*rank_index*} $new_line] && [string match {*pageInfo_exp*} $new_line] && [string match {*lr_index*} $new_line] && [string match {*bank_index*} $new_line]} {
              set new_line "      pages_open_nxt = pages_open_nxt || pageInfo\[rank_index\]\[lr_index\]\[bank_index\]\[0\];"
              incr n 1
	      set line_added 1
	    }
           
           if {[string match {*cas_fifo_valid*} $new_line] && [string match {*<=*} $new_line] && [string match {*#TCQ '0;*} $new_line]} {
              set new_line "    pages_open             <= #TCQ '0;"
              incr n 1
	      set line_added 1
	    }


           if {[string match {*cas_fifo_valid*} $new_line] && [string match {*<=*} $new_line] && [string match {*#TCQ cas_fifo_valid_nxt;*} $new_line]} {
              set new_line "  pages_open             <= #TCQ pages_open_nxt;"
              incr n 1
	      set line_added 1
	    }

            if {[string match {*,output reg*} $new_line] && [string match {*cas_fifo_full*} $new_line]} {
              set new_line "   ,output reg          pages_open"
              incr n 1
	      set line_added 1
	    }

            if {[string match {*reg*} $new_line] && [string match {*issue_cas_dly;*} $new_line]} {
              set new_line "reg       pages_open_nxt;"
              incr n 1
	      set line_added 1
	    }


            if {[string match {*wire*} $new_line] && [string match {*=*} $new_line] && [string match {*act_req_nxt*} $new_line] && [string match {*actReqR*} $new_line] && [string match {*set_act_req*} $new_line] && [string match {*~act_won;*} $new_line]} {
              set new_line " wire \[1:0\]         cmd_group_cas_nxt;
wire \[1:0\]         cmd_rank_cas_nxt;
wire \[1:0\]         cmd_bank_cas_nxt;
wire \[ABITS-1:0\]   cmd_row_cas_nxt;
wire               cas_pend_output_valid;
wire               cas_page_hit_for_req_nxt;

 assign  cas_page_hit_for_req_nxt
    =    cas_pend_output_valid
    && !cmdAP
    && ( cmd_row_cas   == cmd_row_cas_nxt )
    && ( cmd_rank_cas  == cmd_rank_cas_nxt )
    && ( cmd_group_cas == cmd_group_cas_nxt )
    && ( cmd_bank_cas  == cmd_bank_cas_nxt );"
              incr n 1
	      set line_added 1
	    }

           if {[string match {*rdReq*=*cas_won*?*1'b0*:* rdReqR;*} $new_line]} {
              set new_line "   rdReq  = cas_won ? ( rdReqR && rd_req_nxt & cas_page_hit_for_req_nxt ) : rdReqR;"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wrReq*=*cas_won*?*1'b0*:*wrReqR;*} $new_line]} {
              set new_line "   wrReq  = cas_won ? ( wrReqR && wr_req_nxt & cas_page_hit_for_req_nxt ) : wrReqR;"
              incr n 1
	      set line_patched 1
	    }
             
             if {[string match {*wire*[1:0]*cmd_group_cas_nxt*=*cas_pend_output_group;*} $new_line]} {
              set new_line "assign             cmd_group_cas_nxt     = cas_pend_output_group;"
              incr n 1
	      set line_patched 1
	    }

             if {[string match {*wire*[1:0]*cmd_rank_cas_nxt*=*cas_pend_output_rank;*} $new_line]} {
              set new_line "assign             cmd_rank_cas_nxt      = cas_pend_output_rank;"
              incr n 1
	      set line_patched 1
	    }


             if {[string match {*wire*[1:0]*cmd_bank_cas_nxt*=*cas_pend_output_bank;*} $new_line]} {
              set new_line "assign             cmd_bank_cas_nxt      = cas_pend_output_bank;"
              incr n 1
	      set line_patched 1
	    }

             if {[string match {*wire*[ABITS-1:0]*cmd_row_cas_nxt*=*cas_pend_output_row;*} $new_line]} {
              set new_line "assign             cmd_row_cas_nxt       = cas_pend_output_row;"
              if {$Ecc_Autocorrect==1} {
                set new_line "wire               cmd_rd_cmd_nxt        = ( cmd_cmd_nxt == NATRD );
wire               cmd_wr_cmd_nxt        = ( cmd_cmd_nxt == NATWR );
$new_line"
              }
              incr n 1
	      set line_patched 1
	    }
             if {[string match {*buf_fifo_nxt\[txn_fifo_wptr\]*=*} $new_line]} {
              set new_line "  buf_fifo_nxt\[txn_fifo_wptr\]     = buf_fifo_input;"
              incr n 1
	      set line_patched 1
	    }
            if {[string match {*CAS pending FIFO. end*} $new_line]} {
              set new_line "// ===========================================

// ===========================================
//  FIFO autocorrect disable. start
// ===========================================

integer                         i,j,k,l;
reg \[TXN_FIFO_PWIDTH-1:0\]       txn_fifo_index;
reg \[CAS_FIFO_PWIDTH-1:0\]       cas_fifo_index;


always @(*) begin
  if (txn_fifo_empty || rst_r1) begin
    txn_fifo_autocor_disable_nxt = 'b0;
  end
  else begin
    txn_fifo_autocor_disable_nxt = txn_fifo_autocor_disable;
    for (j=0;j<TXN_FIFO_DEPTH;j=j+1) begin
      if (txn_fifo_entry_valid\[j\]) begin
        if (     inc_txn_fifo_wptr_accept_useadr
             &&  (txn_fifo\[j\]\[CMD_MSB:CMD_LSB\] == NATRD)
             &&  (    (txn_fifo\[txn_fifo_wptr\]\[CMD_MSB:CMD_LSB\] == NATWR)
                   || (txn_fifo\[txn_fifo_wptr\]\[CMD_MSB:CMD_LSB\] == NATRMW))
             &&  (txn_fifo\[j\]\[ROW_MSB:0\] == txn_fifo\[txn_fifo_wptr\]\[ROW_MSB:0\])) begin
          txn_fifo_autocor_disable_nxt\[j\] = 1'b1;
        end
      end
      else begin
        txn_fifo_autocor_disable_nxt\[j\] = 1'b0;
      end
    end
  end 
end
  
always @(*) begin
  if (cas_fifo_empty || rst_r1) begin 
    cas_fifo_autocor_disable_nxt = 'b0;
  end
  else begin
    cas_fifo_autocor_disable_nxt = cas_fifo_autocor_disable;
    for (l=0;l<CAS_FIFO_DEPTH;l=l+1) begin
      if (cas_fifo_entry_valid\[l\]) begin
        if (     inc_txn_fifo_wptr_accept_useadr
             &&  (cas_pend_fifo\[l\]\[CMD_MSB:CMD_LSB\] == NATRD)
             &&  (    (txn_fifo\[txn_fifo_wptr\]\[CMD_MSB:CMD_LSB\] == NATWR)
                   || (txn_fifo\[txn_fifo_wptr\]\[CMD_MSB:CMD_LSB\] == NATRMW))
             &&  (cas_pend_fifo\[l\]\[ROW_MSB:0\] == txn_fifo\[txn_fifo_wptr\]\[ROW_MSB:0\])) begin
          cas_fifo_autocor_disable_nxt\[l\] = 1'b1;
        end
      end
      else begin
        cas_fifo_autocor_disable_nxt\[l\] = 1'b0;
      end
    end
  end 
  if (cas_pend_fifo_push) begin
    cas_fifo_autocor_disable_nxt\[cas_fifo_wptr\] = txn_fifo_autocor_disable_nxt\[txn_fifo_rptr\];
  end
end

wire cas_fifo_autocor_disable_output;

assign cas_fifo_autocor_disable_output = ( cas_fifo_empty && ( CAS_FIFO_BYPASS == \"ON\" ) ) 
           ? txn_fifo_autocor_disable_nxt\[txn_fifo_rptr\]
           : ( ( !cas_fifo_empty && cas_pend_fifo_pop && cas_fifo_valid\[cas_fifo_rptr_nxt\] )
               ? cas_fifo_autocor_disable_nxt\[cas_fifo_rptr_nxt\]
               : cas_fifo_autocor_disable_nxt\[cas_fifo_rptr\] );


// ===========================================
//  FIFO autocorrect disable. end"
              incr n 1
              set line_added 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*set_rd_req * cmd_cmd_nxt * NATRD*} $new_line]} {
              set new_line "          set_rd_req       = cmd_rd_cmd_nxt;"
              incr n 1
              set line_patched 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*set_wr_req * cmd_cmd_nxt * NATWR*} $new_line]} {
              set new_line "          set_wr_req       = cmd_wr_cmd_nxt;"
              incr n 1
              set line_patched 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*CAS_WAIT: begin*} $new_line]} {
              set new_line "/*
$new_line"
              incr n 1
              set line_patched 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*RMW_RDWAIT: begin*} $new_line]} {
              set new_line "*/
      CAS_WAIT: begin
        cas_state_wait = 1'b1;
        if (cas_won) begin
   // Parameter patched to support Everspin 1Gb DDR4 ST-MRAM.
   //          gr_cas_state_nxt = CAS_IDLE;
          if (    ~cas_fifo_empty
               && cas_fifo_valid\[cas_fifo_rptr_nxt\]
               && trcd_cntr_is_zero\[cmd_rank_cas_nxt\]\[cmd_l_rank_cas_nxt_3ds\]\[cmd_bank_cas_nxt\]
               && (    cmd_rd_cmd_nxt
                    || cmd_wr_cmd_nxt ) ) begin
            set_rd_req       = cmd_rd_cmd_nxt;
            set_wr_req       = cmd_wr_cmd_nxt;
            gr_cas_state_nxt = CAS_WAIT;
          end
          else begin
            gr_cas_state_nxt = CAS_IDLE;
            set_rd_req       = 1'b0;
            set_wr_req       = 1'b0;
          end
        end
      end
$new_line"
              incr n 1
              set line_patched 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*txn_fifo_rptr * <= * \'0;*} $new_line]} {
              set new_line "    txn_fifo_entry_valid <= #TCQ '0;
    cas_fifo_entry_valid <= #TCQ '0;
    cas_rd_txn2autocor   <= #TCQ '0;
    cas_rd_Buf2autocor   <= #TCQ '0;
    cas_rd_cmd2autocor   <= #TCQ 1'b0;
    wr_ack2autocor       <= #TCQ 1'b0;"
              incr n 1
              set line_added 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*txn_fifo_rptr * <= * txn_fifo_rptr_nxt;*} $new_line]} {
              set new_line "    if (inc_txn_fifo_wptr) txn_fifo_entry_valid\[txn_fifo_wptr\] <= #TCQ 1'b1;
    if (inc_txn_fifo_rptr) txn_fifo_entry_valid\[txn_fifo_rptr\] <= #TCQ 1'b0;
    //rank_activity        <= #TCQ rank_activity_nxt;
    if (cas_pend_fifo_push) cas_fifo_entry_valid\[cas_fifo_wptr\] <= #TCQ 1'b1; 
    if (cas_pend_fifo_pop)  cas_fifo_entry_valid\[cas_fifo_rptr\] <= #TCQ 1'b0;
    cas_rd_txn2autocor   <= #TCQ (S_HEIGHT > 1) ? cas_pend_fifo_output
                                                : { {(TXN_FIFO_MAX_WIDTH - TXN_FIFO_WIDTH){1'b1}}
                                                   ,cas_pend_fifo_output};
    cas_rd_Buf2autocor   <= #TCQ cmd_buf_nxt;
    cas_rd_cmd2autocor   <= #TCQ rd_req_nxt & !rd_rmw_nxt;
    cas_ac_dis2autocor   <= #TCQ cas_fifo_autocor_disable_output;
    wr_ack2autocor       <= #TCQ autocor_req_to_this_group_at_right_time;"
              incr n 1
              set line_added 1
            }
            if {$Ecc_Autocorrect==1 && [string match {*buf_fifo * <= * buf_fifo_nxt;*} $new_line]} {
              set new_line "    txn_fifo_autocor_disable    <= #TCQ txn_fifo_autocor_disable_nxt;
    cas_fifo_autocor_disable    <= #TCQ cas_fifo_autocor_disable_nxt;"
              incr n 1
              set line_added 1
            }

            # Patch trasf_width
	    if {[string match -nocase {*input*} $new_line] && [string match -nocase {*tRASF*} $new_line] && [string match -nocase "*${trasf_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trasf_width_old $new_line $trasf_width_new new_line]
	      set line_patched 1
	    }
	    
	    # Patch trcd_cntr
	    if {[string match -nocase {*reg*} $new_line] && [string match -nocase {*trcd_cntr *} $new_line] && [string match -nocase "*${trcd_cntr_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trcd_cntr_width_old $new_line $trcd_cntr_width_new new_line]
	      set line_patched 1
	    }

	    # Patch trcd_cntr_nxt
	    if {[string match -nocase {*reg*} $new_line] && [string match -nocase {*trcd_cntr_nxt *} $new_line] && [string match -nocase "*${trcd_cntr_nxt_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trcd_cntr_nxt_width_old $new_line $trcd_cntr_nxt_width_new new_line]
	      set line_patched 1
	    }
	    
	    # Patch trp_cntr
	    if {[string match -nocase {*reg*} $new_line] && [string match -nocase {*trp_cntr;*} $new_line] && [string match -nocase "*${trp_cntr_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trp_cntr_width_old $new_line $trp_cntr_width_new new_line]
	      set line_patched 1
	    }

	    # Patch trp_cntr_nxt
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*trp_cntr_nxt *} $new_line] && [string match -nocase "*${trp_cntr_nxt_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trp_cntr_nxt_width_old $new_line $trp_cntr_nxt_width_new new_line]
	      set line_patched 1
	    }
	    
	    # Patch tras_cntr_rb
	    if {[string match -nocase {*reg*} $new_line] && [string match -nocase {*tras_cntr_rb *} $new_line] && [string match -nocase "*${tras_cntr_rb_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $tras_cntr_rb_width_old $new_line $tras_cntr_rb_width_new new_line]
	      set line_patched 1
	    }    

	    # Patch tras_cntr_rb_nxt
	    if {[string match -nocase {*reg*} $new_line] && [string match -nocase {*tras_cntr_rb_nxt *} $new_line] && [string match -nocase "*${tras_cntr_rb_nxt_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $tras_cntr_rb_nxt_width_old $new_line $tras_cntr_rb_nxt_width_new new_line]
	      set line_patched 1
	    }
	    
	    # Patch tras_cntr_extend
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*tras_cntr_extend =*} $new_line] && [string match -nocase "*${tras_cntr_extend_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $tras_cntr_extend_width_old $new_line $tras_cntr_extend_width_new new_line]
	      set line_patched 1
	    }

	    # Patch e_trcd_cntr_0_0
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*e_trcd_cntr_0_0*} $new_line] && [string match -nocase "*${e_trcd_cntr_0_0_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $e_trcd_cntr_0_0_width_old $new_line $e_trcd_cntr_0_0_width_new new_line]
	      set line_patched 1
	    }
	    
	    # Patch e_trcd_cntr_0_1
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*e_trcd_cntr_0_1*} $new_line] && [string match -nocase "*${e_trcd_cntr_0_1_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $e_trcd_cntr_0_1_width_old $new_line $e_trcd_cntr_0_1_width_new new_line]
	      set line_patched 1
	    }
	    
	    # Patch e_trcd_cntr_0_2
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*e_trcd_cntr_0_2*} $new_line] && [string match -nocase "*${e_trcd_cntr_0_2_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $e_trcd_cntr_0_2_width_old $new_line $e_trcd_cntr_0_2_width_new new_line]
	      set line_patched 1
	    }

	    # Patch e_trcd_cntr_0_3
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*e_trcd_cntr_0_3*} $new_line] && [string match -nocase "*${e_trcd_cntr_0_3_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $e_trcd_cntr_0_3_width_old $new_line $e_trcd_cntr_0_3_width_new new_line]
	      set line_patched 1
	    }

	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }
	    
            if { $line_added } {    
              puts_line_added $line $out_file
	    }
            puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $Ecc_Autocorrect==1 && $n != 56 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 38 line(s)."
	  }
	  if { $Ecc_Autocorrect==0 && $n != 38 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 38 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


 	  #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_mc_arb_mux_p.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_mc_arb_mux_p.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]

	  #============================================
	  #	Old settings generated by the MIG to search
	  #============================================
  		# set trasf_width_old				{}	#already declared
  		set tras_temp_width_old			{3}

	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  		# set trasf_width_new				{}	#already declared
  		set tras_temp_width_new		{6}

	  # line-by-line, read the original file
	  set n 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
	    set line_patched 0
	    
	    # Patch trasf_width
	    if {[string match -nocase {*output*} $new_line] && [string match -nocase {*tRASF*} $new_line] && [string match -nocase "*${trasf_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $trasf_width_old $new_line $trasf_width_new new_line]
	      set line_patched 1
	    }

	    # Patch tras_temp
	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {* tRAS_TEMP *} $new_line] && [string match -nocase "*${tras_temp_width_old}*" $new_line]} {
	      incr n [regsub -nocase -- $tras_temp_width_old $new_line $tras_temp_width_new new_line]
	      set line_patched 1
	    }

	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }
	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 2 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 2 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."

#----------------------------------
 
 #----------------------------------
   #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_ui_cmd.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_ui_cmd.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================

	  # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
            

	    
            if {[string match {*Addressing with ROW - BANK - COLUMN for DDR4*}  $new_line]} {
	      set new_line "/*"
              incr n 1
	      set line_added 1
	    }
            if {[string match {*Addressing with ROW - COLUMN - BANK for DDR3*}  $new_line]} {
	      set new_line "*/
      else if (MEM_ADDR_ORDER == \"ROW_BANK_COLUMN\" && MEM == \"DDR4\") begin
        assign col = app_rdy_r ? app_addr_r1\[0+:COL_WIDTH\] : app_addr_r2\[0+:COL_WIDTH\];
        assign group = app_rdy_r ? app_addr_r1\[COL_WIDTH+:BANK_GROUP_WIDTH\]
                                 : app_addr_r2\[COL_WIDTH+:BANK_GROUP_WIDTH\];
        assign bank = app_rdy_r ? app_addr_r1\[COL_WIDTH+BANK_GROUP_WIDTH+:BANK_WIDTH\]
                                : app_addr_r2\[COL_WIDTH+BANK_GROUP_WIDTH+:BANK_WIDTH\];
        assign row = app_rdy_r ? app_addr_r1\[COL_WIDTH+BANK_GROUP_WIDTH+BANK_WIDTH+:ROW_WIDTH\]
                               : app_addr_r2\[COL_WIDTH+BANK_GROUP_WIDTH+BANK_WIDTH+:ROW_WIDTH\];
        if (S_HEIGHT == 1)
          assign lr = 'b0;
        else
          assign lr = app_rdy_r 
                          ? app_addr_r1\[COL_WIDTH+ROW_WIDTH+BANK_WIDTH+BANK_GROUP_WIDTH+:LR_WIDTH\]
                          : app_addr_r2\[COL_WIDTH+ROW_WIDTH+BANK_WIDTH+BANK_GROUP_WIDTH+:LR_WIDTH\];
        if (RANKS == 1)
          assign rank = 'b0;
        else if (S_HEIGHT == 1)
          assign rank = app_rdy_r
                          ? app_addr_r1\[COL_WIDTH+ROW_WIDTH+BANK_WIDTH+BANK_GROUP_WIDTH+:RANK_WIDTH\]
                          : app_addr_r2\[COL_WIDTH+ROW_WIDTH+BANK_WIDTH+BANK_GROUP_WIDTH+:RANK_WIDTH\];
        else
          assign rank = app_rdy_r
                          ? app_addr_r1\[COL_WIDTH+ROW_WIDTH+BANK_WIDTH+BANK_GROUP_WIDTH+LR_WIDTH+:RANK_WIDTH\]
                          : app_addr_r2\[COL_WIDTH+ROW_WIDTH+BANK_WIDTH+BANK_GROUP_WIDTH+LR_WIDTH+:RANK_WIDTH\];
      end  // end of patch

      else if (MEM_ADDR_ORDER ==  \"ROW_BANK_COLUMN_BANK\" && MEM == \"DDR3\") begin
        // Copy and modify Addressing with ROW - BANK - COLUMN for DDR3
        // original ROW_BANK_COLUMN code:
        // assign col = app_rdy_r ? app_addr_r1\[0+:COL_WIDTH\] : app_addr_r2\[0+:COL_WIDTH\];
        // assign row = app_rdy_r ? app_addr_r1\[COL_WIDTH+BANK_WIDTH+:ROW_WIDTH\]
        //                        : app_addr_r2\[COL_WIDTH+BANK_WIDTH+:ROW_WIDTH\];
        // assign bank = app_rdy_r ? app_addr_r1\[COL_WIDTH+:BANK_WIDTH\]
        //                         : app_addr_r2\[COL_WIDTH+:BANK_WIDTH\];
        //
        // COL_WIDTH  = 6
        // BANK_WIDTH = 3
        // ROW_WIDTH  = 16
        // RANKS      = 1
        //
        // 22222222221111111111
        // 987654321098765432109876543210
        //      rrrrrrrrrrrrrrrrbbcccbccc
        //                      01   2
        //
        // For best performance, 
        // bank\[0\] must be the slowest to change
        // bank\[2:1\] must be the fastest to change
        // The controller uses bank\[2:1\] to select the group it uses.
        //
        // synthesis translate_off
        // \$error(\"Need to improve this to use parameters for widths\");
        if (BANK_WIDTH != 3) begin
            \$error(\"Need to determine which bank bits to use for other cases\");
        end
        if (RANKS != 1) begin
            \$error(\"Need to determine which rank bits to use for other cases\");
        end
        // synthesis translate_on
        assign col = app_rdy_r ? {app_addr_r1\[6:4\], app_addr_r1\[2:0\]} : {app_addr_r2\[6:4\], app_addr_r2\[2:0\]};
        assign row = app_rdy_r ? app_addr_r1\[24:9\] : app_addr_r2\[24:9\];
        assign bank = app_rdy_r ? {app_addr_r1\[3\], app_addr_r1\[7\], app_addr_r1\[8\]} : {app_addr_r2\[3\], app_addr_r2\[7\], app_addr_r2\[8\]} ;
        assign rank = 'b0;
        assign group = 'b0;
        assign lr = 'b0;
      end\n"

              incr n 1
	      set line_added 1
	    }




	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 2 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


 #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_mc_ref.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_mc_ref.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================

	  # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
            

	    
            if {[string match -nocase {*,input*}  $new_line] && [string match {*rst*}  $new_line]} {
	      set new_line "   ,input scram_writing_done_pulse"
              incr n 1
	      set line_added 1

	    }

	    # Patch for_loop
	    if {[string match -nocase {*for*}  $new_line] && [string match {*=*}  $new_line] && [string match {*==*}  $new_line] && [string match -nocase {*RANKS*} $new_line] && [string match -nocase {*inc_pend_ref_due*} $new_line] && [string match -nocase {*inc_pend_ref_due*} $new_line] &&  [string match -nocase {*refi*} $new_line]} {
	      set new_line "for (i = 0; i < RANKS; i = i + 1)  inc_pend_ref_due\[i\] =    (~USER_MODE & ~REFI_DISABLE & ( refi\[i\] == '0 ))  || scram_writing_done_pulse;"
              incr n 1
	      set line_patched 1

	    }

	   if {$ddr == "ddr4"} {
              if {[string match {*wire *zq_intvl_load *=*} $new_line]} {
                 set new_line "wire        zq_intvl_load       = 0; // Never do ZQCL"
                 incr n 1
                 set line_patched 1

               }
            }


           if {[string match {*localparam ALL_RANKS =*} $new_line]} {
	      set new_line "localparam REFI_DISABLE = (tREFI == 17'h1ffff);"
              incr n 1
	      set line_added 1

	    }



	    if {[string match -nocase {*for (i = 0; i < RANKS; i = i + 1)  refi_nxt\[i\]*} $new_line]} {
	      set new_line "for (i = 0; i < RANKS; i = i + 1)  refi_nxt\[i\]         = { 15 { ~USER_MODE } } & ( ( REFI_DISABLE || (refi\[i\] == '0 )) ? tREFIF : ( refi\[i\] - { 14'b0, calDone } ) );"
              incr n 1
	      set line_patched 1

	    }


	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 5  && $ddr == "ddr4"} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 5 line(s)."
	  }
	  if { $n != 4  && $ddr == "ddr3"} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 4 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


  #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_ui_wr_data.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_ui_wr_data.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================

	  # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
            
	    
            if {[string match -nocase {*parameter TCQ = 100,*}  $new_line]} {
	      set new_line "   parameter FIFO_ADDR_WIDTH      = 5,"
              incr n 1
	      set line_added 1

	    }

            if {[string match -nocase {*app_wdf_rdy, wr_req_16, wr_data_buf_addr, wr_data, wr_data_mask*}  $new_line]} {
	      set new_line "  app_wdf_rdy, wr_req_full, wr_data_buf_addr, wr_data, wr_data_mask,"
              incr n 1
              set flag 1
	      set line_patched 1

	    }

            if {[string match {*raw_not_ecc,*}  $new_line] && $flag ==1} {
	      set new_line "  raw_not_ecc, wrdata_fifo_empty,"
              incr n 1
	      set line_patched 1
              set flag 0

	    }

            if {[string match {*localparam FULL_RAM_CNT*=*}  $new_line]} {
	      set new_line "  localparam WR_BUF_BITS_PER_RAM =   (FIFO_ADDR_WIDTH == 4)? 6
                                   : (FIFO_ADDR_WIDTH == 5)? 3
                                   :                         1;
  localparam FULL_RAM_CNT = (WR_BUF_WIDTH/WR_BUF_BITS_PER_RAM);"                     
              incr n 1
	      set line_patched 1
             }

            if {[string match {*localparam REMAINDER*=*}  $new_line]} {
	      set new_line "  localparam REMAINDER = WR_BUF_WIDTH % WR_BUF_BITS_PER_RAM;"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*localparam RAM_WIDTH*=*}  $new_line]} {
	      set new_line "  localparam RAM_WIDTH = (RAM_CNT*WR_BUF_BITS_PER_RAM);"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*input*wr_data_addr;*}  $new_line]} {
	      set new_line "  input \[FIFO_ADDR_WIDTH-1:0\]	wr_data_addr;"
              incr n 1
	      set line_patched 1
	    }
            if {[string match {*reg*\[3:0\]*wr_data_addr_r;*}  $new_line]} {
	      set new_line "  reg \[FIFO_ADDR_WIDTH-1:0\]	wr_data_addr_r;"
              incr n 1
	      set line_patched 1
	    }
          
          if {[string match {*rd_data_indx_r <= #TCQ rd_data_indx_r + 5'h1;*}  $new_line]} {
	      set new_line "        rd_data_indx_r <= #TCQ rd_data_indx_r + 1'b1;"
              incr n 1
	      set line_patched 1
	    }
          
          if {[string match {*reg*\[3:0\]*data_buf_addr_cnt_r;*}  $new_line]} {
	      set new_line "      reg \[FIFO_ADDR_WIDTH-1:0\]	data_buf_addr_cnt_r;"
              incr n 1
	      set line_patched 1
	    }

          if {[string match {*if (rst) data_buf_addr_cnt_ns = 4'b0;*}  $new_line]} {
	      set new_line "        if (rst) data_buf_addr_cnt_ns = 'b0;"
              incr n 1
	      set line_patched 1
	    }

                     
         if {[string match {*data_buf_addr_cnt_r + 4'h1;*}  $new_line]} {
	      set new_line "                                data_buf_addr_cnt_r + 1'b1;"
              incr n 1
	      set line_patched 1
	    }


          if {[string match {*input*\[3:0\]*ram_init_done_r;*}  $new_line]} {
	      set new_line "  input \[FIFO_ADDR_WIDTH-1:0\] ram_init_done_r;
  output wire wrdata_fifo_empty;
  localparam ENTRIES_WIDTH = 32'b1 << FIFO_ADDR_WIDTH;"
              incr n 1
	      set line_patched 1
	    }

          if {[string match {*reg*\[15:0\]*occ_cnt;*}  $new_line]} {
	      set new_line "      reg \[ENTRIES_WIDTH-1:0\] occ_cnt;
        reg                     wrdata_fifo_empty_r;"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*occ_cnt <= #TCQ 16'h0000;*}  $new_line]} {
	      set new_line "           occ_cnt <= #TCQ 'b0;"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*2'b01 : occ_cnt <= #TCQ \{1'b0,occ_cnt\[15:1\]\};*}  $new_line]} {
	      set new_line "              2'b01 : occ_cnt <= #TCQ \{1'b0,occ_cnt\[ENTRIES_WIDTH-1:1\]\};"
              incr n 1
	      set line_patched 1
	    }
  

           if {[string match {*2'b10 : occ_cnt <= #TCQ \{occ_cnt\[14:0\],1'b1\};*} $new_line]} {
	      set new_line "              2'b10 : occ_cnt <= #TCQ \{occ_cnt\[ENTRIES_WIDTH-2:0\],1'b1\};"
              incr n 1
	      set line_patched 1
	    }


           if {[string match {*(occ_cnt\[14\] && wr_data_end && ~rd_data_upd_indx_cpy_r)*} $new_line]} {
	      set new_line "                           (occ_cnt\[ENTRIES_WIDTH-2\] && wr_data_end && ~rd_data_upd_indx_cpy_r) ||"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*(occ_cnt\[15\] && ~rd_data_upd_indx_cpy_r));*} $new_line]} {
	      set new_line "                           (occ_cnt\[ENTRIES_WIDTH-1\] && ~rd_data_upd_indx_cpy_r));"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*(occ_cnt\[14\] && wr_data_end && ~rd_data_upd_indx_r) ||*} $new_line]} {
	      set new_line "                               (occ_cnt\[ENTRIES_WIDTH-2\] && wr_data_end && ~rd_data_upd_indx_r) ||"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*(occ_cnt\[15\] && ~rd_data_upd_indx_r));*} $new_line]} {
	      set new_line "                               (occ_cnt\[ENTRIES_WIDTH-1\] && ~rd_data_upd_indx_r));
/*"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*assign app_wdf_rdy = app_wdf_rdy_r;*} $new_line]} {
	      set new_line "
*/

      always @(posedge clk) 
        if (rst) begin
          app_wdf_rdy_r <= #TCQ 1'b0;
          wrdata_fifo_empty_r <= #TCQ 1'b1;
        end
        else begin
          app_wdf_rdy_r <= #TCQ wdf_rdy_ns_cpy;
          wrdata_fifo_empty_r <= #TCQ !(rd_data_upd_indx_r || |occ_cnt);
        end

      assign app_wdf_rdy = app_wdf_rdy_r;
      assign wrdata_fifo_empty = wrdata_fifo_empty_r;"
              incr n 1
	      set line_added 1
	    }


           if {[string match {*output wire wr_req_16;*} $new_line]} {
	      set new_line "  output wire wr_req_full;"
              incr n 1
	      set line_patched 1
	    }

          if {[string match {*reg \[3:0\] rd_data_indx_r;*} $new_line]} {
	      set new_line "  reg \[FIFO_ADDR_WIDTH-1:0\]	rd_data_indx_r;"
              incr n 1
	      set line_patched 1
	    }

          if {[string match {*reg \[3:0\] data_buf_addr_cnt_ns;*} $new_line]} {
	      set new_line "      reg \[FIFO_ADDR_WIDTH-1:0\]	data_buf_addr_cnt_ns;"
              incr n 1
	      set line_patched 1
	    }


           if {[string match {*reg \[4:0\] wr_req_cnt_ns;*} $new_line]} {
	      set new_line "      reg \[FIFO_ADDR_WIDTH:0\] wr_req_cnt_ns;"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*reg \[4:0\] wr_req_cnt_r;*} $new_line]} {
	      set new_line "      reg \[FIFO_ADDR_WIDTH:0\] wr_req_cnt_r;"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*if (rst) wr_req_cnt_ns = 5'b0;*} $new_line]} {
	      set new_line "        if (rst) wr_req_cnt_ns = 'b0;"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*2'b01 : wr_req_cnt_ns = wr_req_cnt_r - 5'b1;*} $new_line]} {
	      set new_line "               2'b01 : wr_req_cnt_ns = wr_req_cnt_r - 1'b1;"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*2'b10 : wr_req_cnt_ns = wr_req_cnt_r + 5'b1;*} $new_line]} {
	      set new_line "               2'b10 : wr_req_cnt_ns = wr_req_cnt_r + 1'b1;"
              incr n 1
	      set line_patched 1
	    }


           if {[string match {*assign wr_req_16 = (wr_req_cnt_ns == 5'h10);*} $new_line]} {
	      set new_line "      assign wr_req_full = (wr_req_cnt_ns == ENTRIES_WIDTH);"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*wr_req_mc_full: cover property (@(posedge clk) (~rst && wr_req_16));*} $new_line]} {
	      set new_line "  wr_req_mc_full: cover property (@(posedge clk) (~rst && wr_req_full));"
              incr n 1
	      set line_patched 1
	    }
          
          if {[string match {*(~rst && wr_accepted && rd_data_upd_indx_r && (wr_req_cnt_r == 5'hf)));*} $new_line]} {
	      set new_line "       (~rst && wr_accepted && rd_data_upd_indx_r && (wr_req_cnt_r == ENTRIES_WIDTH-1)));"
              incr n 1
	      set line_patched 1
	    }

         if {[string match {*(rst || !((wr_req_cnt_r == 5'b0) && (wr_req_cnt_ns == 5'h1f))));*} $new_line]} {
	      set new_line "         (rst || !((wr_req_cnt_r == 'b0) && (wr_req_cnt_ns == ENTRIES_WIDTH*2-1))));"
              incr n 1
	      set line_patched 1
	    }
         
         if {[string match {*(rst || !((wr_req_cnt_r == 5'h10) && (wr_req_cnt_ns == 5'h11))));*} $new_line]} {
	      set new_line "         (rst || !((wr_req_cnt_r == ENTRIES_WIDTH) && (wr_req_cnt_ns == ENTRIES_WIDTH+1))));"
              incr n 1
	      set line_patched 1
	    }


         if {[string match {*input \[3:0\] ram_init_addr;*} $new_line]} {
             set new_line "  input \[FIFO_ADDR_WIDTH-1:0\]		ram_init_addr;"
             incr n 1
             set line_patched 1
            }

         if {[string match {*output wire \[3:0\] wr_data_buf_addr;*} $new_line]} {
             set new_line "  output wire \[FIFO_ADDR_WIDTH-1:0\]	wr_data_buf_addr;"
             incr n 1
             set line_patched 1
            }

         if {[string match {*localparam PNTR_RAM_CNT = 2;*} $new_line]} {
             set new_line "
  localparam PNTR_RAM_CNT =   (FIFO_ADDR_WIDTH == 4)? 2
                            : (FIFO_ADDR_WIDTH == 5)? 3
                            :                         6;"
             incr n 1
             set line_patched 1
            }

        
         if {[string match {*wire \[3:0\] pointer_wr_data = ram_init_done_r\[2\]*} $new_line]} {
             set new_line "      wire \[PNTR_RAM_CNT*2-1:0\] pointer_wr_data = ram_init_done_r\[2\]"
             incr n 1
             set line_patched 1
            }
          
         
         if {[string match {*wire \[3:0\] pointer_wr_addr = ram_init_done_r\[2\]*} $new_line]} {
             set new_line "      wire \[FIFO_ADDR_WIDTH-1:0\] pointer_wr_addr = ram_init_done_r\[2\] "
             incr n 1
             set line_patched 1
            }
 
         if {[string match {*genvar i;*} $new_line]} {
             set new_line "      wire \[PNTR_RAM_CNT*2-1:0\] wr_data_buf_addr_tmp;
      wire \[PNTR_RAM_CNT*2-1:0\] wr_data_pntr_tmp;
             /*"
             incr n 1
             set line_patched 1
            }
 
         if {[string match {*used in single write, single read, 6 bit wide mode.*} $new_line]} {
             set new_line "
*/

      genvar i;

      for (i=0; i<PNTR_RAM_CNT; i=i+1) begin : rams
        if (FIFO_ADDR_WIDTH > 5) begin
          RAM64M
            #(.INIT_A(64'h0000000000000000),
              .INIT_B(64'h0000000000000000),
              .INIT_C(64'h0000000000000000),
              .INIT_D(64'h0000000000000000)
            ) RAM64M0 (
              .DOA(),
              .DOB(wr_data_buf_addr_tmp\[i\]),
              .DOC(wr_data_pntr_tmp\[i\]),
              .DOD(),
              .DIA(1'b0),
              .DIB(pointer_wr_data\[i\]),
              .DIC(pointer_wr_data\[i\]),
              .DID(1'b0),
              .ADDRA(6'b0),
              .ADDRB(data_buf_addr_cnt_r),
              .ADDRC(wr_data_indx_r),
              .ADDRD(pointer_wr_addr), 
              .WE(pointer_we),
              .WCLK(clk)
             );
        end
        else if (FIFO_ADDR_WIDTH > 4) begin
          RAM32M
            #(.INIT_A(64'h0000000000000000),
              .INIT_B(64'h0000000000000000),
              .INIT_C(64'h0000000000000000),
              .INIT_D(64'h0000000000000000)
            ) RAM32M0 (
              .DOA(),
              .DOB(wr_data_buf_addr_tmp\[i*2+:2\]),
              .DOC(wr_data_pntr_tmp\[i*2+:2\]),
              .DOD(),
              .DIA(2'b0),
              .DIB(pointer_wr_data\[i*2+:2\]),
              .DIC(pointer_wr_data\[i*2+:2\]),
              .DID(2'b0),
              .ADDRA(5'b0),
              .ADDRB(data_buf_addr_cnt_r),
              .ADDRC(wr_data_indx_r),
              .ADDRD(pointer_wr_addr),
              .WE(pointer_we),
              .WCLK(clk)
             );
        end
        else begin
          RAM32M
            #(.INIT_A(64'h0000000000000000),
              .INIT_B(64'h0000000000000000),
              .INIT_C(64'h0000000000000000),
              .INIT_D(64'h0000000000000000)
            ) RAM32M0 (
              .DOA(),
              .DOB(wr_data_buf_addr_tmp\[i*2+:2\]),
              .DOC(wr_data_pntr_tmp\[i*2+:2\]),
              .DOD(),
              .DIA(2'b0),
              .DIB(pointer_wr_data\[i*2+:2\]),
              .DIC(pointer_wr_data\[i*2+:2\]),
              .DID(2'b0),
              .ADDRA(5'b0),
              .ADDRB(\{1'b0, data_buf_addr_cnt_r\}),
              .ADDRC(\{1'b0, wr_data_indx_r\}),
              .ADDRD(\{1'b0, pointer_wr_addr\}),
              .WE(pointer_we),
              .WCLK(clk)
             );
        end
      end 
      assign wr_data_buf_addr = wr_data_buf_addr_tmp;
      assign wr_data_pntr     = wr_data_pntr_tmp;
  endgenerate "
             incr n 1
             set line_added 1
            }

         if {[string match {*wire \[4:0\] rd_addr_w;*} $new_line]} {
             set new_line "      wire \[FIFO_ADDR_WIDTH:0\] rd_addr_w;"
             incr n 1
             set line_patched 1
            }

         if {[string match {*genvar ii;*} $new_line]} {
             set new_line "/*"
             incr n 1
             set line_patched 1
            }

         if {[string match {*output \[APP_DATA_WIDTH-1:0\] wr_data;*} $new_line]} {
             set new_line "
*/

      genvar ii;
      for (ii=0; ii<RAM_CNT; ii=ii+1) begin : wr_buffer_ram
        if (FIFO_ADDR_WIDTH > 5) begin
          RAM128X1D
            #(.INIT(128'h00000000000000000000000000000000)
            ) RAM128X1D0 (
              .DPO(wr_buf_out_data_w\[ii\]),  // Read port 1-bit output
              .SPO(),                       // Read/write port 1-bit output
              .A(wb_wr_data_addr_w),        // Read/write port 7-bit address input
              .D(wr_buf_in_data\[ii\]),       // RAM data input
              .DPRA(rd_addr_w),             // Read port 7-bit address input
              .WE(wdf_rdy_ns),
              .WCLK(clk)
            );
        end
        else if (FIFO_ADDR_WIDTH > 4) begin
          RAM64M
            #(.INIT_A(64'h0000000000000000),
              .INIT_B(64'h0000000000000000),
              .INIT_C(64'h0000000000000000),
              .INIT_D(64'h0000000000000000)
            ) RAM64M0 (
              .DOA(wr_buf_out_data_w\[(ii*3)+2\]),
              .DOB(wr_buf_out_data_w\[(ii*3)+1\]),
              .DOC(wr_buf_out_data_w\[(ii*3)+0\]),
              .DOD(),
              .DIA(wr_buf_in_data\[(ii*3)+2\]),
              .DIB(wr_buf_in_data\[(ii*3)+1\]),
              .DIC(wr_buf_in_data\[(ii*3)+0\]),
              .DID(1'b0),
              .ADDRA(rd_addr_w),
              .ADDRB(rd_addr_w),
              .ADDRC(rd_addr_w),
              .ADDRD(wb_wr_data_addr_w),
              .WE(wdf_rdy_ns),
              .WCLK(clk)
             );
        end
        else begin
          RAM32M
            #(.INIT_A(64'h0000000000000000),
              .INIT_B(64'h0000000000000000),
              .INIT_C(64'h0000000000000000),
              .INIT_D(64'h0000000000000000)
            ) RAM32M0 (
              .DOA(wr_buf_out_data_w\[((ii*6)+4)+:2\]),
              .DOB(wr_buf_out_data_w\[((ii*6)+2)+:2\]),
              .DOC(wr_buf_out_data_w\[((ii*6)+0)+:2\]),
              .DOD(),
              .DIA(wr_buf_in_data\[((ii*6)+4)+:2\]),
              .DIB(wr_buf_in_data\[((ii*6)+2)+:2\]),
              .DIC(wr_buf_in_data\[((ii*6)+0)+:2\]),
              .DID(2'b0),
              .ADDRA(rd_addr_w),
              .ADDRB(rd_addr_w),
              .ADDRC(rd_addr_w),
              .ADDRD(wb_wr_data_addr_w),
              .WE(wdf_rdy_ns),
              .WCLK(clk)
             );
        end
      end // block: wr_buffer_ram
  endgenerate
  output \[APP_DATA_WIDTH-1:0\] wr_data;"
             incr n 1
             set line_added 1
            }


           if {[string match {*wire*wr_data_pntr;*} $new_line]} {
             set new_line "  wire \[FIFO_ADDR_WIDTH-1:0\]	wr_data_pntr; "
             incr n 1
             set line_patched 1
            }
            
            if {[string match {*wire*wb_wr_data_addr;*} $new_line]} {
             set new_line "  wire \[FIFO_ADDR_WIDTH:0\]	wb_wr_data_addr;"
             incr n 1
             set line_patched 1
            }

            if {[string match {*wire*wb_wr_data_addr_w;*} $new_line]} {
             set new_line "  wire \[FIFO_ADDR_WIDTH:0\]	wb_wr_data_addr_w;"
             incr n 1
             set line_patched 1
            }
           
           if {[string match {*reg*wr_data_indx_r;*} $new_line]} {
             set new_line "  reg \[FIFO_ADDR_WIDTH-1:0\]	wr_data_indx_r;"
             incr n 1
             set line_patched 1
            }
           
           if {[string match {*wr_data_indx_r <= #TCQ 4'h1;*} $new_line]} {
             set new_line "          wr_data_indx_r <= #TCQ 9'b1;"
             incr n 1
             set line_patched 1
            }
          
          if {[string match {*wr_data_indx_r <= #TCQ wr_data_indx_r + 4'h1;*} $new_line]} {
             set new_line "          wr_data_indx_r <= #TCQ wr_data_indx_r + 1'b1;"
             incr n 1
             set line_patched 1
            }

          if {[string match {*reg*wb_wr_data_addr_ns;*} $new_line]} {
             set new_line "      reg \[FIFO_ADDR_WIDTH:1\] wb_wr_data_addr_ns;"
             incr n 1
             set line_patched 1
            }
          if {[string match {*reg*wb_wr_data_addr_r;*} $new_line]} {
             set new_line "      reg \[FIFO_ADDR_WIDTH:1\] wb_wr_data_addr_r;"
             incr n 1
             set line_patched 1
            }
         
         if {[string match {*if (rst) wb_wr_data_addr_ns = 4'h0;*} $new_line]} {
             set new_line "        if (rst) wb_wr_data_addr_ns = 'b0;"
             incr n 1
             set line_patched 1
            }

  
	   
	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }
	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 54 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


   #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_ui_rd_data.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_ui_rd_data.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  	 			 

	  # line-by-line, read the original file
	  set n 0
          set line_num 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
            set line_removed 0
            set flag 0

	    
            if {[string match -nocase {*output*wire*ram_init_done_r,*}  $new_line] } {
	      set new_line "  output wire \[DATA_BUF_ADDR_WIDTH-1:0\] ram_init_done_r,"
              incr n 1
	      set line_patched 1
	    }

            if {[string match -nocase {*output*wire*ram_init_addr,*}  $new_line] } {
	      set new_line "  output wire \[DATA_BUF_ADDR_WIDTH-1:0\] ram_init_addr,"
              incr n 1
	      set line_patched 1
	    }


           if {[string match  {*reg*rd_buf_indx_r*;*}  $new_line] } {
	      set new_line "localparam DATA_BUF_DEPTH =            32'b1 << DATA_BUF_ADDR_WIDTH;
reg \[DATA_BUF_ADDR_WIDTH:0\]            rd_buf_indx_r \[0:DATA_BUF_DEPTH-1\];"
              incr n 1
	      set line_patched 1
	    }


            if {[string match -nocase {*assign*ram_init_done_r*=*ram_init_done_r_lcl*;*}  $new_line] } {
	      set new_line ""
              incr n 1
              # aolsen the line is being deleted, but using line_patched instead of line_deleted?
	      set line_patched 1
	    }

            if {[string match -nocase {*assign*app_ecc_multiple_err*=*app_ecc_multiple_err_r;*}  $new_line] } {
	      set new_line "  assign ram_init_done_r\[DATA_BUF_ADDR_WIDTH-1:0\] = ram_init_done_r_lcl\[DATA_BUF_ADDR_WIDTH-1:0\];"
              incr n 1
	      set line_added 1
	    }


            if {[string match -nocase {*assign*ram_init_addr*=*rd_buf_indx_r*;}  $new_line] } {
	      set new_line "  assign ram_init_addr = rd_buf_indx_r\[2\]\[DATA_BUF_ADDR_WIDTH-1:0\];"
              incr n 1
	      set line_patched 1
	    }


            if {[string match {*reg*rd_buf_indx_sts_r*;*}  $new_line] } {
              # aolsen patching commented code
	      set new_line "      reg \[DATA_BUF_ADDR_WIDTH-1:0\]  rd_buf_indx_sts_r \[0:3\];"
              incr n 1
	      set line_patched 1
	    }

            
            if {[string match {*reg*ram_init_done_r_lcl_sts;*}  $new_line] } {
              # aolsen patching commented code
	      set new_line "      (* keep = \"true\" *) reg \[DATA_BUF_ADDR_WIDTH-1:0\]  ram_init_done_r_lcl_sts;"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wire*rd_buf_wr_addr_sts*;*}  $new_line] } {
              # aolsen patching commented code
	      set new_line "      wire \[DATA_BUF_ADDR_WIDTH-1:0\] rd_buf_wr_addr_sts \[0:3\];"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wire*rd_buf_wr_addr = (DATA_BUF_ADDR_WIDTH == 5) ? rd_data_addr*}  $new_line] } {
              # aolsen patching commented code
              set new_line "      wire \[DATA_BUF_ADDR_WIDTH-1:0\] rd_buf_wr_addr
                                = (DATA_BUF_ADDR_WIDTH == 6) ? rd_data_addr\[5:0\] :
                                                               (DATA_BUF_ADDR_WIDTH == 5) ? rd_data_addr\[4:0\] :"
              incr n 1
	      set line_patched 1
	    }



            if {[string match {*always*@(*AS*rd_data_en*}  $new_line] } {
              # aolsen this code doesn't seem to do anything
	      set new_line "//"
              incr n 1
	      set line_removed 1
	    }



            if {[string match {*the status and pointer RAMs*}  $new_line] } {
	      set new_line "/*"
              incr n 1
	      set line_added 1
	    }


            if {[string match {*endmodule*}  $new_line] } {
	      set new_line "*/
  wire end_loop_flag;
  assign end_loop_flag = rd_buf_indx_r\[0\]\[DATA_BUF_ADDR_WIDTH-1:0\] == {DATA_BUF_ADDR_WIDTH{1'b1}};

  generate
    genvar j;
      wire upd_rd_buf_indx = (ram_init_done_r_lcl\[DATA_BUF_ADDR_WIDTH\] ? 
                              ((ORDERING == \"NORM\") && (bypass_cpy || rd_data_rdy_cpy)) : 1'b1);

      always @(posedge clk)
        if (rst)
          ram_init_done_r_lcl <= #TCQ 8'h00;
        else if (end_loop_flag)
          ram_init_done_r_lcl <= #TCQ 8'hFF;

    if (DATA_BUF_ADDR_WIDTH < 6) begin
      for (j=0; j<20; j=j+1) begin : rd_buf_index_cpy
        always @(posedge clk) begin
          if (rst) rd_buf_indx_r\[j\] <= #TCQ 'b0;
          else if (upd_rd_buf_indx) rd_buf_indx_r\[j\] <= #TCQ
            rd_buf_indx_r\[j\] + 6'h1 + (DATA_BUF_ADDR_WIDTH == 5 ? 0 : (single_data && ~rd_buf_indx_r\[j\]\[0\]));
        end
      end
    end
    else if (DATA_BUF_ADDR_WIDTH == 6) begin
      for (j=0; j<DATA_BUF_DEPTH; j=j+1) begin : rd_buf_index_cpy
        always @(posedge clk) begin
          if (rst) rd_buf_indx_r\[j\] <= #TCQ 'b0;
          else if (upd_rd_buf_indx) rd_buf_indx_r\[j\] <= #TCQ
            rd_buf_indx_r\[j\] + 6'h1 + (DATA_BUF_ADDR_WIDTH == 6 ? 0 : 
                                       (single_data && ~rd_buf_indx_r\[j\]\[0\]));
        end
      end
    end
  endgenerate


// Compute dimensions of read data buffer.  Depending on width of
// DQ bus and DRAM CK
// to fabric ratio, number of RAM32Ms is variable.  RAM32Ms are used in
// single write, single read, 6 bit wide mode.
  localparam RD_BUF_WIDTH = APP_DATA_WIDTH + (ECC == \"OFF\" ? 0 : 2*nCK_PER_CLK);
  localparam FULL_RAM_CNT = (RD_BUF_WIDTH/6);
  localparam REMAINDER = RD_BUF_WIDTH % 6;
  localparam RAM_CNT = FULL_RAM_CNT + ((REMAINDER == 0 ) ? 0 : 1);
  localparam RAM_WIDTH = (RAM_CNT*6);

// STRICT MODE
  generate
    if (ORDERING == \"STRICT\") begin : strict_mode
      assign single_data = 1'b0;
      assign rd_buf_full = 1'b0;
      reg \[DATA_BUF_ADDR_WIDTH-1:0\] rd_data_buf_addr_r_lcl;
      reg \[APP_DATA_WIDTH-1:0\] rd_data_r;
      wire \[DATA_BUF_ADDR_WIDTH-1:0\] rd_data_buf_addr_ns =
                   rst
                    ? 0
                    : rd_data_buf_addr_r_lcl + rd_accepted;
      always @(posedge clk) rd_data_buf_addr_r_lcl <=
                                #TCQ rd_data_buf_addr_ns;
      assign rd_data_buf_addr_r = rd_data_buf_addr_ns;
// app_* signals required to be registered.      
      if (ECC == \"OFF\") begin : ecc_off
        assign app_rd_data = rd_data;
        always @(/*AS*/rd_data_en) app_rd_data_valid = rd_data_en;
        always @(/*AS*/rd_data_end) app_rd_data_end = rd_data_end;
      end
      else begin : ecc_on  
        assign app_rd_data = rd_data_r;
        always @(posedge clk) rd_data_r <= #TCQ rd_data;
        always @(posedge clk) app_rd_data_valid <= #TCQ rd_data_en;
        always @(posedge clk) app_rd_data_end <= #TCQ rd_data_end;
        always @(posedge clk) app_ecc_multiple_err_r <= #TCQ ecc_multiple;
      end
    end

 // NON-STRICT MODE
// In configurations where read data is returned in a single fabric cycle
// the offset is always zero and we can use the bit to get a deeper
// FIFO. The RAMB32 has 5 address bits, so when the DATA_BUF_ADDR_WIDTH
// is set to use them all, discard the offset. Otherwise, include the
// offset.
    else begin : not_strict_mode
      genvar k;
      reg \[DATA_BUF_ADDR_WIDTH:0\]  rd_buf_indx_sts_r \[0:3\]; // use one bit wider
      (* keep = \"true\" *) reg        rd_buf_we_r1;
      (* keep = \"true\" *) reg \[3:0\]  ram_init_done_r_lcl_sts;
      reg \[3:0\]  upd_rd_buf_indx_sts;
      wire \[DATA_BUF_ADDR_WIDTH-1:0\] rd_status\[0:6\];
      wire \[3:0\] address_match_sts_0;
      wire \[3:0\] address_match_sts_1;
      wire \[3:0\] bypass_sts;
      wire \[3:0\] app_rd_data_end_sts;
      wire \[3:0\] single_data_sts;
      wire \[3:0\] rd_buf_we_sts;
      wire \[DATA_BUF_ADDR_WIDTH-1:0\] rd_buf_wr_addr_sts \[0:3\];
      wire \[3:0\] rd_data_rdy_sts;
      wire \[DATA_BUF_ADDR_WIDTH-1:0\] rd_buf_wr_addr
       = (DATA_BUF_ADDR_WIDTH == 6) ? rd_data_addr\[5:0\]
                                    : (DATA_BUF_ADDR_WIDTH == 5) ? rd_data_addr\[4:0\]
                                                                 : {rd_data_addr\[3:0\], rd_data_offset};
  
      for (k = 0; k < 4; k = k +1) begin : status_ram_signals
      
        assign address_match_sts_0\[k\] = match6_1({rd_buf_wr_addr_sts\[k\]\[2:0\],rd_buf_indx_sts_r\[k\]\[2:0\]});
        assign address_match_sts_1\[k\]
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr_sts\[k\]\[5:3\],rd_buf_indx_sts_r\[k\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr_sts\[k\]\[4:3\],rd_buf_indx_sts_r\[k\]\[4:3\]});
    
        assign bypass_sts\[k\] = rd_data_en && address_match_sts_0\[k\] && address_match_sts_1\[k\];
        assign app_rd_data_end_sts\[k\] = bypass_sts\[k\] ? rd_data_end : rd_status\[k\]\[1\];
        assign single_data_sts\[k\] = ram_init_done_r_lcl_sts\[k\] && app_rd_data_end_sts\[k\];
        assign rd_buf_we_sts\[k\] = ~ram_init_done_r_lcl_sts\[k\] || rd_data_en;

        assign rd_buf_wr_addr_sts\[k\] = (DATA_BUF_ADDR_WIDTH == 5 || DATA_BUF_ADDR_WIDTH == 6) ? rd_data_addr :
                                                                     {rd_data_addr, rd_data_offset};

        assign rd_data_rdy_sts\[k\] = (rd_status\[k\]\[0\] == rd_buf_indx_sts_r\[k\]\[DATA_BUF_ADDR_WIDTH\]);

        always @(*) begin
          casez ({ram_init_done_r_lcl_sts\[k\],address_match_sts_0\[k\],address_match_sts_1\[k\],
                  rd_data_en, rd_data_rdy_sts\[k\]})
            5'b0???? : upd_rd_buf_indx_sts\[k\] = 1'b1;
            5'b1???1 : upd_rd_buf_indx_sts\[k\] = 1'b1;
            5'b11110 : upd_rd_buf_indx_sts\[k\] = 1'b1;
            default : upd_rd_buf_indx_sts\[k\] = 1'b0;
          endcase 
        end

        always @(posedge clk)
          if (rst)
            ram_init_done_r_lcl_sts\[k\] <= #TCQ 1'b0;
          else if (rd_buf_indx_sts_r\[k\]\[DATA_BUF_ADDR_WIDTH-1:0\] == {DATA_BUF_ADDR_WIDTH{1'b1}})
            ram_init_done_r_lcl_sts\[k\] <= #TCQ 1'b1;

        always @(posedge clk) begin
          if (rst) rd_buf_indx_sts_r\[k\] <= #TCQ 'b0;
          else if (upd_rd_buf_indx_sts\[k\]) rd_buf_indx_sts_r\[k\] <= #TCQ
             rd_buf_indx_sts_r\[k\] + 6'h1 + (DATA_BUF_ADDR_WIDTH == 5 || DATA_BUF_ADDR_WIDTH == 6  ? 0 : 
                                           (single_data_sts\[k\] && ~rd_buf_indx_sts_r\[k\]\[0\]));
        end
      
      end

// Instantiate status RAM.  One bit for status and one for \"end\".
// Turns out read to write back status is a timing path.  Update
// the status in the ram on the state following the read.  Bypass
// the write data into the status read path.
// Not guaranteed to write second status bit. If it is written, always
// copy in the first status bit.
      begin : status_ram_0
        reg \[DATA_BUF_ADDR_WIDTH-1:0\] status_ram_wr_addr_r;
        reg \[1:0\] status_ram_wr_data_r;
        reg wr_status_r1;
        wire \[1:0\] wr_status;
        wire \[DATA_BUF_ADDR_WIDTH-1:0\] status_ram_wr_addr_ns = ram_init_done_r_lcl_sts\[0\]
                                           ? rd_buf_wr_addr_sts\[0\]
                                           : rd_buf_indx_sts_r\[0\]\[DATA_BUF_ADDR_WIDTH-1:0\];
        wire \[1:0\] status_ram_wr_data_ns = ram_init_done_r_lcl_sts\[0\] ?
                                           {rd_data_end, ~(rd_data_offset
                                                          ? wr_status_r1
                                                          : wr_status\[0\])}
                                           : 2'b0;
        always @(posedge clk) 
          status_ram_wr_addr_r <= #TCQ status_ram_wr_addr_ns;
        always @(posedge clk) wr_status_r1 <= #TCQ wr_status\[0\];
        always @(posedge clk) 
          status_ram_wr_data_r <= #TCQ status_ram_wr_data_ns;
        always @(posedge clk) rd_buf_we_r1 <= #TCQ rd_buf_we_sts\[0\];

        genvar kk;
        if (DATA_BUF_ADDR_WIDTH > 5) begin
          for (kk=0; kk<2; kk=kk+1) begin : w6rams
            RAM64M
              #(.INIT_A(64'h0000000000000000),
                .INIT_B(64'h0000000000000000),
                .INIT_C(64'h0000000000000000),
                .INIT_D(64'h0000000000000000)
              ) RAM64M0 (
                .DOA(rd_status\[0\]\[kk\]),
                .DOB(),
                .DOC(wr_status\[kk\]),
                .DOD(),
                .DIA(status_ram_wr_data_r\[kk\]),
                .DIB(1'b0),
                .DIC(status_ram_wr_data_r\[kk\]),
                .DID(status_ram_wr_data_r\[kk\]),
                .ADDRA(rd_buf_indx_sts_r\[0\]\[DATA_BUF_ADDR_WIDTH-1:0\]),
                .ADDRB(6'h0),
                .ADDRC(status_ram_wr_addr_ns),
                .ADDRD(status_ram_wr_addr_r),
                .WE(rd_buf_we_r1),
                .WCLK(clk)
               );
          end
        end 
        else begin
          RAM32M
            #(.INIT_A(64'h0000000000000000),
              .INIT_B(64'h0000000000000000),
              .INIT_C(64'h0000000000000000),
              .INIT_D(64'h0000000000000000)
             ) RAM32M0 (
              .DOA(rd_status\[0\]),
              .DOB(),
              .DOC(wr_status),
              .DOD(),
              .DIA(status_ram_wr_data_r),
              .DIB(2'b0),
              .DIC(status_ram_wr_data_r),
              .DID(status_ram_wr_data_r),
              .ADDRA(rd_buf_indx_sts_r\[0\]\[4:0\]),
              .ADDRB(5'h0),
              .ADDRC(status_ram_wr_addr_ns),
              .ADDRD(status_ram_wr_addr_r),
              .WE(rd_buf_we_r1),
              .WCLK(clk)
             );
        end

// Copies of the status RAM to meet timing
        genvar l;
      
        (* keep = \"true\" *) reg \[DATA_BUF_ADDR_WIDTH-1:0\] status_ram_wr_addr_cpy_r \[0:2\];
        (* keep = \"true\" *) reg \[1:0\] status_ram_wr_data_cpy_r \[0:2\];
//       reg \[DATA_BUF_ADDR_WIDTH-1:0\] status_ram_wr_data_cpy_r \[0:2\];
       
        (* keep = \"true\" *) reg \[2:0\] wr_status_r;
        wire \[1:0\] wr_status_cpy \[0:2\];
        (* keep = \"true\" *) wire \[DATA_BUF_ADDR_WIDTH-1:0\] status_ram_wr_addr_cpy \[0:2\];
        (* keep = \"true\" *) wire \[1:0\] status_ram_wr_data_cpy \[0:2\];
        (* keep = \"true\" *) reg \[2:0\] rd_buf_we_r;
        
        for (l = 0; l < 3; l = l+1) begin : copies_of_sts_ram

          assign status_ram_wr_addr_cpy\[l\] = ram_init_done_r_lcl_sts\[l+1\] ?
                                             rd_buf_wr_addr_sts\[l+1\] :
                                             rd_buf_indx_sts_r\[l+1\]\[DATA_BUF_ADDR_WIDTH-1:0\];

          assign status_ram_wr_data_cpy\[l\] = ram_init_done_r_lcl_sts\[l+1\] ?
                                             {rd_data_end, ~(rd_data_offset ?
                                                            wr_status_r\[l\] :
                                                            wr_status_cpy\[l\]\[0\])} :
                                             2'b0; 

          always @(posedge clk) wr_status_r\[l\] <= #TCQ wr_status_cpy\[l\]\[0\];
          always @(posedge clk) 
            status_ram_wr_addr_cpy_r\[l\] <= #TCQ status_ram_wr_addr_cpy\[l\];
          always @(posedge clk) 
            status_ram_wr_data_cpy_r\[l\] <= #TCQ status_ram_wr_data_cpy\[l\];
          always @(posedge clk) rd_buf_we_r\[l\] <= #TCQ rd_buf_we_sts\[l+1\];

     

          genvar jj;
          if (DATA_BUF_ADDR_WIDTH > 5) begin
            for (jj=0; jj<2; jj=jj+1) begin : w6rams1
              RAM64M
                #(.INIT_A(64'h0000000000000000),
                  .INIT_B(64'h0000000000000000),
                  .INIT_C(64'h0000000000000000),
                  .INIT_D(64'h0000000000000000)
                 ) RAM64M0 (
                  .DOA(rd_status\[l+1\]\[jj\]),
                  .DOB(rd_status\[l+4\]\[jj\]),
                  .DOC(wr_status_cpy\[l\]\[jj\]),
                  .DOD(),
                  .DIA(status_ram_wr_data_cpy_r\[l\]\[jj\]),
                  .DIB(status_ram_wr_data_cpy_r\[l\]\[jj\]),
                  .DIC(status_ram_wr_data_cpy_r\[l\]\[jj\]),
                  .DID(1'b0),
                  .ADDRA(rd_buf_indx_sts_r\[l+1\]\[DATA_BUF_ADDR_WIDTH-1:0\]),
                  .ADDRB(rd_buf_indx_sts_r\[l+1\]\[DATA_BUF_ADDR_WIDTH-1:0\]),
                  .ADDRC(status_ram_wr_addr_cpy\[l\]),
                  .ADDRD(status_ram_wr_addr_cpy_r\[l\]),
                  .WE(rd_buf_we_r\[l\]),
                  .WCLK(clk)
                 );
            end
          end  
          else begin
            RAM32M
              #(.INIT_A(64'h0000000000000000),
                .INIT_B(64'h0000000000000000),
                .INIT_C(64'h0000000000000000),
                .INIT_D(64'h0000000000000000)
               ) RAM32M1 (
                .DOA(rd_status\[l+1\]),
                .DOB(rd_status\[l+4\]),
                .DOC(wr_status_cpy\[l\]),
                .DOD(),
                .DIA(status_ram_wr_data_cpy_r\[l\]),
                .DIB(status_ram_wr_data_cpy_r\[l\]),
                .DIC(status_ram_wr_data_cpy_r\[l\]),
                .DID(2'b0),
                .ADDRA(rd_buf_indx_sts_r\[l+1\]\[4:0\]),
                .ADDRB(rd_buf_indx_sts_r\[l+1\]\[4:0\]),
                .ADDRC(status_ram_wr_addr_cpy\[l\]),
                .ADDRD(status_ram_wr_addr_cpy_r\[l\]),
                .WE(rd_buf_we_r\[l\]),
                .WCLK(clk)
               );
          end
        
        end
      end // block: status_ram

      wire \[RAM_WIDTH-1:0\] rd_buf_out_data;
      begin : rd_buf
        wire \[RAM_WIDTH-1:0\] rd_buf_in_data;
        if (REMAINDER == 0)
          if (ECC == \"OFF\")
            assign rd_buf_in_data = rd_data;
          else
            assign rd_buf_in_data = {ecc_multiple, rd_data};
        else
          if (ECC == \"OFF\")
            assign rd_buf_in_data = {{6-REMAINDER{1'b0}}, rd_data};
          else
            assign rd_buf_in_data = {{6-REMAINDER{1'b0}}, ecc_multiple, rd_data};

        reg \[DATA_BUF_ADDR_WIDTH:0\] rd_buf_indx_cpy_r \[0:RAM_CNT-1\];
        reg \[RAM_CNT-1:0\] upd_rd_buf_indx_cpy;
        (* keep = \"true\" *) reg \[RAM_CNT-1:0\] init_done_r;
        wire \[RAM_CNT-1:0\] address_match_buf0;
        wire \[RAM_CNT-1:0\] address_match_buf1;
        wire \[RAM_CNT-1:0\] address_match_dout0;
        wire \[RAM_CNT-1:0\] address_match_dout1;
        wire \[RAM_CNT-1:0\] bypass_buf;
        wire \[RAM_CNT-1:0\] app_rd_data_end_buf;
        (* keep = \"true\" *) reg \[RAM_CNT-1:0\] single_data_buf;
        wire \[RAM_CNT-1:0\] rd_data_rdy_buf;
        (* keep = \"true\" *) wire \[RAM_CNT-1:0\] rd_buf_we;
        reg \[RAM_WIDTH-1:0\] app_rd_data_ns;  // spyglass disable W498



        genvar i;
        for (i=0; i<RAM_CNT; i=i+1) begin : rd_buffer_ram

          // Dedicated copy for driving distributed RAM.
          assign address_match_buf0\[i\] = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_cpy_r\[i\]\[2:0\]});
          assign address_match_buf1\[i\]
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr\[5:3\],rd_buf_indx_cpy_r\[i\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr\[4:3\],rd_buf_indx_cpy_r\[i\]\[4:3\]});
          assign address_match_dout0\[i\] = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_cpy_r\[i\]\[2:0\]});
          assign address_match_dout1\[i\]
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr\[5:3\],rd_buf_indx_cpy_r\[i\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr\[4:3\],rd_buf_indx_cpy_r\[i\]\[4:3\]});
          assign bypass_buf\[i\] = rd_data_en && address_match_buf0\[i\] && address_match_buf1\[i\];
          assign app_rd_data_end_buf\[i\] = bypass_buf\[i\] ? rd_data_end : rd_status\[i%6+1\]\[1\];  // spyglass disable UndrivenNet-ML
          assign rd_data_rdy_buf\[i\] = (rd_status\[i%6+1\]\[0\] == rd_buf_indx_cpy_r\[i\]\[DATA_BUF_ADDR_WIDTH\]);
          assign rd_buf_we\[i\] = ~init_done_r\[i\] || rd_data_en;
            always @(posedge clk)
              if (rst)
                single_data_buf\[i\] <= #TCQ 1'b0;
              else if (init_done_r\[i\]) 
             single_data_buf\[i\] <= #TCQ app_rd_data_end_buf\[i\] && ~(DATA_BUF_ADDR_WIDTH == 6 || DATA_BUF_ADDR_WIDTH == 5)  && 
                                           ~rd_buf_indx_cpy_r\[i\]\[0\];



          always @(posedge clk)
            if (rst)
              init_done_r\[i\] <= #TCQ 1'b0;
            else if (rd_buf_indx_cpy_r\[i\]\[DATA_BUF_ADDR_WIDTH-1:0\] == {DATA_BUF_ADDR_WIDTH{1'b1}})
              init_done_r\[i\] <= #TCQ 1'b1;
   
          always @(*) begin
            casez ({init_done_r\[i\],address_match_buf0\[i\],address_match_buf1\[i\],
                    rd_data_en, rd_data_rdy_buf\[i\]})
              5'b0???? : upd_rd_buf_indx_cpy\[i\] = 1'b1;
              5'b1???1 : upd_rd_buf_indx_cpy\[i\] = 1'b1;
              5'b11110 : upd_rd_buf_indx_cpy\[i\] = 1'b1;
              default : upd_rd_buf_indx_cpy\[i\] = 1'b0;
            endcase 
          end

          always @(posedge clk) begin
            if (rst) 
              rd_buf_indx_cpy_r\[i\] <= #TCQ 'b0;
            else if (upd_rd_buf_indx_cpy\[i\]) 
              rd_buf_indx_cpy_r\[i\] <= #TCQ rd_buf_indx_cpy_r\[i\] + 6'h1;
          end


          genvar ii;
          if (DATA_BUF_ADDR_WIDTH > 5) begin
            for (ii=0; ii<2; ii=ii+1) begin : w6rams
              RAM64M
                #(.INIT_A(64'h0000000000000000),
                  .INIT_B(64'h0000000000000000),
                  .INIT_C(64'h0000000000000000),
                  .INIT_D(64'h0000000000000000)
                ) RAM64M0 (
                  .DOA(rd_buf_out_data\[((i*6)+4)+ii\]),
                  .DOB(rd_buf_out_data\[((i*6)+2)+ii\]),
                  .DOC(rd_buf_out_data\[((i*6)+0)+ii\]),
                  .DOD(),
                  .DIA(rd_buf_in_data\[((i*6)+4)+ii\]),
                  .DIB(rd_buf_in_data\[((i*6)+2)+ii\]),
                  .DIC(rd_buf_in_data\[((i*6)+0)+ii\]),
                  .DID(2'b0),
                  .ADDRA(rd_buf_indx_cpy_r\[i\]\[5:0\] + single_data_buf\[i\]),
                  .ADDRB(rd_buf_indx_cpy_r\[i\]\[5:0\] + single_data_buf\[i\]),
                  .ADDRC(rd_buf_indx_cpy_r\[i\]\[5:0\] + single_data_buf\[i\]),
                  .ADDRD(rd_buf_wr_addr),
                  .WE(rd_buf_we\[i\]),
                  .WCLK(clk)
                 );
            end
          end
          else begin
            RAM32M
              #(.INIT_A(64'h0000000000000000),
                .INIT_B(64'h0000000000000000),
                .INIT_C(64'h0000000000000000),
                .INIT_D(64'h0000000000000000)
            ) RAM32M0 (
                .DOA(rd_buf_out_data\[((i*6)+4)+:2\]),
                .DOB(rd_buf_out_data\[((i*6)+2)+:2\]),
                .DOC(rd_buf_out_data\[((i*6)+0)+:2\]),
                .DOD(),
                .DIA(rd_buf_in_data\[((i*6)+4)+:2\]),
                .DIB(rd_buf_in_data\[((i*6)+2)+:2\]),
                .DIC(rd_buf_in_data\[((i*6)+0)+:2\]),
                .DID(2'b0),
                .ADDRA(rd_buf_indx_cpy_r\[i\]\[4:0\] + single_data_buf\[i\]),
                .ADDRB(rd_buf_indx_cpy_r\[i\]\[4:0\] + single_data_buf\[i\]),
                .ADDRC(rd_buf_indx_cpy_r\[i\]\[4:0\] + single_data_buf\[i\]),
                .ADDRD(rd_buf_wr_addr),
                .WE(rd_buf_we\[i\]),
                .WCLK(clk)
               );
          end


  
          always @(posedge clk)
            if (rd_data_en & address_match_dout0\[i\] & address_match_dout1\[i\])  
              app_rd_data_ns\[i*6+:6\] <= #TCQ rd_buf_in_data\[i*6+:6\]; 
            else
              app_rd_data_ns\[i*6+:6\] <= #TCQ rd_buf_out_data\[i*6+:6\]; // spyglass disable UndrivenNet-ML

        end // block: rd_buffer_ram

        assign app_rd_data = app_rd_data_ns\[APP_DATA_WIDTH-1:0\]; 

      end

      wire address_match_cpy2_0 = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_r\[9\]\[2:0\]});
      wire address_match_cpy2_1
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr\[5:3\],rd_buf_indx_r\[9\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr\[4:3\],rd_buf_indx_r\[9\]\[4:3\]});
      assign bypass_cpy = rd_data_en && address_match_cpy2_0 && address_match_cpy2_1;
      assign rd_data_rdy_cpy = (rd_status\[0\]\[0\] == rd_buf_indx_r\[9\]\[DATA_BUF_ADDR_WIDTH\]);

      wire address_match_cpy_0 = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_r\[1\]\[2:0\]});  
      wire address_match_cpy_1
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr\[5:3\],rd_buf_indx_r\[1\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr\[4:3\],rd_buf_indx_r\[1\]\[4:3\]});  
      wire address_match_cpy6_0 = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_r\[14\]\[2:0\]});
      wire address_match_cpy6_1
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr\[5:3\],rd_buf_indx_r\[14\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr\[4:3\],rd_buf_indx_r\[14\]\[4:3\]});
      wire address_match_cpy11_0 = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_r\[5\]\[2:0\]});
      wire address_match_cpy11_1
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr\[5:3\],rd_buf_indx_r\[5\]\[5:3\]})
                                       : match4_1({rd_buf_wr_addr\[4:3\],rd_buf_indx_r\[5\]\[4:3\]});

      wire bypass = rd_data_en && address_match_cpy_0 && address_match_cpy_1;

      wire rd_data_rdy = (rd_status\[0\]\[0\] == rd_buf_indx_r\[5\]\[DATA_BUF_ADDR_WIDTH\]);
      wire bypass_cpy2 = rd_data_en && address_match_cpy11_0 && address_match_cpy11_1;

      always @(posedge clk) begin
        if (rst)
          app_rd_data_valid <= #TCQ 1'b0;
        else if (ram_init_done_r_lcl\[DATA_BUF_ADDR_WIDTH\])
          app_rd_data_valid <= #TCQ (bypass_cpy2 || rd_data_rdy);
      end

      always @(posedge clk) begin
        if (rst)
          app_rd_data_end <= #TCQ 1'b0;
        else begin
          if (rd_data_en & address_match_cpy6_0 & address_match_cpy6_1)
            app_rd_data_end <= #TCQ rd_data_end;
          else
            app_rd_data_end <= #TCQ rd_status\[0\]\[1\];
        end
      end

      wire address_match_cpy13_0 = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_r\[7\]\[2:0\]});
      wire address_match_cpy13_1
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr[5:3],rd_buf_indx_r[7][5:3]})
                                       : match4_1({rd_buf_wr_addr[4:3],rd_buf_indx_r[7][4:3]});
      wire app_rd_data_end_cpy0 = bypass ? rd_data_end : rd_status\[0\]\[1\];
      assign single_data = ram_init_done_r_lcl\[6\] && app_rd_data_end_cpy0;

      if (ECC != \"OFF\") begin : assign_app_ecc_multiple
        wire \[2*nCK_PER_CLK-1:0\] app_ecc_multiple_err_ns =
                                   bypass
                                   ? ecc_multiple
                                   : rd_buf_out_data\[APP_DATA_WIDTH+:8\];
        always @(posedge clk) app_ecc_multiple_err_r <= 
                                #TCQ app_ecc_multiple_err_ns;
      end

      //Added to fix timing. The signal app_rd_data_valid has 
      //a very high fanout. So making a dedicated copy for usage
      //with the occ_cnt counter.
      (* keep = \"true\" *) reg app_rd_data_valid_cpy_r;
      wire address_match_cpy12_0 = match6_1({rd_buf_wr_addr\[2:0\],rd_buf_indx_r\[19\]\[2:0\]});
      wire address_match_cpy12_1
           = (DATA_BUF_ADDR_WIDTH > 5) ? match6_1({rd_buf_wr_addr[5:3],rd_buf_indx_r[19][5:3]})
                                       : match4_1({rd_buf_wr_addr[4:3],rd_buf_indx_r[19][4:3]});
      wire bypass_cpy1 = rd_data_en && address_match_cpy12_0 && address_match_cpy12_1;      
      wire rd_data_rdy_cpy2 = (rd_status\[0\]\[0\] == rd_buf_indx_r\[19\]\[DATA_BUF_ADDR_WIDTH\]);
 
      always @(posedge clk) begin
        if (rst)
          app_rd_data_valid_cpy_r <= #TCQ 1'b0;
        else if (ram_init_done_r_lcl\[7\])
          app_rd_data_valid_cpy_r <= #TCQ (bypass_cpy1 || rd_data_rdy_cpy2);
      end

      // Keep track of how many entries in the queue hold data.
      // changed to use registered version of the signals in
      // ordered to fix timing
      wire free_rd_buf = app_rd_data_valid_cpy_r && app_rd_data_end; 
                                                                    
      reg \[DATA_BUF_ADDR_WIDTH:0\] occ_cnt_r;
      wire \[DATA_BUF_ADDR_WIDTH:0\] occ_minus_one = occ_cnt_r - 1;
      wire \[DATA_BUF_ADDR_WIDTH:0\] occ_plus_one = occ_cnt_r + 1;
      // synthesis translate_off
      int count_rd_accepteds, count_free_rd_bufs;
      // synthesis translate_on
      begin : occupied_counter
        always @(posedge clk) begin 
          if (rst) occ_cnt_r <= #TCQ 'b0;
          else case ({rd_accepted, free_rd_buf})
                 2'b01 : occ_cnt_r <= #TCQ occ_minus_one;
                 2'b10 : occ_cnt_r <= #TCQ occ_plus_one;
          endcase // case ({wr_data_end, new_rd_data})
        end
        //assign rd_buf_full = occ_cnt_r\[DATA_BUF_ADDR_WIDTH\];
        assign rd_buf_full = (((occ_cnt_r\[DATA_BUF_ADDR_WIDTH-1:0\] == {DATA_BUF_ADDR_WIDTH{1'b1}}) && rd_accepted) ||
	                      occ_cnt_r\[DATA_BUF_ADDR_WIDTH\])
	                     ? 1 : 0;

        // synthesis translate_off
        always @(posedge clk) begin
          if (rst) begin
            count_rd_accepteds <= #TCQ 'b0;
            count_free_rd_bufs <= #TCQ 'b0;
          end
          else begin
            if (rd_accepted) count_rd_accepteds <= #TCQ count_rd_accepteds + 1'b1;
            if (free_rd_buf) count_free_rd_bufs <= #TCQ count_free_rd_bufs + 1'b1;
          end
        end
        // synthesis translate_on

`ifdef MC_SVA
  rd_data_buffer_full: cover property (@(posedge clk) (~rst && rd_buf_full));
  rd_data_buffer_inc_dec_15: cover property (@(posedge clk)
         (~rst && rd_accepted && free_rd_buf && (occ_cnt_r == 'hf)));
  rd_data_underflow: assert property (@(posedge clk)
         (rst || !((occ_cnt_r == 'b0) && (occ_cnt_r == 'h1f))));
  rd_data_overflow: assert property (@(posedge clk)
         (rst || !((occ_cnt_r == 'h10) && (occ_cnt_r == 'h11))));
`endif
      end // block: occupied_counter


// Generate the data_buf_address written into the memory controller
// for reads.  Increment with each accepted read, and rollover at 0xf.
      reg \[DATA_BUF_ADDR_WIDTH-1:0\] rd_data_buf_addr_r_lcl;
      assign rd_data_buf_addr_r = rd_data_buf_addr_r_lcl;
      begin : data_buf_addr
        always @(posedge clk) begin 
          if (rst) rd_data_buf_addr_r_lcl <= #TCQ 'b0;
          else if (rd_accepted) rd_data_buf_addr_r_lcl <= #TCQ
                                rd_data_buf_addr_r_lcl + 1;
        end
      end // block: data_buf_addr
    end // block: not_strict_mode
  endgenerate

endmodule"
      
           
              incr n 1
	      set line_added 1
	    }

	  
	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

	    if { $line_removed } {
              puts_line_removed $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }
	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 17 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 17 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."



   #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_axi.sv
	  #====================================================================================================================================
          if {$axi_protocol==1}  {
	    set target_file "${ddr}_v${file_ver}_axi.sv"
	    puts "--> ${target_file}"
	    set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	    set filename [get_property {NAME} $file]
	 
    	    set tmp_filename "${filename}.tmp"
	    set backup_filename "${filename}.bak"

	    set in_file  [open $filename r]
	    set out_file [open $tmp_filename w]



	    #===========================================
	    #	Patched settings for Everspin DDR ST-MRAM 
	    #===========================================
  	 			 

	    # line-by-line, read the original file
	    set n 0
            set flag 0
	    while {[gets $in_file line] != -1} {
	      set new_line $line
              set line_patched 0
              set line_added 0
            

	    
            if {[string match -nocase {*input*wire*mc_init_complete*,}  $new_line] } {
	      set new_line "output reg                                inflight_writes   ,"
              incr n 1
	      set line_added 1
	    }

            if {[string match -nocase {*mc_init_complete_r*<=*mc_init_complete*;}  $new_line] } {
	      set new_line "reg \[2:0\] inflight_writes_cnt;"
              incr n 1
	      set line_added 1 
             }
              
             if {[string match -nocase {*wire*b_full;*}  $new_line] } {
	      set new_line " 
              always @(posedge aclk) begin
                    if (areset_d1) begin
                    inflight_writes          <= 'b0;
                    inflight_writes_cnt      <= 'b0;
                  end
                    else begin
                 if (awvalid_int) begin
                 inflight_writes       <= 1'b1;
                   inflight_writes_cnt   <= 'b0;
                 end
               else if (inflight_writes) begin
                if (~&inflight_writes_cnt) begin
                 inflight_writes       <= 1'b1;
                  inflight_writes_cnt   <= inflight_writes_cnt + 1'b1;
                end
                else begin
              inflight_writes          <= 'b0;
             inflight_writes_cnt      <= 'b0;
                end
              end
               end
              end"
              incr n 1
	      set line_added 1
	    }
	  
	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 3 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 3 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."
       }

   #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_ui.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_ui.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  	 			 

	  # line-by-line, read the original file
	  set n 0
          set flag 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
	    
           
           
            if {[string match {*zq_req, app_zq_ack, ui_busy,*}  $new_line] } {
	      set new_line "  zq_req, app_zq_ack, ui_busy, inflight_writes,"
              incr n 1
	      set line_patched 1
	    }
            
            if {[string match {*wire*ram_init_addr;*}  $new_line] } {
	      set new_line    "  wire \[DATA_BUF_ADDR_WIDTH-1:0\] ram_init_addr;"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*wire*ram_init_done_r;*}  $new_line] } {
	      set new_line "  wire \[DATA_BUF_ADDR_WIDTH-1:0\] ram_init_done_r;
  wire                  wrdata_fifo_empty;"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*output*size;*}  $new_line] } {
	      set new_line "  output  reg           inflight_writes;"
              incr n 1
	      set line_added 1
	    }

            if {[string match {*if(DATA_BUF_ADDR_WIDTH > 4) begin*}  $new_line] } {
	      set new_line "    if(DATA_BUF_ADDR_WIDTH > 6) begin"
              incr n 1
	      set line_patched 1
	    }

            if {[string match {*assign wr_data_buf_addr\[DATA_BUF_ADDR_WIDTH-1:4\] = 0;*}  $new_line] } {
	      set new_line "      assign wr_data_buf_addr\[DATA_BUF_ADDR_WIDTH-1:6\] = 0;
      assign data_buf_addr\[DATA_BUF_ADDR_WIDTH-1:6\] = 0;"
              incr n 1
	      set line_patched 1
	    }
           
           if {[string match {*endgenerate*}  $new_line] } {
	      set new_line "
  always @(posedge clk) begin
    if (rst_r1) begin
      inflight_writes          <= 'b0;
    end
    else begin
      inflight_writes          <= !(wrdata_fifo_empty);
    end
  end"
              incr n 1
	      set line_added 1
	    }
         # it is already parametized im Lola 2017.2  
           if {[string match {*.DATA_BUF_ADDR_WIDTH*}  $new_line] } {
	      set new_line "     .DATA_BUF_ADDR_WIDTH               (DATA_BUF_ADDR_WIDTH),"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*.data_buf_addr*(data_buf_addr),*}  $new_line] } {
	      set new_line "       .data_buf_addr                   (data_buf_addr\[DATA_BUF_ADDR_WIDTH-1:0\]),"
              incr n 1
	      set line_patched 1
	    }

           if {[string match {*.wr_data_buf_addr*}  $new_line] } {
	      set new_line "       .wr_data_buf_addr                (wr_data_buf_addr\[DATA_BUF_ADDR_WIDTH-1:0\]),"
              incr n 1
	      set line_patched 1
	    }
           if {[string match {*.rd_data_buf_addr_r*}  $new_line] } {
	      set new_line "       .rd_data_buf_addr_r              (rd_data_buf_addr_r\[DATA_BUF_ADDR_WIDTH-1:0\]),"
              
              incr n 1
	      set line_patched 1
	    }
           if {[string match {*.EARLY_WR_DATA*}  $new_line] } {
	      set new_line "     .FIFO_ADDR_WIDTH                   (DATA_BUF_ADDR_WIDTH), "
              incr n 1
	      set line_added 1
	    }
           if {[string match {*.app_wdf_rdy*}  $new_line] } {
	      set flag 1
	    }
           if {[string match {*.wr_req_16*}  $new_line] && $flag ==1} {
	      set new_line "       .wr_req_full                     (wr_req_16),
       .wrdata_fifo_empty               (wrdata_fifo_empty),"
              incr n 1
              set flag 0
	      set line_patched 1
	    }

           if {[string match {*.wr_data_addr*}  $new_line] } {
	      set new_line "       .wr_data_addr                    (wr_data_addr\[DATA_BUF_ADDR_WIDTH-1:0\]),"
              incr n 1
	      set line_patched 1
	    }
           if {[string match {*.rd_data_addr*}  $new_line] } {
	      set new_line "       .rd_data_addr                    (rd_data_addr\[DATA_BUF_ADDR_WIDTH-1:0\]),"
              incr n 1
	      set line_patched 1
	    }
            if {$Ecc_Autocorrect==1 && [string match {*.wr_data_en *} $new_line]} {
              set new_line "       .wr_data_en                      (wr_data_en & !wr_data_addr\[DATA_BUF_ADDR_WIDTH-1\]),"
              incr n 1
              set line_patched 1
            }

	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $Ecc_Autocorrect==1 && $n != 19 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  if { $Ecc_Autocorrect==0 && $n != 18 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


   #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_axi_cmd_arbiter.sv
	  #====================================================================================================================================
	  if {$axi_protocol==1}  {
            set target_file "${ddr}_v${file_ver}_axi_cmd_arbiter.sv"
	    puts "--> ${target_file}"
	    set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	    set filename [get_property {NAME} $file]
	  
	    set tmp_filename "${filename}.tmp"
	    set backup_filename "${filename}.bak"

	    set in_file  [open $filename r]
	    set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  	 			 

	  # line-by-line, read the original file
	  set n 0
          set flag1 0
          set flag2 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
	    
             if {[string match {*wr_cmd_en_d1 <= 1'b0;*} $new_line]} {
              set flag2 0
             } 

             if {[string match {*rnw_i <= 1'b1;*} $new_line]} {
              set flag1 1
              set flag2 1
            }

           if {[string match {*end else begin*} $new_line] && $flag1 ==1 && $flag2 ==1} {
              set new_line "end else if ( !(mc_app_en && !mc_app_rdy)) begin"
              incr n 1
	      set line_patched 1
              set flag1 0
              set flag2 0
            }

	  
	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 1 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."

       }

  
  
  #====================================================================================================================================
	  #rtl/controller/${ddr}_v2_2_mc_arb_c.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_mc_arb_c.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  	 			 

	  # line-by-line, read the original file
	  set n 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
	    
            if {[string match {*reg*win3210_reorder*}  $new_line]} {
	      set new_line "
/*"
              incr n 1
	      set line_added 1
	    }

            if {[string match {*Select arbitration winner based on ordering mode*}  $new_line]} {
	      set new_line "*/

always @(*) begin
   w10 = findWin(last10, reqs[1:0]);
   w32 = findWin(last32, reqs[3:2]);
   if ( |( winPort & reqs ) ) begin
      win3210_reorder = winPort;      
   end else begin
      winner = findWin(last, {|reqs\[3:2\], |reqs\[1:0\]});
      case (winner)
         2'b01:   win3210_reorder = {2'b00, w10};
         2'b10:   win3210_reorder = {w32, 2'b00};
         default: win3210_reorder = 4'b0000;
      endcase
   end
end

"
              incr n 1
	      set line_added 1
	    }



	  
	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 2 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."


  #====================================================================================================================================
	  #rtl/controller/Mig_name.sv
	  #====================================================================================================================================
	  set target_file "${mig_name}.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]



	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================
  	 			 

	  # line-by-line, read the original file
	  set n 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
            set line_patched 0
            set line_added 0
	    
            if {[string match -nocase {*output*dbg_clk,*}  $new_line]} {
	      set new_line "
   // begin Status and control inputs and outputs
   input                 power_fail_has_scramed,
   output                cntr_power_fail_complete,
   output                inflight_writes,          // status output
   // end Status and control inputs and outputs"
              incr n 1
	      set line_added 1
	    }


            if {[string match -nocase {*.sys_rst*(sys_rst),*}  $new_line]} {
	      set new_line "
   // begin Status and control inputs and outputs
   .power_fail_has_scramed        (power_fail_has_scramed),
   .cntr_power_fail_complete      (cntr_power_fail_complete),
   .inflight_writes               (inflight_writes),
   // end Status and control inputs and outputs"
              incr n 1
	      set line_added 1
	    }



	  
	    if { $line_patched } {    
              puts_line_patched $line $out_file
	    }

            if { $line_added } {    
              puts_line_added $line $out_file
	    }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 2 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."




  #====================================================================================================================================
          #rtl/cal/${ddr}_v_2_2_cal.sv
          #====================================================================================================================================
          set target_file "${ddr}_v${file_ver}_cal.sv"
          puts "--> ${target_file}"
          set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
          set filename [get_property {NAME} $file]
          
          set tmp_filename "${filename}.tmp"
          set backup_filename "${filename}.bak"

          set in_file  [open $filename r]
          set out_file [open $tmp_filename w]



          #===========================================
          #        Patched settings for Everspin DDR ST-MRAM 
          #===========================================
                                    

          # line-by-line, read the original file
          set n 0
          set flag1 0
          set flag2 0
          set flag3 0
          set tMOD_old			{tMOD}
          set tMOD_new                  {t200us}

          while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0
            
            
            if {[string match -nocase {*,parameter*BGBITS*=*2*}  $new_line]} {
              set new_line "   ,   parameter       MRBITS                     = 14"
              incr n 1
              set line_added 1
            } elseif {[string match -nocase {*wire*bramA_do;*}  $new_line]} {
              set new_line ""
              if {$lola==0} {
                append new_line " typedef struct packed
               {
   logic \[31:0\]  during_cal_mr0;
   logic \[31:0\]  during_cal_mr1;
   logic \[31:0\]  during_cal_mr2;
   logic \[31:0\]  during_cal_mr3;
   logic \[31:0\]  during_cal_mr4;
   logic \[31:0\]  during_cal_mr5;
   logic \[31:0\]  during_cal_mr6;
   logic \[31:0\]  after_cal_mr0;
   logic \[31:0\]  after_cal_mr1;
   logic \[31:0\]  after_cal_mr2;
   logic \[31:0\]  after_cal_mr3;
   logic \[31:0\]  after_cal_mr4;
   logic \[31:0\]  after_cal_mr5;
   logic \[31:0\]  after_cal_mr6;
   logic \[63:0\]  treg_adc;
 } ddr_mp_struct_t;
  
 ddr_mp_struct_t ddr_mp;
 
 initial begin
     ddr_mp.during_cal_mr0 = MR0 | 32'h2000;   // a13 = nomem=1
     ddr_mp.during_cal_mr1 = MR1;
     ddr_mp.during_cal_mr2 = MR2;
     ddr_mp.during_cal_mr3 = MR3;
     ddr_mp.during_cal_mr4 = MR4;
     ddr_mp.during_cal_mr5 = MR5;
     ddr_mp.during_cal_mr6 = MR6;
     ddr_mp.after_cal_mr0  = MR0;
     ddr_mp.after_cal_mr1  = MR1;
     ddr_mp.after_cal_mr2  = MR2;
     ddr_mp.after_cal_mr3  = MR3;
     ddr_mp.after_cal_mr4  = MR4;
     ddr_mp.after_cal_mr5  = MR5;
     ddr_mp.after_cal_mr6  = MR6;
     ddr_mp.treg_adc       = 64'h0;
 end"
              }
              append new_line "
 
 wire        ub_owns_cal;
 reg ub_cal_now;
 assign ub_owns_cal   = (RTL_DDR_INIT == 0) | calDone | ( ub_ready & ub_cal_now & ~ub_calDone);
 reg \[3:0\] MR6_count;
 reg mr2_pass2;          //ddh perform second RTL pass to re-initialize MR2
 reg mr2_done;            //ddh complete second RTL pass to re-initialize MR2"

              # TODO: For now just put these declarations for all modes
              #if {$lola==1} {}
              if {1} {
                append new_line "
// ----------------------------------------------------------------
// TREG
// ----------------------------------------------------------------
localparam   TREG_AFI_RATE_RATIO      =   4;
localparam   TREG_AFI_ADDR_WIDTH      =  64;
localparam   TREG_AFI_BANKADDR_WIDTH  =  12;
localparam   TREG_AFI_CONTROL_WIDTH   =   4;
localparam   TREG_AFI_CS_WIDTH        =   4;
localparam   TREG_AFI_CLK_EN_WIDTH    =   4;
localparam   TREG_AFI_DM_WIDTH        =  72;
localparam   TREG_AFI_DQ_WIDTH        = 576;
localparam   TREG_AFI_ODT_WIDTH       =   4;
localparam   TREG_AFI_WRITE_DQS_WIDTH =  36;

logic  treg_mux_sel_in = 0;
logic  treg_mux_sel_out;
logic  treg_tmri_stall_req;
logic  treg_tmri_stall_ack = 1;

// Mux output to the PHY logic
logic   \[TREG_AFI_ADDR_WIDTH-1:0\]       treg_afi_addr;
logic   \[TREG_AFI_BANKADDR_WIDTH-1:0\]   treg_afi_ba;
logic   \[TREG_AFI_CONTROL_WIDTH-1:0\]    treg_afi_cas_n;
logic   \[TREG_AFI_CLK_EN_WIDTH-1:0\]     treg_afi_cke;
logic   \[TREG_AFI_CS_WIDTH-1:0\]         treg_afi_cs_n;
logic   \[TREG_AFI_ODT_WIDTH-1:0\]        treg_afi_odt;
logic   \[TREG_AFI_CONTROL_WIDTH-1:0\]    treg_afi_ras_n;
logic   \[TREG_AFI_CONTROL_WIDTH-1:0\]    treg_afi_we_n;
logic   \[TREG_AFI_DM_WIDTH-1:0\]         treg_afi_dm;
logic   \[TREG_AFI_CONTROL_WIDTH-1:0\]    treg_afi_rst_n;
logic   \[TREG_AFI_WRITE_DQS_WIDTH-1:0\]  treg_afi_dqs_burst;
logic   \[TREG_AFI_DQ_WIDTH-1:0\]         treg_afi_wdata;
logic   \[TREG_AFI_WRITE_DQS_WIDTH-1:0\]  treg_afi_wdata_valid;
logic   \[TREG_AFI_RATE_RATIO-1:0\]       treg_afi_rdata_en;
logic   \[TREG_AFI_RATE_RATIO-1:0\]       treg_afi_rdata_en_full;

logic          treg_nowmem_valid;
logic   \[7:0\]  treg_nowmem_count;
logic   \[3:0\]  treg_nowmem_cmd  ;
logic   \[2:0\]  treg_nowmem_ba   ;
logic   \[15:0\] treg_nowmem_addr ;
logic          treg_nowmem_done ;

logic   \[7:0\]   treg_init_cal_BG;
logic   \[15:0\]  treg_init_cal_BA;
logic   \[143:0\] treg_init_cal_ADR;
logic   \[7:0\]   treg_init_cal_CS_n;


// Test Mode interface
logic               treg_TM_write_start;
logic               treg_TM_write_complete;

logic   \[1:0\]       treg_TMR_select = 2'b11;
logic   \[6*4-1:0\]   treg_TMR_input_values = 0;
logic               treg_cfg_auto_nomem = 0;
logic               treg_local_init_done = 1;

logic   \[63:0\]      treg_nowmem_adc;    // No W Mem Address, Data, Command
logic   \[31:0\]      treg_nowmem_data;   // No W Mem RAM Data out

assign  treg_nowmem_adc = ddr_mp.treg_adc\[63:0\];   // No W Mem Address, Data, Command"
              }

              incr n 1
              set line_added 1
            } elseif {[string match {*,output*win_status*}  $new_line] && $lola==1} {
              set new_line "   ,ddr_modded_interface.ddr_mp     ddr_mp"
              incr n 1
              set line_added 1
            } elseif {[string  match -nocase {*reg*calSt;*}  $new_line]} {
              set new_line "reg  \[6:0\] calSt;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase {*reg*retSt;*}  $new_line]} {
              set new_line "reg  \[6:0\] retSt;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase {*assign*caldone*=*}  $new_line]} {
              set new_line "assign caldone = (BYPASS_CAL == \"TRUE\")? ((RTL_DDR_INIT == 1) ? (initDone) : ub_initDone) : ub_calDone & initDone;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign*cal_BG_int*=*}  $new_line]} {
              set new_line "assign cal_BG_int   = ub_owns_cal ? ub_cal_BG : init_cal_BG;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign*cal_BA_int*=*}  $new_line]} {
              set new_line "assign cal_BA_int   = ub_owns_cal ? ub_cal_BA : init_cal_BA;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign*cal_ADR_int*=*}  $new_line]} {
              set new_line "assign cal_ADR_int  = ub_owns_cal ? ub_cal_ADR : init_cal_ADR\[ABITS*8-1:0\];"
              incr n 1
              set line_patched 1
            } elseif {[string  match  {*assign*cal_inv_int*=*}  $new_line]} {
              set new_line "assign cal_inv_int  = ub_owns_cal ? ub_cal_inv : init_cal_inv;"
              incr n 1
              set line_patched 1
            } elseif {[string  match  {*assign*cal_mrs_int*=*}  $new_line]} {
              set new_line "assign cal_mrs_int  = ub_owns_cal ? ub_cal_mrs : init_cal_mrs;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign rtl_initDone*}  $new_line]} {
              set new_line "assign rtl_initDone = (RTL_DDR_INIT == 1) ? ub_cal_now : 1'b0;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_RESET_n*}  $new_line]} {
              set new_line "assign cal_RESET_n = ub_owns_cal ? ub_cal_RESET_n : init_cal_RESET_n;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_ACT_n*}  $new_line]} {
              set new_line "assign cal_ACT_n   = ub_owns_cal ? ub_cal_ACT_n : init_cal_ACT_n;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase {*assign cal_C       =*}  $new_line]} {
              set new_line "assign cal_C       = ub_owns_cal ? ub_cal_C : 1'b0;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_CKE*}  $new_line]} {
              set new_line "assign cal_CKE     = ub_owns_cal ? ub_cal_CKE : init_cal_CKE;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_CS_n*}  $new_line]} {
              set new_line "assign cal_CS_n    = ub_owns_cal ? ub_cal_CS_n : init_cal_CS_n;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_ODT*}  $new_line]} {
              set new_line "assign cal_ODT     = ub_owns_cal ? ub_cal_ODT : init_cal_ODT;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_PAR*}  $new_line]} {
              set new_line "assign cal_PAR     = ub_owns_cal ? ub_cal_PAR : init_cal_PAR;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_WE_n*}  $new_line]} {
              set new_line "assign cal_WE_n    = ub_owns_cal ? ub_cal_WE : init_cal_WE_n;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_CAS_n*}  $new_line]} {
              set new_line "assign cal_CAS_n   = ub_owns_cal ? ub_cal_CAS : init_cal_CAS_n;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*assign cal_RAS_n*}  $new_line]} {
              set new_line "assign cal_RAS_n   = ub_owns_cal ? ub_cal_RAS : init_cal_RAS_n;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*,calStMR7*=*6'h0F*}  $new_line]} {
              set new_line "   ,calStTREG  = 6'h10"
              incr n 1
              set line_added 1
            } elseif {[string  match -nocase  {*for (i = 0; i <= 12; i = i + 1)*}  $new_line]} {
              set new_line "for (i = 0; i <= MRBITS-1; i = i + 1) init_cal_ADR\[i*8+:8\] <= #TCQ {8{mr\[i\]}};"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*for(i = 13; i < ABITS; i = i + 1)*}  $new_line]} {
              set new_line "for(i = 13; i <= ABITS; i = i + 1) init_cal_ADR\[i*8+:8\] <= #TCQ 8'b0;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*init_cal_ADR\[((ABITS-1)\*8)+:8\]*}  $new_line]} {
              set new_line " "
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*init_cal_ADR\[111:104\] <= #TCQ 8'b0;*}  $new_line]} {
              set new_line " "
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*Side A will be written without inversion*}  $new_line]} {
              set flag1  1
            } elseif {[string  match -nocase  {*if (calSt == calStRESET) begin*}  $new_line] && $flag1 ==1} {
              set new_line "if ((calSt == calStRESET) || (calSt == calStGOGO)) begin"
              incr n 1
              set line_patched 1
              set flag1 0
            } elseif {[string  match  {*if (LRDIMM_QUAD_RANK) begin*}  $new_line]} {
            set new_line "/* if (LRDIMM_QUAD_RANK) begin"
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*setMR(MR5);*}  $new_line]} {
            set flag2 1
            } elseif {[string  match -nocase  {*setDDROP(MRS);*}  $new_line] && $flag2 ==1} {
              set new_line "
              */
              setMR(mr2_pass2 ? ddr_mp.after_cal_mr5 : ddr_mp.during_cal_mr5);
               setDDROP(MRS);"
              incr n 1
              set line_patched 1
              set flag2 0
            } elseif {[string  match  {*if ((LRDIMM_EN == 0) && (SLOT0_CONFIG == 8'b1111 || SLOT1_CONFIG == 8'b1111)) begin*}  $new_line]} {
            set new_line "/* if ((LRDIMM_EN == 0) && (SLOT0_CONFIG == 8'b1111 || SLOT1_CONFIG == 8'b1111)) begin"
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*setMR(MR1);*}  $new_line]} {
            set flag3 1
            } elseif {[string  match -nocase  {*setDDROP(MRS);*}  $new_line] && $flag3 ==1} {
              set new_line "
              */
     setMR(mr2_pass2 ? ddr_mp.after_cal_mr1 : ddr_mp.during_cal_mr1);
     setDDROP(MRS);"
              incr n 1
              set line_patched 1
              set flag3 0
            } elseif {[string  match -nocase  {*if (REG_CTRL == \"OFF\" || init_cal_inv == SIDE_B) begin*}  $new_line]} {
              set new_line " if \(mr2_pass2 == 1\) begin
               twiddle\(tMOD, calStZQCL\); 
               end
                else begin
               if (REG_CTRL == \"OFF\" || init_cal_inv == SIDE_B) begin"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*twiddle(tMOD, calStMR3);*}  $new_line]} {
              set new_line "twiddle(t200us, calStMR3);
                    end"
              incr n 1
              set line_added 1
            } elseif {[string  match -nocase  {*if (mrs_done)*}  $new_line]} {
              set new_line "         if (mrs_done) begin
            if (mr2_pass2 == 1'b0) begin
               mr2_pass2 <= 1'b1;
               twiddle(tZQINIT, calStGOGO);
            end else begin
               mr2_done <= 1'b1;
               twiddle(tZQINIT, calStTREG);
            end
         end"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*twiddle(tZQINIT, calStGOGO);*}  $new_line]} {
              set new_line ""
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*calStGOGO: initDone <= 1'b1;*Now we are ready for operations*}  $new_line]} {
              set new_line " calStGOGO: begin
            if (mr2_pass2 == 1'b1 && mr2_done == 1'b0) begin
            ub_cal_now <= 1'b1;
            if (BYPASS_CAL == \"TRUE\" || ub_calDone == 1'b1)
               twiddle(tZQINIT, MEM==\"DDR3\" ? calStMR2 : calStMR1);   
          end
          else initDone <= 1'b1;  // RTL initialization complete
           end"

              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*calStWAIT: begin*}  $new_line]} {
              if {$lola==1} {
                set new_line "
      calStTREG: begin
          // aolsen
          if (!ddr_mp.treg_ctrl[0]) begin
              twiddle(tZQINIT, calStGOGO);
          end else if (!treg_TM_write_complete) begin
              treg_TM_write_start <= 1;
              // TREG always uses MR0:
              init_cal_BG[7:0] <= #TCQ treg_init_cal_BG;
              init_cal_BA <= #TCQ treg_init_cal_BA;
              init_cal_ADR <= #TCQ treg_init_cal_ADR;
              init_cal_CS_n <= #TCQ treg_init_cal_CS_n;
          end else begin
              treg_TM_write_start <= 0;
              setDDROP(NOP);
              init_cal_CS_n <= #TCQ {CSBITS{8'b11111111}};
              twiddle(tZQINIT, calStGOGO);
          end
      end
${new_line}"
              } else {
                # TODO: For non-lola just skip treg
                set new_line "
      calStTREG: twiddle(tZQINIT, calStGOGO);
${new_line}"
              }
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*input*\[5:0\]*st;*}  $new_line]} {
              set new_line "   input  \[6:0\] st;"
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*input \[12:0\] mr;*}  $new_line]} {
              set new_line "input \[MRBITS-1:0\] mr;" 
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*if (BGBITS == 2)*}  $new_line]} {
              set new_line "for(i = MRBITS; i <= ABITS; i = i + 1) init_cal_ADR\[i*8+:8\] <= #TCQ 8'b0;
               if (BGBITS == 2)" 
              incr n 1
              set line_patched 1
            } elseif {[string  match -nocase  {*init_cal_PAR <= 8'b0;*}  $new_line]} {
              set new_line "MR6_count <= 0;" 
              incr n 1
              set line_added 1
            } elseif {[string  match -nocase  {*cntr <= 18'bx;*}  $new_line]} {
              set new_line "   mr2_pass2 <= 1'b0;  
   mr2_done <= 1'b0; 
   ub_cal_now <= 1'b0;
   treg_TM_write_start <= 0;"
              incr n 1
              set line_added 1
            } elseif {[string  match  {*setMR(MR6);*}  $new_line]} {
            set new_line "setMR(mr2_pass2 ? ddr_mp.after_cal_mr6 : ddr_mp.during_cal_mr6);"
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*setMR(MR4);*}  $new_line]} {
            set new_line " setMR(mr2_pass2 ? ddr_mp.after_cal_mr4 : ddr_mp.during_cal_mr4);"
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*setMR(MR2);*}  $new_line]} {
            set new_line "setMR(mr2_pass2 ? ddr_mp.after_cal_mr2 : ddr_mp.during_cal_mr2); "
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*setMR(MR0);*}  $new_line]} {
            set new_line "setMR(mr2_pass2 ? ddr_mp.after_cal_mr0 : ddr_mp.during_cal_mr0); "
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*setMR(MR3);*}  $new_line]} {
            set new_line "setMR(mr2_pass2 ? ddr_mp.after_cal_mr3 : ddr_mp.during_cal_mr3);"
            incr n 1
            set line_patched 1
            } elseif {[string  match  {*twiddle(tMRD, calStMR5);*}  $new_line]} {
            set new_line "if(MR6_count > 4) begin
              twiddle(tMRD, calStMR5);
             MR6_count <= 0;
           end else begin
             twiddle(tMRD, calStMR6);
             MR6_count <= MR6_count +1;
            end"
              incr n 1
              set line_patched 1
            } elseif {[string  match  {*endmodule*}  $new_line] && $lola==1} {
              set new_line "
Mram_Test_Mode_drive Mram_Test_Mode_drive (
    .clk                   (clk),
    .reset_n               (~rst),
    .mux_sel_in            (treg_mux_sel_in),
    .mux_sel_out           (treg_mux_sel_out),
    .tmri_stall_req        (treg_tmri_stall_req),
    .tmri_stall_ack        (treg_tmri_stall_ack),
                        
    // Mux output to the PHY logic
    .afi_addr              (treg_afi_addr),
    .afi_ba                (treg_afi_ba),
    .afi_cas_n             (treg_afi_cas_n),
    .afi_cke               (treg_afi_cke),
    .afi_cs_n              (treg_afi_cs_n),
    .afi_odt               (treg_afi_odt),
    .afi_ras_n             (treg_afi_ras_n),
    .afi_we_n              (treg_afi_we_n),
    .afi_dm                (treg_afi_dm),
    .afi_rst_n             (treg_afi_rst_n),
    .afi_dqs_burst         (treg_afi_dqs_burst),
    .afi_wdata             (treg_afi_wdata),
    .afi_wdata_valid       (treg_afi_wdata_valid),
    .afi_rdata_en          (treg_afi_rdata_en),
    .afi_rdata_en_full     (treg_afi_rdata_en_full),

    .nowmem_valid          (treg_nowmem_valid),
    .nowmem_count          (treg_nowmem_count),
    .nowmem_cmd            (treg_nowmem_cmd  ),
    .nowmem_ba             (treg_nowmem_ba   ),
    .nowmem_addr           (treg_nowmem_addr ),
    .nowmem_done           (treg_nowmem_done ),

    .init_cal_BG           (treg_init_cal_BG),
    .init_cal_BA           (treg_init_cal_BA),
    .init_cal_ADR          (treg_init_cal_ADR),
    .init_cal_CS_n         (treg_init_cal_CS_n),

                        
    // Test Mode interface
    .TM_write_start        (treg_TM_write_start),
    .TM_write_complete     (treg_TM_write_complete),
    .TMR_select            (treg_TMR_select),
    .TMR_input_values      (treg_TMR_input_values),
    .cfg_auto_nomem        (treg_cfg_auto_nomem),
    .local_init_done       (treg_local_init_done),

    .nowmem_adc            (treg_nowmem_adc),         // No W Mem Address, Data, Command
    .nowmem_data           (treg_nowmem_data)         // No W Mem RAM Data out

);

$new_line"
              incr n 1
              set line_patched 1
            }

            if {[string match -nocase {*twiddle(tMOD*}  $new_line]} {
              incr n [regsub -nocase -- $tMOD_old $new_line $tMOD_new new_line]
              set line_patched 1
            }
           
            if { $line_patched } {    
              puts_line_patched $line $out_file
            }

            if { $line_added } {    
              puts_line_added $line $out_file
            }



	    puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

	  if { $n != 54 && $lola==1} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }

      	  if { $n != 52 && $lola==0} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }


	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."



        #======================================
        # BEGIN *mc_ecc.sv
        #=====================================
        if {$Ecc==1} {
        set target_file "${ddr}_v${file_ver}_mc_ecc.sv"
        set n 0 
        puts "--> ${target_file}"
        set file [get_files  ${target_file}]
        set filename [get_property {NAME} $file]
        set tmp_filename "${filename}.tmp"
        set backup_filename "${filename}.bak"
        set in_file  [open $filename r]
        set out_file [open $tmp_filename w]

        while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0
            
            if {[string match {*,input * rst*} $new_line]} {
              set new_line "   ,input                                   enable_autocorrect_sync
   ,input                                   enable_ERR_syndrome_recording"
              incr n 1
              set line_added 1
            }

            if {[string match {*,input * wr_data_ni2mc*} $new_line]} {
              set new_line "   ,input      \[PAYLOAD_WIDTH*2*nCK_PER_CLK-1:0\]        wr_data_ni2mc"
              incr n 1
              set line_patched 1
            }

            if {[string match {*,input * wr_data_mask_ni2mc*} $new_line]} {
              set new_line "   ,input      \[PAYLOAD_DM_WIDTH*2*nCK_PER_CLK-1:0\]     wr_data_mask_ni2mc"
              incr n 1
              set line_patched 1
            }

            if {[string match {*,input * rd_data_phy2mc_xif*} $new_line]} {
              set new_line "   ,input      \[DQ_WIDTH*2*nCK_PER_CLK-1:0\]             rd_data_phy2mc_xif   // Xiphy format"
              incr n 1
              set line_patched 1
            }

            if {[string match {*,output * wr_data_mc2phy*} $new_line]} {
              set new_line "   ,output reg \[DQ_WIDTH*2*nCK_PER_CLK-1:0\]             wr_data_mc2phy"
              incr n 1
              set line_patched 1
            }

            if {[string match {*,output * wr_data_mask_mc2phy*} $new_line]} {
              set new_line "   ,output reg \[DM_WIDTH*2*nCK_PER_CLK-1:0\]             wr_data_mask_mc2phy"
              incr n 1
              set line_patched 1
            }

            if {[string match {*,output * rd_data_mc2ni*} $new_line]} {
              set new_line "   ,output reg \[PAYLOAD_WIDTH*2*nCK_PER_CLK-1:0\]        rd_data_mc2ni"
              incr n 1
              set line_patched 1
            }

            if {[string match {*,output * eccSingle*} $new_line]} {
              set new_line "   ,output     \[2*nCK_PER_CLK-1:0\]          eccDouble"
              incr n 1
              set line_added 1
            }

            if {[string match {*wire * wr_data_enc2xor*} $new_line]} {
              set new_line "`ifdef  TWO_BIT_ERR_CORRECTION
localparam ECCS_PER_DQ = 1;
localparam ECC_WIDTH_LOCAL = DQ_WIDTH - PAYLOAD_WIDTH;
`else
localparam ECCS_PER_DQ = (    ECC == \"ON\"
                           && ( DQ_WIDTH > 72 ) ) ?  (( DQ_WIDTH - PAYLOAD_WIDTH ) / ECC_WIDTH) : 1;
localparam ECC_WIDTH_LOCAL = ECC_WIDTH;
`endif


wire \[DQ_WIDTH*2*nCK_PER_CLK-1:0\]   wr_data_enc2xor;"

              incr n 1
              set line_patched 1
            }
            if {[string match {*wire * rd_merge_data*} $new_line]} {
              set new_line "wire \[DATA_WIDTH*2*nCK_PER_CLK-1:0\]         rd_merge_data;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wire * h_rows*} $new_line]} {
              set new_line "wire \[(DQ_WIDTH/ECCS_PER_DQ)*ECC_WIDTH_LOCAL-1:0\]   h_rows;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wire * rd_data_fix2buf*} $new_line]} {
              set new_line "wire \[PAYLOAD_WIDTH*2*nCK_PER_CLK-1:0\]      rd_data_fix2buf;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wire * ecc_single;*} $new_line]} {
              set new_line "wire \[2*nCK_PER_CLK-1:0\]        ecc_double;"
              incr n 1
              set line_added 1
            }
            if {[string match {*assign * eccSingle*=*ecc_single*} $new_line]} {
              set new_line "assign  eccDouble   = ecc_double;"
              incr n 1
              set line_added 1
            }
            if {[string match {*reg * rd_data_phy2mc_burst*} $new_line]} {
              set new_line "/*
$new_line"
              incr n 1
              set line_patched 1
            }

            if {[string match {*reg * wr_data_mask_mc2phy_byte*} $new_line]} {
              set new_line "$new_line
*/
reg  \[DQ_WIDTH-1:0\]                        rd_data_phy2mc_burst\[7:0\];
wire \[PAYLOAD_WIDTH*2*nCK_PER_CLK-1:0\]     rd_data_mc2ni_cw_order;
reg  \[63:0\]                                rd_data_mc2ni_byte\[PAYLOAD_WIDTH/8-1:0\];
reg  \[PAYLOAD_WIDTH*2*nCK_PER_CLK-1:0\]     wr_data_ni2mc_cw_order;
reg  \[PAYLOAD_WIDTH-1:0\]                   wr_data_ni2mc_burst\[2*nCK_PER_CLK-1:0\];
wire \[DQ_WIDTH*2*nCK_PER_CLK-1:0\]          wr_data_mc2phy_cw_order;
reg  \[8*2*nCK_PER_CLK-1:0\]                 wr_data_mc2phy_byte\[DQ_WIDTH/8-1:0\];
reg  \[DQ_WIDTH*2*nCK_PER_CLK-1:0\]          wr_data_mc2phy_tmp;
reg  \[DM_WIDTH*2*nCK_PER_CLK-1:0\]          wr_data_mask_mc2phy_tmp;
reg  \[PAYLOAD_DM_WIDTH*2*nCK_PER_CLK-1:0\]  wr_data_mask_ni2mc_cw_order;
reg  \[PAYLOAD_DM_WIDTH-1:0\]                wr_data_mask_ni2mc_burst\[2*nCK_PER_CLK-1:0\];
wire \[DM_WIDTH*2*nCK_PER_CLK-1:0\]          wr_data_mask_mc2phy_cw_order;
reg  \[2*nCK_PER_CLK-1:0\]                   wr_data_mask_mc2phy_byte\[DM_WIDTH-1:0\];
"
              incr n 1
              set line_patched 1
            }

            if {[string match {*for \(burst = 0;*} $new_line]} {
              set new_line "     for (burst = 0; burst < 2*nCK_PER_CLK; burst++) begin"
              incr n 1
              set line_patched 1
            }
            if {[string match {*rd_data_phy2mc_burst * = rd_data_phy2mc_xif *} $new_line]} {
              set new_line "        rd_data_phy2mc_burst \[burst\]\[pl_byte*8 +: 8\] = rd_data_phy2mc_xif \[burst*8 + pl_byte*8*2*nCK_PER_CLK +: 8\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*rd_data_phy2mc_cw_order * = rd_data_phy2mc_burst *} $new_line]} {
              set new_line "     rd_data_phy2mc_cw_order \[burst*DQ_WIDTH+: DQ_WIDTH\] = rd_data_phy2mc_burst \[burst\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_ni2mc_burst * = wr_data_ni2mc *} $new_line]} {
              set new_line "        wr_data_ni2mc_burst     \[burst\]\[pl_byte*8 +: 8\] = wr_data_ni2mc     \[burst*8 + pl_byte*8*2*nCK_PER_CLK +: 8\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mask_ni2mc_burst*= wr_data_mask_ni2mc*} $new_line]} {
              set new_line "        wr_data_mask_ni2mc_burst\[burst\]\[pl_byte\]        = wr_data_mask_ni2mc\[burst   + pl_byte*2*nCK_PER_CLK\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mc2phy * = wr_data_mc2phy_byte *} $new_line]} {
              set new_line "     wr_data_mc2phy_tmp     \[dq_byte*8*2*nCK_PER_CLK +: 8*2*nCK_PER_CLK\] = wr_data_mc2phy_byte     \[dq_byte\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mask_mc2phy* = wr_data_mask_mc2phy_byte*} $new_line]} {
              set new_line "     wr_data_mask_mc2phy_tmp\[dq_byte * 2*nCK_PER_CLK +:   2*nCK_PER_CLK\] = wr_data_mask_mc2phy_byte\[dq_byte\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mc2phy_byte*= wr_data_mc2phy_cw_order*ECC_WIDTH*} $new_line]} {
              set new_line "        wr_data_mc2phy_byte     \[dq_byte\]\[burst*8 +: 8\] = wr_data_mc2phy_cw_order     \[(dq_byte-DATA_WIDTH/8)*8  + burst*ECC_WIDTH_LOCAL*ECCS_PER_DQ + 8*DATA_WIDTH +: 8\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mask_mc2phy_byte*= wr_data_mask_mc2phy_cw_order*DM_WIDTH-PAYLOAD_DM_WIDTH*} $new_line]} {
              set new_line "        wr_data_mask_mc2phy_byte\[dq_byte\]\[burst\]        = wr_data_mask_mc2phy_cw_order\[(dq_byte-PAYLOAD_WIDTH/8) + burst*(DM_WIDTH-PAYLOAD_DM_WIDTH) + 8*PAYLOAD_DM_WIDTH\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mc2phy*= wr_data_mc2phy_byte*} $new_line]} {
              set new_line "     wr_data_mc2phy_tmp     \[dq_byte*8*2*nCK_PER_CLK +: 8*2*nCK_PER_CLK\] = wr_data_mc2phy_byte     \[dq_byte\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wr_data_mask_mc2phy*= wr_data_mask_mc2phy_byte*} $new_line]} {
              set new_line "     wr_data_mask_mc2phy_tmp\[dq_byte * 2*nCK_PER_CLK +:   2*nCK_PER_CLK\] = wr_data_mask_mc2phy_byte\[dq_byte\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*reg* rd_data_addr_dly*} $new_line]} {
              set new_line "always @(*) begin
   if (ECCS_PER_DQ == 2) begin
     wr_data_mc2phy       = {wr_data_mc2phy_tmp\[1151:1088\],
                             wr_data_mc2phy_tmp\[1023: 512\],
                             wr_data_mc2phy_tmp\[1087:1024\],
                             wr_data_mc2phy_tmp\[ 511:   0\]};
     wr_data_mask_mc2phy =  {wr_data_mask_mc2phy_tmp\[ 143: 136\],
                             wr_data_mask_mc2phy_tmp\[ 127:  64\],
                             wr_data_mask_mc2phy_tmp\[ 135: 128\],
                             wr_data_mask_mc2phy_tmp\[  63:   0\]};
   end
   else begin
     wr_data_mc2phy      = wr_data_mc2phy_tmp;
     wr_data_mask_mc2phy = wr_data_mask_mc2phy_tmp;
   end
end // always


reg  \[DATA_BUF_ADDR_WIDTH-1:0\] rd_data_addr_dly;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*assign* ecc_single*\'0*} $new_line]} {
              set new_line "assign  ecc_double                   = '0;"
              incr n 1
              set line_added 1
            }
            if {[string match {*,.ECC_WIDTH*} $new_line]} {
              set new_line "   ,.ECC_WIDTH           (ECC_WIDTH_LOCAL)
   ,.ECCS_PER_DQ         (ECCS_PER_DQ)"
              incr n 1
              set line_patched 1
            }
            if {[string match {*)u_ddr_mc_ecc_dec_fix(*} $new_line]} {
              set new_line "   .enable_autocorrect_sync         (enable_autocorrect_sync)
   ,.enable_ERR_syndrome_recording   (enable_ERR_syndrome_recording)
   ,.rd_data_en_mc2ni     (rd_data_en_mc2ni)
   ,.ecc_double           (ecc_double)
   ,.ecc_err2fifo_data    (ecc_err2fifo_data)
   ,.ecc_err2fifo_push    (ecc_err2fifo_push)
   ,"
              incr n 1
              set line_added 1
            }
            if {[string match {*,.DATA_BUF_OFFSET_WIDTH*} $new_line]} {
              set new_line "   ,.DATA_BUF_ADDR_WIDTH_USED (5)
   ,.DATA_BUF_OFFSET_WIDTH    (0)"
              incr n 1
              set line_patched 1
            }

            if { $line_patched } {
              puts_line_patched $line $out_file
            }
            if { $line_added } {    
              puts_line_added $line $out_file
            }

            puts $out_file $new_line
        }

        close $in_file
        close $out_file

        if { $n != 43 } {
          puts "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
        }

        # Rename the original HDL file to have .bak extension.
        file rename -force $filename $backup_filename
        # Move the temporary file to have the filename of the original HDL file.
        file rename -force $tmp_filename $filename
        
        puts "Successfully patched \"${filename}\"."
        }
        #======================================
        # END *mc_ecc.sv
        #=====================================


        #======================================
        # BEGIN *mc_ecc_buf.sv
        #=====================================
        if {$Ecc==1} {
        set target_file "${ddr}_v${file_ver}_mc_ecc_buf.sv"
        set n 0 
        puts "--> ${target_file}"
        set file [get_files  ${target_file}]
        set filename [get_property {NAME} $file]
        set tmp_filename "${filename}.tmp"
        set backup_filename "${filename}.bak"
        set in_file  [open $filename r]
        set out_file [open $tmp_filename w]

        while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0
            
            if {[string match {*parameter DATA_BUF_ADDR_WIDTH*} $new_line]} {
              set new_line "    parameter DATA_BUF_ADDR_WIDTH_USED = 4,"
              incr n 1
              set line_added 1
            }
            if {[string match {*wire * buf_wr_addr;*} $new_line]} {
              set new_line "  wire \[DATA_BUF_ADDR_WIDTH_USED-1:0\] buf_wr_addr;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*reg * buf_rd_addr_r;*} $new_line]} {
              set new_line "  reg \[DATA_BUF_ADDR_WIDTH_USED-1:0\] buf_rd_addr_r;
/*"
              incr n 1
              set line_patched 1
            }
            if {[string match {*input * rd_data;*} $new_line]} {
              set new_line "*/
  generate
    if (DATA_BUF_OFFSET_WIDTH > 0) begin
      if (DATA_BUF_ADDR_WIDTH >= DATA_BUF_ADDR_WIDTH_USED) begin : ge_used_addr_bits
        always @(posedge clk)  
          buf_rd_addr_r <= #TCQ{wr_data_addr\[DATA_BUF_ADDR_WIDTH_USED-1:0\], wr_data_offset};
        assign buf_wr_addr = {rd_data_addr\[DATA_BUF_ADDR_WIDTH_USED-1:0\], rd_data_offset};
      end
      else begin : lt_used_addr_bits
        always @(posedge clk) 
          buf_rd_addr_r <= #TCQ{{DATA_BUF_ADDR_WIDTH_USED-DATA_BUF_ADDR_WIDTH{1'b0}},
                                 wr_data_addr\[DATA_BUF_ADDR_WIDTH-1:0\],
                                 wr_data_offset};
        assign buf_wr_addr = {{DATA_BUF_ADDR_WIDTH_USED-DATA_BUF_ADDR_WIDTH{1'b0}},
                              rd_data_addr\[DATA_BUF_ADDR_WIDTH-1:0\], 
                              rd_data_offset};
      end
    end
    else begin
      if (DATA_BUF_ADDR_WIDTH >= DATA_BUF_ADDR_WIDTH_USED) begin : ge_used_addr_bits
        always @(posedge clk)  
          buf_rd_addr_r <= #TCQ wr_data_addr\[DATA_BUF_ADDR_WIDTH_USED-1:0\];
        assign buf_wr_addr = rd_data_addr\[DATA_BUF_ADDR_WIDTH_USED-1:0\];
      end
      else begin : lt_used_addr_bits
        always @(posedge clk) 
          buf_rd_addr_r <= #TCQ{{DATA_BUF_ADDR_WIDTH_USED-DATA_BUF_ADDR_WIDTH{1'b0}},
                                 wr_data_addr\[DATA_BUF_ADDR_WIDTH-1:0\]};
        assign buf_wr_addr = {{DATA_BUF_ADDR_WIDTH_USED-DATA_BUF_ADDR_WIDTH{1'b0}},
                              rd_data_addr\[DATA_BUF_ADDR_WIDTH-1:0\]};
      end
    end
  endgenerate

$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*localparam FULL_RAM_CNT*} $new_line]} {
              set new_line "  localparam BITS_PER_RAM =   ((DATA_BUF_ADDR_WIDTH_USED+DATA_BUF_OFFSET_WIDTH) < 6)? 6
                            : ((DATA_BUF_ADDR_WIDTH_USED+DATA_BUF_OFFSET_WIDTH) < 7)? 3
                            :                                                         1;
  localparam FULL_RAM_CNT = (BUF_WIDTH/BITS_PER_RAM);"
              incr n 1
              set line_patched 1
            }
            if {[string match {*localparam* REMAINDER*} $new_line]} {
              set new_line "  localparam REMAINDER = BUF_WIDTH % BITS_PER_RAM;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*localparam* RAM_WIDTH*} $new_line]} {
              set new_line "  localparam RAM_WIDTH = (RAM_CNT*BITS_PER_RAM);"
              incr n 1
              set line_patched 1
            }
            if {[string match {*assign buf_in_data *REMAINDER*} $new_line]} {
              set new_line "        assign buf_in_data = {{BITS_PER_RAM-REMAINDER{1'b0}}, payload};"
              incr n 1
              set line_patched 1
            }

            if { $line_patched } {
              puts_line_patched $line $out_file
            }
            if { $line_added } {    
              puts_line_added $line $out_file
            }
            puts $out_file $new_line
        }

        close $in_file
        close $out_file

        if { $n != 8 } {
          error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
        }

        # Rename the original HDL file to have .bak extension.
        file rename -force $filename $backup_filename
        # Move the temporary file to have the filename of the original HDL file.
        file rename -force $tmp_filename $filename
        
        puts "Successfully patched \"${filename}\"."
        }
        #======================================
        # END *mc_ecc_buf.sv
        #=====================================

        #======================================
        # BEGIN *mc_ecc_dec_fix.sv
        #=====================================
        if {$Ecc==1} {
        set target_file "${ddr}_v${file_ver}_mc_ecc_dec_fix.sv"
        set n 0 
        puts "--> ${target_file}"
        set file [get_files  ${target_file}]
        set filename [get_property {NAME} $file]
        set tmp_filename "${filename}.tmp"
        set backup_filename "${filename}.bak"
        set in_file  [open $filename r]
        set out_file [open $tmp_filename w]

        while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0
            
            if {[string match {*always*AS*correct_en*} $new_line]} {
              set new_line " always @(correct_en or ecc_rddata_r or flip_bits)"
              incr n 1
              set line_patched 1
            }
            if {[string match {*parameter ECC_WIDTH*} $new_line]} {
              set new_line "    parameter ECCS_PER_DQ        = 1,"
              incr n 1
              set line_added 1
            }
            if {[string match {*rd_data, ecc_single, ecc_multiple, ecc_err_addr*} $new_line]} {
              set new_line "  rd_data, ecc_single, ecc_double, ecc_multiple, ecc_err_addr,
  ecc_err2fifo_data, ecc_err2fifo_push, // not used"
              incr n 1
              set line_patched 1
            }
            if {[string match {*clk, rst, h_rows*} $new_line]} {
              set new_line "  clk, rst, 
  enable_autocorrect_sync, enable_ERR_syndrome_recording,
  rd_data_en_mc2ni,
  h_rows, phy_rddata, correct_en, ecc_status_valid, non_per_rd_cas, ecc_status_valid_nxt,"
              incr n 1
              set line_patched 1
            }
            if {[string match {*localparam ADDR_FIFO_FULL_THRESHOLD*} $new_line]} {
              set new_line "  localparam SYNDROME_DECODE_ROM_DUALPORT = 1;
  localparam PAYLOAD_IN_ECC_WIDTH = PAYLOAD_WIDTH / ECCS_PER_DQ;
  localparam CODE_IN_ECC_WIDTH = CODE_WIDTH / ECCS_PER_DQ;
  localparam DATA_IN_ECC_WIDTH = DATA_WIDTH / ECCS_PER_DQ;
  localparam DQ_IN_ECC_WIDTH = DQ_WIDTH / ECCS_PER_DQ;
$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*input rst;*} $new_line]} {
              set new_line "  input enable_autocorrect_sync;
  input enable_ERR_syndrome_recording;
  input rd_data_en_mc2ni;"
              incr n 1
              set line_added 1
            }
            if {[string match {*input * h_rows;*} $new_line]} {
              set new_line "  input \[(CODE_IN_ECC_WIDTH)*ECC_WIDTH-1:0\] h_rows;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*input * phy_rddata;*} $new_line]} {
              set new_line "  wire \[2*nCK_PER_CLK*ECC_WIDTH-1:0\] ecc_reencode_bits;"
              incr n 1
              set line_added 1
            }
            if {[string match {*Swizzle data and check bits from the XiPhy interface*} $new_line]} {
              set new_line "/*
$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*phy_rddata* & h_rows*} $new_line]} {
              set new_line "*/
    for (k=0; k<2*nCK_PER_CLK*ECCS_PER_DQ; k=k+1) begin : ecc_word
      if((ECC_WIDTH == 16) && (DATA_IN_ECC_WIDTH == 128)) begin
        Hamming_144_128_encoder encoder(
          .data_in  (phy_rddata\[k*DQ_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\]),
          .code_out (ecc_reencode_bits\[k*ECC_WIDTH +: ECC_WIDTH\])
        );

        assign syndrome_ns\[k*ECC_WIDTH +: ECC_WIDTH\] =
             phy_rddata\[k*DQ_IN_ECC_WIDTH+DATA_IN_ECC_WIDTH+:ECC_WIDTH\]
           ^ ecc_reencode_bits\[k*ECC_WIDTH +: ECC_WIDTH\];

      end else begin
        // Swizzle data and check bits from the XiPhy interface's
        // systematic format to the ECC block's per burst systematic format.
        for (m=0; m<ECC_WIDTH; m=m+1) begin : ecc_bit
          assign syndrome_ns\[k*ECC_WIDTH+m\] =
           ^(   phy_rddata\[k*DQ_IN_ECC_WIDTH+:CODE_IN_ECC_WIDTH\]
              & h_rows\[m*CODE_IN_ECC_WIDTH+:CODE_IN_ECC_WIDTH\]);
        end"
              incr n 1
              set line_added 1
            }
            if {[string match {*reg * syndrome_r;*} $new_line]} {
              set new_line "  reg [2*nCK_PER_CLK*ECC_WIDTH*ECCS_PER_DQ-1:0] syndrome_r;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*for * begin : extract_payload*} $new_line]} {
              set new_line "    for (i=0; i<2*nCK_PER_CLK*ECCS_PER_DQ; i=i+1) begin : extract_payload"
              incr n 1
              set line_patched 1
            }
            if {[string match {*assign ecc_rddata_ns*} $new_line]} {
              set new_line "      assign ecc_rddata_ns\[i*PAYLOAD_IN_ECC_WIDTH+:PAYLOAD_IN_ECC_WIDTH\] ="
              incr n 1
              set line_patched 1
            }
            if {[string match {*phy_rddata\[i*DQ_WIDTH+:PAYLOAD_WIDTH\]*} $new_line]} {
              set new_line "               phy_rddata\[i*DQ_IN_ECC_WIDTH+:PAYLOAD_IN_ECC_WIDTH\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*wire * h_matrix*} $new_line]} {
              set new_line "  wire \[ECC_WIDTH-1:0\] h_matrix \[DATA_IN_ECC_WIDTH-1:0\];"
              incr n 1
              set line_patched 1
            }
            if {[string match {*// Correct data.*} $new_line]} {
              set new_line "/*
$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*// Copy raw payload bits if ECC_TEST is ON.*} $new_line]} {
              set new_line "*/
  // Correct data.
  output reg \[2*nCK_PER_CLK*PAYLOAD_WIDTH-1:0\] rd_data;
  input correct_en;
  integer s;
  always @(/*AS*/correct_en or ecc_rddata_r or flip_bits)
    for (s=0; s<2*nCK_PER_CLK*ECCS_PER_DQ; s=s+1)
      if (correct_en)
        rd_data\[s*PAYLOAD_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\] = 
          ecc_rddata_r\[s*PAYLOAD_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\] ^ 
              flip_bits\[s*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\];
      else rd_data\[s*PAYLOAD_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\] = 
           ecc_rddata_r\[s*PAYLOAD_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\];

$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*output * ecc_single*} $new_line]} {
              set new_line "  output wire \[2*nCK_PER_CLK-1:0\] ecc_double;
  output wire ecc_err2fifo_data, ecc_err2fifo_push; // not used"
              incr n 1
              set line_added 1
            }
            if {[string match {*assign ecc_single*} $new_line]} {
              set new_line "      assign ecc_double\[v\] = 1'b0;"
              incr n 1
              set line_added 1
            }
            if {[string match {*reg * addr_fifo_wptr*} $new_line]} {
              set new_line "reg  \[ADDR_FIFO_WIDTH-1:0\] ecc_err_addr_r;"
              incr n 1
              set line_added 1
            }
            if {[string match {*COLBITS == 11 *} $new_line]} {
              set new_line "                                        ( COLBITS == 10 ) ? { 4'b0,                                     { { ( 16 - COLBITS ) { 1'b0 } },      cmdCol\[winPortEncC*COLBITS+:10\] } } :"
              incr n 1
              set line_added 1
            }
            if {[string match {*ecc_err_addr * addr_fifo*} $new_line]} {
              set new_line "  ecc_err_addr_r <= #TCQ ecc_err_addr;"
              incr n 1
              set line_added 1
            }

            if { $line_patched } {
              puts_line_patched $line $out_file
            }
            if { $line_added } {    
              puts_line_added $line $out_file
            }

            puts $out_file $new_line
        }

        close $in_file
        close $out_file

        if { $n != 22 } {
          error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
        }

        file rename -force $filename $backup_filename
        file rename -force $tmp_filename $filename
        
        puts "Successfully patched \"${filename}\"."
        }
        #======================================
        # END *mc_ecc_dec_fix.sv
        #=====================================

        #======================================
        # BEGIN *mc_ecc_fi_xor.sv
        #=====================================
        if {$Ecc==1} {
        set target_file "${ddr}_v${file_ver}_mc_ecc_fi_xor.sv"
        set n 0 
        puts "--> ${target_file}"
        set file [get_files  ${target_file}]
        set filename [get_property {NAME} $file]
        set tmp_filename "${filename}.tmp"
        set backup_filename "${filename}.bak"
        set in_file  [open $filename r]
        set out_file [open $tmp_filename w]

        while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0
            
            if {[string match {*parameter integer ECC_WIDTH*} $new_line]} {
              set new_line "  parameter integer ECCS_PER_DQ            = 1,"
              incr n 1
              set line_added 1
            }
            if {[string match {*localparam CBYTES_BURST1_LSB*} $new_line]} {
              set new_line "localparam CBYTES_BURST1_LSB          = CBYTES_BURST0_LSB + ECC_WIDTH*ECCS_PER_DQ;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*localparam CBYTES_BURST7TO1_WIDTH*} $new_line]} {
              set new_line "localparam CBYTES_BURST7TO1_WIDTH     = (2*nCK_PER_CLK-1)*ECC_WIDTH*ECCS_PER_DQ;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*assign wrdata_out*CBYTES_BURST0_LSB*} $new_line]} {
              set new_line "assign wrdata_out\[CBYTES_BURST0_LSB+:ECC_WIDTH*ECCS_PER_DQ\] = wrdata_in\[CBYTES_BURST0_LSB+:ECC_WIDTH*ECCS_PER_DQ\] ^ fi_xor_data\[DATA_WIDTH+:ECC_WIDTH*ECCS_PER_DQ\]; // Check byte of burst 0"
              incr n 1
              set line_patched 1
            }

            if { $line_patched } {
              puts_line_patched $line $out_file
            }
            if { $line_added } {    
              puts_line_added $line $out_file
            }

            puts $out_file $new_line
        }

        close $in_file
        close $out_file

        if { $n != 4 } {
          error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
        }

        # Rename the original HDL file to have .bak extension.
        file rename -force $filename $backup_filename
        # Move the temporary file to have the filename of the original HDL file.
        file rename -force $tmp_filename $filename
        
        puts "Successfully patched \"${filename}\"."
        }
        #======================================
        # END *mc_ecc_fi_xor.sv
        #=====================================

        #======================================
        # BEGIN *mc_ecc_merge_enc.sv
        #=====================================
        if {$Ecc==1} {
        set target_file "${ddr}_v${file_ver}_mc_ecc_merge_enc.sv"
        set n 0 
        puts "--> ${target_file}"
        set file [get_files  ${target_file}]
        set filename [get_property {NAME} $file]
        set tmp_filename "${filename}.tmp"
        set backup_filename "${filename}.bak"
        set in_file  [open $filename r]
        set out_file [open $tmp_filename w]

        while {[gets $in_file line] != -1} {
            set new_line $line
            set line_patched 0
            set line_added 0
            
            if {[string match {*parameter ECC_WIDTH*} $new_line]} {
              set new_line "    parameter ECCS_PER_DQ           = 1,"

              incr n 1
              set line_added 1
            }
            if {[string match {*input clk;*} $new_line]} {
              set new_line "  localparam PAYLOAD_IN_ECC_WIDTH = PAYLOAD_WIDTH / ECCS_PER_DQ;
  localparam CODE_IN_ECC_WIDTH = CODE_WIDTH / ECCS_PER_DQ;
  localparam DATA_IN_ECC_WIDTH = DATA_WIDTH / ECCS_PER_DQ;
  localparam DQ_IN_ECC_WIDTH = DQ_WIDTH / ECCS_PER_DQ;

$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*input * h_rows;*} $new_line]} {
              set new_line "  input \[(CODE_WIDTH/ECCS_PER_DQ)*ECC_WIDTH-1:0\] h_rows;"
              incr n 1
              set line_patched 1
            }
            if {[string match {*reg * raw_not_ecc_r;*} $new_line]} {
              set new_line "  wire  \[2*nCK_PER_CLK*ECC_WIDTH-1:0\] ecc_encode_bits;"
              incr n 1
              set line_added 1
            }
            if {[string match {*for * begin : ecc_word*} $new_line]} {
              set new_line "/*
$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*always *AS*h_rows or*} $new_line]} {
              set new_line "      always @(h_rows or merged_data or raw_not_ecc_r) begin"
              incr n 1
              set line_patched 1
            }
            if {[string match {*always * mc_wrdata <=*} $new_line]} {
              set new_line "*/
    for (j=0; j<2*nCK_PER_CLK*ECCS_PER_DQ; j=j+1) begin : ecc_word
      if((ECC_WIDTH == 16) && (DATA_IN_ECC_WIDTH == 128)) begin
        Hamming_144_128_encoder encoder(
          .data_in  (merged_data\[j*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\]),
          .code_out (ecc_encode_bits\[j*ECC_WIDTH +: ECC_WIDTH\])
        );
      
        always @(ecc_encode_bits or merged_data or raw_not_ecc_r) begin
          mc_wrdata_c\[j*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\] =
             merged_data\[j*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\];
          for (k=0; k<ECC_WIDTH; k=k+1) begin
            if(~raw_not_ecc_r\[j\]) begin
              mc_wrdata_c\[ (2*nCK_PER_CLK*DATA_WIDTH) + j*ECC_WIDTH+ECC_WIDTH-k-1\] =
                ecc_encode_bits\[j*ECC_WIDTH + ECC_WIDTH-k-1\];
            end else begin
              mc_wrdata_c\[ (2*nCK_PER_CLK*DATA_WIDTH) + j*ECC_WIDTH+ECC_WIDTH-k-1\] = 
                  merged_data\[ (2*nCK_PER_CLK*DATA_WIDTH) + j*ECC_WIDTH+ECC_WIDTH-k-1\];
            end // if
          end // for k
        end

      end else begin
      
        always @(/*AS*/h_rows or merged_data or raw_not_ecc_r) begin
          // Keep check bits in MSBs of mc_wrdata_c.  This is the
          // arrangement required by the XiPhy interface for
          // a systematic ecc format.
          mc_wrdata_c\[j*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\] =
             merged_data\[j*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\];
          for (k=0; k<ECC_WIDTH; k=k+1) begin
            if (~raw_not_ecc_r\[j\] | ~(PAYLOAD_IN_ECC_WIDTH > DATA_IN_ECC_WIDTH) ) begin
              mc_wrdata_c\[ (2*nCK_PER_CLK*DATA_WIDTH) + j*ECC_WIDTH+ECC_WIDTH-k-1\] =
                ^(merged_data\[j*DATA_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\] & 
                  h_rows\[k*CODE_IN_ECC_WIDTH+:DATA_IN_ECC_WIDTH\]);
            end else begin
              mc_wrdata_c\[ (2*nCK_PER_CLK*DATA_WIDTH) + j*ECC_WIDTH+ECC_WIDTH-k-1\] = 
                  merged_data\[ (2*nCK_PER_CLK*DATA_WIDTH) + j*ECC_WIDTH+ECC_WIDTH-k-1\];
            end // if
          end // for k
        end // always

      end // if ((ECC_WIDTH == 16) && (DATA_IN_ECC_WIDTH == 128))
    end // for j
  endgenerate

$new_line"
              incr n 1
              set line_patched 1
            }
            if {[string match {*reg * raw_not_ecc_r;*} $new_line]} {
              set new_line "  wire  \[2*nCK_PER_CLK*ECC_WIDTH-1:0\] ecc_encode_bits;"
              incr n 1
              set line_added 1
            }

            if { $line_patched } {
              puts_line_patched $line $out_file
            }
            if { $line_added } {    
              puts_line_added $line $out_file
            }

            puts $out_file $new_line
        }

        close $in_file
        close $out_file

        if { $n != 7 } {
          error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
        }

        # Rename the original HDL file to have .bak extension.
        file rename -force $filename $backup_filename
        # Move the temporary file to have the filename of the original HDL file.
        file rename -force $tmp_filename $filename
        
        puts "Successfully patched \"${filename}\"."
        }
        #======================================
        # END *mc_ecc_merge_enc.sv
        #=====================================

        #======================================
        # BEGIN Template for changing a file
        #=====================================
#        set target_file "filename"
#        set n 0 
#        puts "--> ${target_file}"
#        set file [get_files  ${target_file}]
#        set filename [get_property {NAME} $file]
#        set tmp_filename "${filename}.tmp"
#        set backup_filename "${filename}.bak"
#        set in_file  [open $filename r]
#        set out_file [open $tmp_filename w]
#
#        while {[gets $in_file line] != -1} {
#            set new_line $line
#            set line_patched 0
#            set line_added 0
#            
#            # list of regexs:
#            if {[string match {*regex*} $new_line]} {
#              set new_line "blah"
#              incr n 1
#              # pick one of these:
#              #set line_patched 1
#              #set line_added 1
#            }
#
#            # Look at flag vars, patch if one is set
#            if { $line_patched } {
#              puts_line_patched $line $out_file
#            }
#            if { $line_added } {    
#              puts_line_added $line $out_file
#            }
#
#            # Output original or patched lines
#            puts $out_file $new_line
#        }
#
#        close $in_file
#        close $out_file
#
#        if { $n != 7 } {
#          error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
#        }
#
#        # Rename the original HDL file to have .bak extension.
#        file rename -force $filename $backup_filename
#        # Move the temporary file to have the filename of the original HDL file.
#        file rename -force $tmp_filename $filename
#        
#        puts "Successfully patched \"${filename}\"."
#        #======================================
#        # END Template for changing a file
#        #=====================================


        #======================================
        #regbank MR update -- only for lola
        #=====================================
        if {$lola==1} {
        set target_file "regbank.sv"
        set n 0 
          puts "--> ${target_file}"
         
         set file [get_files  ${target_file}]
          set filename [get_property {NAME} $file]
          
          set tmp_filename "${filename}.tmp"
          set backup_filename "${filename}.bak"

          set in_file  [open $filename r]
          set out_file [open $tmp_filename w]

          while {[gets $in_file line] != -1} {
	    set new_line $line
	    set line_patched 0
            set line_added 0
            

  	    
            
            if {[string match {*ddr_tester_mp.after_cal_mr0* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.after_cal_mr0 = 32'h0004;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }

            
            if {[string match {*ddr_tester_mp.during_cal_mr1* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.during_cal_mr1 = 32'h0005;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }

           
  	    if {[string match {*ddr_tester_mp.after_cal_mr1* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.after_cal_mr1 = 32'h0005;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }


            if {[string match {*ddr_tester_mp.during_cal_mr3* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.during_cal_mr3 = 32'h0080;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }

           
  	    if {[string match {*ddr_tester_mp.after_cal_mr3* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.after_cal_mr3 = 32'h0080;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }



  	    
            if {[string match {*ddr_tester_mp.during_cal_mr5* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.during_cal_mr5 = 32'h04E0;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }

            if {[string match {*ddr_tester_mp.after_cal_mr5* = *} $new_line] && $lola==1} {
              set new_line "ddr_tester_mp.after_cal_mr5 = 32'h04E0;"
               # puts $new_line 
              incr n 1
	      set line_patched 1
            }

          if { $line_patched } {
              puts_line_patched $line $out_file
	    }

          puts $out_file $new_line
	  }

	  close $in_file
	  close $out_file

          if { $n != 7 && $lola==1} {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
          if { $lola==0} {
	    error "*** This code should only execute for lola"
	  }
          # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."
          }


 	  #====================================================================================================================================
	  # Patch for bug #708
	  # rtl/controller/${ddr}_v2_2_mc_ctl.sv
	  #====================================================================================================================================
	  set target_file "${ddr}_v${file_ver}_mc_ctl.sv"
	  puts "--> ${target_file}"
	  set file [get_files -of_objects [get_ips ${mig_name}] ${target_file}]
	  set filename [get_property {NAME} $file]
	  
	  set tmp_filename "${filename}.tmp"
	  set backup_filename "${filename}.bak"

	  set in_file  [open $filename r]
	  set out_file [open $tmp_filename w]

	  #============================================
	  #	Old settings generated by the MIG to search
	  #============================================
  		set b2b_cas_not_allowed_old		{rank_different \| group_unchanged}

	  #===========================================
	  #	Patched settings for Everspin DDR ST-MRAM 
	  #===========================================

  		set b2b_cas_not_allowed_new		{1}

	  # line-by-line, read the original file
	  set n 0
	  while {[gets $in_file line] != -1} {
	    set new_line $line
	    set line_patched 0
	    
	    # Patch tCCD_S for Bug #708
#	    if {[string match -nocase {*wire*} $new_line] && [string match -nocase {*b2b_cas_not_allowed*} $new_line] && [string match -nocase "*${b2b_cas_not_allowed_old}*" $new_line]} {
#	      incr n [regsub -nocase -- $b2b_cas_not_allowed_old $new_line $b2b_cas_not_allowed_new new_line]
#	      set line_patched 1
#	    }

#wire group_unchanged         = ( (MEM == "DDR4") & (BGBITS == 1) ) ? (winGroupC == prevGroup) : 1'b0;

 # Performance 
             if {[string match -nocase {*wire*group_unchanged*=*(*(MEM*==*"DDR4")*&*(BGBITS*==*1)*)*?*(winGroupC*==*prevGroup)*:*1'b0;*} $new_line]} {
               set new_line "wire group_unchanged         = 1'b0;"
               incr n 1
	       set line_patched 1
	    }



	    if { $line_patched } {    
              puts_line_patched $line $out_file
            }
            puts $out_file $new_line
	  }
	  close $in_file
	  close $out_file

	  if { $n != 1 } {
	    error "*** Failed to patch $filename as expected. Patched $n of expected 1 line(s)."
	  }
	  
	  # Rename the original HDL file to have .bak extension.
	  file rename -force $filename $backup_filename
	  # Move the temporary file to have the filename of the original HDL file.
	  file rename -force $tmp_filename $filename
	  
	  puts "Successfully patched \"${filename}\"."

	  puts "DDR MIG patch for Everspin 1Gb ST-MRAM successful!"

}

patch_mig ${mig_ip_name} ${mig_no_mem_en} ${mig_timing_set} ${lola} ${device} ${MR_update}



#----------------------------------

proc patch_mc_ctl {in_file out_file comment} {
    set n 0
    set line_num 0
    while {[gets $in_file line] != -1} {
        set new_line $line
        set line_patched 0
        if {[string match -nocase {*wire b2b_cas_not_allowed*} $line] && [string match -nocase {*rank_different*group_unchanged*} $line]} {
            puts $out_file "//$line"
            puts $out_file "// aolsen tCCD_S hack, Just never allow b2b_cas  ${comment}"
            set new_line "wire b2b_cas_not_allowed     = 1;   ${comment}"
            incr n
            set line_patched 1
        }
        puts $out_file $new_line
        incr line_num
    }
    return $n
}

proc patch_ddr4_0 {in_file out_file comment} {
    set n 0
    set line_num 0
    while {[gets $in_file line] != -1} {
        set new_line $line
        set line_patched 0
        if {[string match -nocase {*output*} $line] && [string match -nocase {*c0_ddr4_app_wdf_rdy,*} $line]} {
            puts $out_file $line
            set new_line "   ddr_modded_interface.ddr_mp     ddr_mp,   ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*.c0_ddr4_app_wdf_rdy*} $line] && [string match -nocase {*(c0_ddr4_app_wdf_rdy),*} $line]} {
            puts $out_file $line
            set new_line "   .ddr_mp                 ( ddr_mp ),   ${comment}"
            incr n
            set line_patched 1
        }
        puts $out_file $new_line
        incr line_num
    }
    return $n
}
    
proc patch_ddr4 {in_file out_file comment} {
    set n 0
    set line_num 0
    while {[gets $in_file line] != -1} {
        set new_line $line
        set line_patched 0
        
        #done in other patch
        #if {[string match -nocase {*parameter*} $line] && [string match -nocase {*MR6*} $line] && [string match -nocase {*= 13'b*} $line]} {
        #    puts $out_file "//$line"
        #    set new_line "parameter  MR6  = 13'b0_0100_0001_1000, // vrefdq = 0x18, tccd_l = 1 (0x5) ${comment}"
        #    incr n
        #    set line_patched 1
        #}
        if {[string match -nocase {*output*} $line] && [string match -nocase {*c0_ddr4_app_wdf_rdy,*} $line]} {
            puts $out_file $line
            set new_line "   ddr_modded_interface.ddr_mp     ddr_mp,   ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*.rd_data_phy2mc*} $line] && [string match -nocase {*(c0_ddr4_rd_data_phy2mc),*} $line]} {
            puts $out_file $line
            set new_line "   .ddr_mp                 ( ddr_mp ),   ${comment}"
            incr n
            set line_patched 1
        }
        puts $out_file $new_line
        incr line_num
    }
    return $n
}

proc patch_cal_top {in_file out_file comment} {
    set n 0
    set line_num 0
    while {[gets $in_file line] != -1} {
        set new_line $line
        set line_patched 0
        if {[string match -nocase {*,output*} $line] && [string match -nocase {*xsdb_rdy*} $line]} {
            puts $out_file $line
            set new_line "   ,ddr_modded_interface.ddr_mp     ddr_mp   ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.usr_xsdb_rdy*} $line] && [string match -nocase {*(xsdb_rdy)*} $line]} {
            puts $out_file $line
            set new_line "   ,.ddr_mp                 (ddr_mp)   ${comment}"
            incr n
            set line_patched 1
        }
        puts $out_file $new_line
        incr line_num
    }
    return $n
}

proc patch_mem_intfc {in_file out_file comment} {
    set n 0
    set line_num 0
    while {[gets $in_file line] != -1} {
        set new_line $line
        set line_patched 0
      
        if {[string match -nocase {*,.mcal_ODT*} $line] && [string match -nocase {*(mcal_ODT)*} $line] && $line_num < 750} {
            set new_line "     ,.mcal_ODT                        (user_odt_sel ? user_odt  : mcal_ODT) ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.mcal_ADR*} $line] && [string match -nocase {*(mcal_ADR)*} $line] && $line_num < 750} {
            set new_line "     ,.mcal_ADR                        (user_mrs_sel ? user_addr  : mcal_ADR) ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.mcal_BA*} $line] && [string match -nocase {*(mcal_BA)*} $line] && $line_num < 750} {
            set new_line "     ,.mcal_BA                         (user_mrs_sel ? user_ba    : mcal_BA) ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.mcal_BG*} $line] && [string match -nocase {*(mcal_BG)*} $line] && $line_num < 750} {
            set new_line "     ,.mcal_BG                         (user_mrs_sel ? user_bg    : mcal_BG) ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.mcal_CS_n*} $line] && [string match -nocase {*(mcal_CS_n)*} $line] && $line_num < 750} {
            set new_line "     ,.mcal_CS_n                       (user_mrs_sel ? user_cs_n  : mcal_CS_n) ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.mcal_ACT_n*} $line] && [string match -nocase {*(mcal_ACT_n)*} $line] && $line_num < 750} {
            set new_line "     ,.mcal_ACT_n                      (user_mrs_sel ? user_act_n : mcal_ACT_n) ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,output*} $line] && [string match -nocase {*ddr4_mcs_lmb_ce*} $line]} {
            puts $out_file $line
            set new_line "   ,ddr_modded_interface.ddr_mp     ddr_mp   ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*assign row_mc_phy = {{(18-ROW_WIDTH){1'b0}}, row};*} $line]} {
            puts $out_file $line
            set new_line "\n`include \"bus_snoop.sv\"  ${comment}\n"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.xsdb_rdy*} $line] && [string match -nocase {*(xsdb_rdy)*} $line]} {
            puts $out_file $line
            set new_line "  ,.ddr_mp                      (ddr_mp)  ${comment}"
            incr n
            set line_patched 1
        }
        if {[string match -nocase {*,.srx_req*} $line] && [string match -nocase {*(srx_req)*} $line]} {
            puts $out_file $line
            set new_line "   ,.ddr_mp                  (ddr_mp)  ${comment}"
            incr n
            set line_patched 1
        }
        puts $out_file $new_line
        incr line_num
    }
    return $n
}

proc patch_mc {in_file out_file comment} {
    set n 0
    set line_num 0
    while {[gets $in_file line] != -1} {
        set new_line $line
        set line_patched 0
      
        if {[string match -nocase {*,output*} $line] && [string match -nocase {*dbg_out*} $line]} {
            puts $out_file $line
            set new_line "   ,ddr_modded_interface.ddr_mp     ddr_mp   ${comment}"
            incr n
            set line_patched 1
        }
      # peridic_config is already defined in the other script
        if {[string match -nocase {*wire*[31:0]*periodic_config = ( PER_RD_INTVL*} $line]} {
            puts $out_file "wire [31:0] PER_RD_INTVL_CPM = ddr_mp.periodic_read_count;  ${comment}"
            set new_line "wire [31:0] periodic_config = ( PER_RD_INTVL_CPM == 0 ) ? 32'b0 : { 30'b0, calDone, calDone };   ${comment}"
            incr n
            set line_patched 1
        }

     

        if {[string match -nocase {*wire*[31:0]*periodic_interval_config = PER_RD_INTVL*} $line]} {
            set new_line "wire [31:0] periodic_interval_config = PER_RD_INTVL_CPM;   ${comment}"
            incr n
            set line_patched 1
        }
        
        puts $out_file $new_line
        incr line_num
    }
    return $n
}

proc patch_mig {mig_name} {

    puts "=======================================================================";
    puts "||  Everspin ST-MRAM DDR4 Patch Script for Xilinx MIG v2.1 (2016.4)  ||";
    puts "=======================================================================";
    puts "  Target Device:  EMD4E001G08AG1-125-ES  WS1                           ";

    set this_script [info script]
    set mig_ver "2.1"
    set mig_ip [get_ips $mig_name]
    set mig_ip_filename [get_property {IP_FILE} $mig_ip]
    set mig_ip_file [get_files $mig_ip_filename]

    set comment "// patched by: ${this_script}"
    
    set patch_list {}
    lappend patch_list patch_mem_intfc "${mig_name}_ddr4_mem_intfc.sv" 10
    lappend patch_list patch_cal_top   "ddr4_v2_2_cal_top.sv" 2
    lappend patch_list patch_ddr4      "${mig_name}_ddr4.sv" 2
    lappend patch_list patch_ddr4_0    "${mig_name}.sv" 2
    # This is done by the standard customer patch_ddr4mig_mram.tcl now.
    #lappend patch_list patch_mc_ctl    "ddr4_v2_2_mc_ctl.sv" 1
    lappend patch_list patch_mc        "ddr4_v2_2_mc.sv" 3
    
    #puts "patch_list=$patch_list"
    foreach {proc_name fn num_expected} $patch_list {
        #puts "proc=$proc_name fn=$fn num=$num_expected"
        set ipFile [get_files $fn]
        set file_name [get_property {NAME} $ipFile]
        set tmp_file_name    "${file_name}.tmp"
        set backup_file_name "${file_name}.bak"
        
        set in_file  [open $file_name r]
        set out_file [open $tmp_file_name w]
        puts "Processing File: $file_name  . . .\n"
        set num_patched [$proc_name $in_file $out_file $comment]
        #
        close $in_file
        close $out_file
        #
        if { $num_patched != $num_expected } {
            error "*** Failed to patch ${file_name} as expected. Patched ${num_patched} of expected ${num_expected} line(s)."
        } else {
            file rename -force $file_name $backup_file_name
            file rename -force $tmp_file_name $file_name
            puts "Successfully patched ${file_name}."
        }
    }

}

if {$lola==1} {
patch_mig ${mig_ip_name}
}


